/* This is a script that lists the users on the BBS.
 *
 * Author: Eric Oulashin (AKA Nightfox)
 * BBS: Digital Distortion
 * BBS address: digdist.bbsindex.com
 *
 * Date       User              Description
 * 2009-05-02 Eric Oulashin     Created
 * 2009-05-03 Eric Oulashin     Added a Synchronet version check.
 * 2009-05-04 Eric Oulashin     Changed the User variable's name from user
 *                              to theUser to avoid stomping on the user
 *                              object declared in sbbsdefs.js.
 * 2009-05-07 Eric Oulashin     Updated to allow the user to sort the
 *                              list alphabetically by user alias.
 * 2009-05-10 Eric Oulashin     Removed the "donePrompt" color from the colors
 *                              array, because it wasn't being used.
 * 2009-05-13 Eric Oulashin     Updated to be free of DigitalDistortionDefs.js.
 *                              Added VERSION and VER_DATE, although they aren't
 *                              displayed to the user.
 */

load("sbbsdefs.js");

// This script requires Synchronet version 3.10 or higher.
// Exit if the Synchronet version is below the minimum.
if (system.version_num < 31000)
{
	var message = "\1n\1h\1y\1i* Warning:\1n\1h\1w Digital Distortion List Today's Callers "
	             + "requires version \1g3.10\1w or\r\n"
	             + "higher of Synchronet.  This BBS is using version \1g" + system.version
	             + "\1w.  Please notify the sysop.";
	console.crlf();
	console.print(message);
	console.crlf();
	console.print("\1p");
	exit();
}

// Colors
var colors = new Array();
colors["veryTopHeader"] = "\1n\1h\1w";
colors["border"] = "\1n\1c";
colors["colLabelText"] = "\1n4\1h\1w";
colors["userNum"] = "\1h\1c";
colors["userName"] = "\1h\1y";
colors["location"] = "\1h\1m";
colors["lastCallDate"] = "\1h\1r";
colors["connectionType"] = "\1h\1b";
colors["numCalls"] = "\1n\1h\1g";


// Program option variables
// Whether or not to exclude the sysop in the user listing
var excludeSysop = false;

const VERSION = "1.01";
const VER_DATE = "2009-05-13";

// Column inner lengths
var USER_NUM_LEN = 5;
var USER_NAME_LEN = 18;
var LOCATION_LEN = 25;
var LAST_CALL_DATE_LEN = 8;
var NUM_CALLS_LEN = 7;
var CONNECTION_TYPE_LEN = 8;

// Single-line box drawing/border characters.
const UPPER_LEFT_SINGLE = "";
const HORIZONTAL_SINGLE = "";
const UPPER_RIGHT_SINGLE = "";
const VERTICAL_SINGLE = "";
const LOWER_LEFT_SINGLE = "";
const LOWER_RIGHT_SINGLE = "";
const T_SINGLE = "";
const LEFT_T_SINGLE = "";
const RIGHT_T_SINGLE = "";
const BOTTOM_T_SINGLE = "";
const CROSS_SINGLE = "";
// Double-line box drawing/border characters.
const UPPER_LEFT_DOUBLE = "";
const HORIZONTAL_DOUBLE = "";
const UPPER_RIGHT_DOUBLE = "";
const VERTICAL_DOUBLE = "";
const LOWER_LEFT_DOUBLE = "";
const LOWER_RIGHT_DOUBLE = "";
const T_DOUBLE = "";
const LEFT_T_DOUBLE = "";
const RIGHT_T_DOUBLE = "";
const BOTTOM_T_DOUBLE = "";
const CROSS_DOUBLE = "";

///////////////////////////
// Script execution code //
///////////////////////////


// Ask the user if they want the list sorted alphabetically
var sortUserList = false;
var sortType = "alpha"; // Alphabetical sort
console.crlf();
// Note: If console.noyes() returns false, that means the
// user chose Yes.
if (!console.noyes("Sort alphabetically"))
{
	sortUserList = true;
	sortType = "alpha";
}

// Clear the screen
console.clear();

// The following line which sets console.line_counter to 0 is a
// kludge to disable Synchronet's automatic pausing after a
// screenful of text.
console.line_counter = 0;


// Write the header
console.print(colors["veryTopHeader"]);
console.center("User List");
// Upper border lines
// User number section
var lineText = colors["border"] + UPPER_LEFT_SINGLE;
for (var i = 0; i < USER_NUM_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += T_SINGLE;
// User name section
for (var i = 0; i < USER_NAME_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += T_SINGLE;
// User location section
for (var i = 0; i < LOCATION_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += T_SINGLE;
// Last call date section
for (var i = 0; i < LAST_CALL_DATE_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += T_SINGLE;
// Connection type section
for (var i = 0; i < CONNECTION_TYPE_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += T_SINGLE;
// # calls section
for (var i = 0; i < NUM_CALLS_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += UPPER_RIGHT_SINGLE;
console.center(lineText);
// Column labels
lineText = colors["border"] + VERTICAL_SINGLE + colors["colLabelText"]
         + centerText("User#", USER_NUM_LEN) + colors["border"] + VERTICAL_SINGLE
         + colors["colLabelText"] + centerText("User Name", USER_NAME_LEN) + colors["border"]
         + VERTICAL_SINGLE + colors["colLabelText"]
         + centerText("Location", LOCATION_LEN) + colors["border"]
         + VERTICAL_SINGLE + colors["colLabelText"] + centerText("Last On", LAST_CALL_DATE_LEN)
         + colors["border"] + VERTICAL_SINGLE + colors["colLabelText"]
         + colors["colLabelText"] + centerText("Via", CONNECTION_TYPE_LEN)
         + colors["border"] + VERTICAL_SINGLE + colors["colLabelText"]
         + centerText("# calls", NUM_CALLS_LEN) + colors["border"] + VERTICAL_SINGLE;
console.center(lineText);
// Lower border lines for header
// User number section
var lineText = colors["border"] + LEFT_T_SINGLE;
for (var i = 0; i < USER_NUM_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += CROSS_SINGLE;
// User name section
for (var i = 0; i < USER_NAME_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += CROSS_SINGLE;
// User location section
for (var i = 0; i < LOCATION_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += CROSS_SINGLE;
// Last call date section
for (var i = 0; i < LAST_CALL_DATE_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += CROSS_SINGLE;
// Connection type section
for (var i = 0; i < CONNECTION_TYPE_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += CROSS_SINGLE;
// # calls section
for (var i = 0; i < NUM_CALLS_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += RIGHT_T_SINGLE;
console.center(lineText);

// List the users
var theUser = null;
var userArray = null;
for (var i = 1; i <= system.stats.total_users; ++i)
{
	// If the sysop is to be excluded, then skip user # 1 (user #1
	// is more than likely the sysop).
	if (excludeSysop && (i == 1))
		continue;

	// Load the user record
	theUser = new User(i);

	// If the user wants the user list sorted, then add the user to
	// the user array so it can be sorted and displayed later.
	// Otherwise, just display the user record.
	if (sortUserList)
	{
		if (userArray == null)
			userArray = new Array();
		userArray.push(theUser);
	}
	else
		displayUserInfo(theUser);
}
// If the user wants th euser list sorted, then sort it and
// display the sorted list.
if (sortUserList && (userArray != null))
{
	// If sortType is "alpha", then sort the list alphabetically by
	// user alias.
	if (sortType == "alpha")
	{
		// Sort the array by user alias
		userArray.sort(function(pA, pB)
		{
			// Return -1, 0, or 1, depending on whether user A's alias
			// comes before, is equal to, or comes after user B's alias.
			var returnValue = 0;
			if (pA.alias < pB.alias)
				returnValue = -1;
			else if (pA.alias > pB.alias)
				returnValue = 1;
				
			return returnValue;
		});
	}

	// Display the user records
	for (var i = 0; i < userArray.length; ++i)
		displayUserInfo(userArray[i]);
}

// Write the end border
// User number section
var lineText = colors["border"] + LOWER_LEFT_SINGLE;
for (var i = 0; i < USER_NUM_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += BOTTOM_T_SINGLE;
// User name section
for (var i = 0; i < USER_NAME_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += BOTTOM_T_SINGLE;
// User location section
for (var i = 0; i < LOCATION_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += BOTTOM_T_SINGLE;
// Last call date section
for (var i = 0; i < LAST_CALL_DATE_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += BOTTOM_T_SINGLE;
// Connection type section
for (var i = 0; i < CONNECTION_TYPE_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += BOTTOM_T_SINGLE;
// # calls section
for (var i = 0; i < NUM_CALLS_LEN; ++i)
	lineText += HORIZONTAL_SINGLE;
lineText += LOWER_RIGHT_SINGLE;
console.center(lineText);
console.print("\1h\1w" + system.stats.total_users + " \1btotal users.\r\n");
// Display the system's pause/"hit a key" prompt
console.print("\1p");

// End of script execution


///////////////////////////////////////////////////////////////////////////////////
// Functions

// Displays a User record in the list.
//
// Parameters:
//  pUser: A User object
function displayUserInfo(pUser)
{
	if (pUser != null)
	{
		var lineText = colors["border"] + VERTICAL_SINGLE
				 + colors["userNum"] + centerText(format("%" + USER_NUM_LEN + "d", pUser.number), USER_NUM_LEN)
				 + colors["border"] + VERTICAL_SINGLE
				 + colors["userName"] + centerText(pUser.alias.substring(0, USER_NAME_LEN), USER_NAME_LEN)
				 + colors["border"] + VERTICAL_SINGLE
				 + colors["location"] + centerText(pUser.location.substring(0, LOCATION_LEN), LOCATION_LEN)
				 + colors["border"] + VERTICAL_SINGLE
				 + colors["lastCallDate"] + centerText(strftime("%m/%d/%y", pUser.stats.laston_date), LAST_CALL_DATE_LEN)
				 + colors["border"] + VERTICAL_SINGLE
				 + colors["connectionType"] + centerText(pUser.connection.substring(0, CONNECTION_TYPE_LEN), CONNECTION_TYPE_LEN)
				 + colors["border"] + VERTICAL_SINGLE
				 + colors["numCalls"] + centerText(format("%" + NUM_CALLS_LEN + "d", pUser.stats.total_logons), NUM_CALLS_LEN)
				 + colors["border"] + VERTICAL_SINGLE;
		console.center(lineText);
	}
}

// Centers some text within a specified field length.
//
// Parameters:
//  pText: The text to center
//  pFieldLen: The field length
function centerText(pText, pFieldLen)
{
	var centeredText = "";
	
	// If pFieldLen is less than the text length, then truncate the string.
	if (pFieldLen < pText.length)
	{
		centeredText = pText.substring(0, pFieldLen);
	}
	else
	{
		// pFieldLen is at least equal to the text length, so we can
		// center the text.
		// Calculate the number of spaces needed to center the text.
		var numSpaces = pFieldLen - pText.length;
		if (numSpaces > 0)
		{
			var rightSpaces = (numSpaces/2).toFixed(0);
			var leftSpaces = numSpaces - rightSpaces;
			// Build centeredText
			for (var i = 0; i < leftSpaces; ++i)
				centeredText += " ";
			centeredText += pText;
			for (var i = 0; i < rightSpaces; ++i)
				centeredText += " ";
		}
		else
		{
			// pFieldLength is the same length as the text.
			centeredText = pText;
		}
	}

	return centeredText;
}