UNIT CharSets;

INTERFACE

FUNCTION  InitCharSets : BOOLEAN;
PROCEDURE LoadDefaultCharSets;
PROCEDURE JunkCharSets;
PROCEDURE MimeBodyToFtn;
FUNCTION  FtnBodyToMime : LONGINT;
FUNCTION  BuildCHRSKludge : STRING;
PROCEDURE AddStandardMimeHeaders;
FUNCTION  MimeHeaderToFtn (Line : STRING) : STRING;


IMPLEMENTATION

USES Ramon,
     Logs,
     Cfg,
     Globals,
     Msgs,
     SwapMem,
     Decode;

CONST MAX_CHARSETS_CACHED = 6; { 2/4 default, rest vrij }
      CHARSETS_FILENAME   = 'CHARSETS.TDB';
      MAXLEN_SETNAME      = 30;
      LAST_INTERNAL_CODE  = 255;      { can be increased in the future }

TYPE CharSetType      = (stFtn,stMime);
     InternalCodeType = BYTE;

CONST SetDescr : ARRAY[CharSetType] OF STRING[4] = ('FTN','MIME');

TYPE CharSetRecord = RECORD
                           SetType    : CharSetType;
                           SetName    : STRING[MAXLEN_SETNAME];
                           UsageCount : BYTE;

                           MapIn      : ARRAY[128..255] OF InternalCodeType;
                           MapOut     : ARRAY[128..LAST_INTERNAL_CODE] OF BYTE;
                     END;

     CharSetRecordPtr = ^CharSetRecord;

VAR CharSetCount : 0..MAX_CHARSETS_CACHED;
    CharSetPtrs  : ARRAY[1..MAX_CHARSETS_CACHED] OF CharSetRecordPtr;

    FtnInSetNr,
    FtnOutSetNr,
    MimeInSetNr,
    MimeOutSetNr,
    FixedSets    : BYTE;

    { alleen voor overgang InitCharSets, LoadDefaultCharSets }
    FtnIn,
    MimeIn,
    FtnOut,
    MimeOut       : STRING[MAXLEN_SETNAME];

    { voor MimeBodyToFtn processing }
    MimeIsText    : BOOLEAN; { geen plaatjes gaan omzetten! }
    MimeIsQP      : BOOLEAN; { if true, then decode =xx }
    MimeCharSetNr : BYTE;
    MimeIsMulti   : BOOLEAN; { if true, then is multi-part msg }

    { voor FtnBodyToMime processing }
    FtnCharSetNr  : BYTE;
    FtnLines      : LONGINT;
    FtnHaveHigh   : BOOLEAN;
    FtnDoQuotePrn : BOOLEAN;  { use quoted printable encoding? }
    FtnSinceCR    : BYTE;     { Ftn_UnparagraphBody }


{--------------------------------------------------------------------------}
{ LoadCharSet                                                              }
{                                                                          }
{ Deze routine laadt een karakter set in. Als het nodig is wordt deze uit  }
{ de language file gehaald. Als de cache vol zit, dan wordt er een         }
{ uit gegooid. Als de set niet bestaat, dan wordt een kopie van de default }
{ set gebruikt.                                                            }
{ Sinds deze routine alleen intern gebruikt wordt, wordt het set nummer    }
{ terug gegeven voor directe access.                                       }
{                                                                          }
FUNCTION LoadCharSet (SetName : STRING; SetType : CharSetType; OutSet : BOOLEAN) : BYTE; { set index }

VAR SetNr : BYTE; { assigned by CheckSetStart }

    PROCEDURE CheckSetStart (Regel : STRING);

    VAR Lp : 128..LAST_INTERNAL_CODE;

    BEGIN
         IF (UpCaseString (Regel) <> SetName) THEN
            Exit;

         { vraag geheugen aan voor de set; kijk of er een vrij is }
         IF (CharSetCount < MAX_CHARSETS_CACHED) THEN
         BEGIN
              Inc (CharSetCount);
              SetNr:=CharSetCount;
              GetMem (CharSetPtrs[SetNr],SizeOf (CharSetRecord));
         END ELSE
         BEGIN
              { een set eruit gooien }
              SetNr:=FixedSets+1;

              { zoek de oudste set met de laagste UsageCount }
              FOR Lp:=FixedSets+2 TO CharSetCount DO
                  IF (CharSetPtrs[Lp]^.UsageCount < CharSetPtrs[SetNr]^.UsageCount) THEN
                     SetNr:=Lp;

              { SetNr moet er nu uitgegooid worden }
              IF Config.LogDebug THEN
                 LogMessage ('Discarding character set '+Regel);
         END;

         { initialiseer de nieuwe set }
         WITH CharSetPtrs[SetNr]^ DO
         BEGIN
              { zet een een-op-een mapping op }
              FOR Lp:=128 TO 255 DO
              BEGIN
                   MapIn[Lp]:=Lp;
                   MapOut[Lp]:=Lp;
              END;

              { alle interne codes >= 255 moeten naar een byte mappen }
              FOR Lp:=255 TO LAST_INTERNAL_CODE DO
                  MapOut[Lp]:=255;

              SetName:=Regel;
              UsageCount:=0;
         END; { with }

         CharSetPtrs[SetNr]^.SetType:=SetType;

         IF Config.LogDebug THEN
            LogMessage ('Loading '+SetDescr[SetType]+' character set '+Regel);
    END;

{ LoadCharSet }

VAR Lp      : BYTE;
    InFile  : TEXT;
    IORes   : BYTE;
    Regel   : STRING;
    Keyword : STRING[20];
    P       : BYTE;
    Map1,
    Map2    : WORD;
    Nop     : ValNop;

BEGIN
     SetName:=UpCaseString (SetName);

     { kijk of de karakter set al ingeladen is }
     FOR Lp:=1 TO CharSetCount DO
         IF (SetType = CharSetPtrs[Lp]^.SetType) AND
            (SetName = UpCaseString (CharSetPtrs[Lp]^.SetName)) THEN
         BEGIN
              IF (CharSetPtrs[Lp]^.UsageCount < 255) THEN
                 Inc (CharSetPtrs[Lp]^.UsageCount);

              LoadCharSet:=Lp;
              Exit; { gevonden }
         END;

     { default result code, in case set is not found }
     IF (SetType = stFtn) THEN
     BEGIN
          IF OutSet THEN
             LoadCharSet:=FtnOutSetNr
          ELSE
              LoadCharSet:=FtnInSetNr;
     END ELSE
     BEGIN
          IF OutSet THEN
             LoadCharSet:=MimeOutSetNr
          ELSE
              LoadCharSet:=MimeInSetNr;
     END;

     { laadt de set in }
     Assign (InFile,Config.SystemDir+CHARSETS_FILENAME);
     {$I-} Reset (InFile); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          { log de fout en default to the default character set }
          LogDiskIOError (IORes,'Cannot re-open '+Config.SystemDir+CHARSETS_FILENAME);

          LogExtraMessage ('Cannot load charset "'+SetName+'"; using default');

          Exit;
     END;

     { Open gelukt. Lees tot de character set definitie gevonden is }
     SetNr:=0; { assume not found }

     WHILE (NOT Eof (InFile)) DO
     BEGIN
          ReadLn (InFile,Regel);

          IF (Regel = '') OR (Regel[1] = ';') THEN
             Continue;

          Regel:=DeleteFrontAndBackSpaces (CleanTabs (Regel,1));

          IF (Regel = '') THEN
             Continue;

          P:=Pos (';',Regel);
          IF (P > 0) THEN
             Regel:=DeleteBackSpaces (Copy (Regel,1,P-1));

          IF (Regel = 'SET-END') THEN
          BEGIN
               { einde van een set bereikt }

               { waren we deze set aan het inladen? }
               IF (SetNr <> 0) THEN
               BEGIN
                    LoadCharSet:=SetNr;
                    Close (InFile);
                    Exit;
               END;

               { ignore }

               Continue;
          END;

          P:=Pos (' ',Regel);
          IF (P = 0) THEN
             Continue; { warned for elsewhere }

          Keyword:=UpCaseString (Copy (Regel,1,P-1));
          Delete (Regel,1,P);
          Regel:=DeleteFrontSpaces (Regel);

          IF (Keyword = 'FTN-SET') THEN
          BEGIN
               { kijk of dit de juiste set is }
               IF (SetType = stFtn) THEN
                  CheckSetStart (Regel);

               Continue;
          END;

          IF (Keyword = 'MIME-SET') THEN
          BEGIN
               { kijk of dit de juiste set is }
               IF (SetType = stMime) THEN
                  CheckSetStart (Regel);

               Continue;
          END;

          IF (SetNr = 0) THEN
             Continue; { niet de juiste set }

          IF (Keyword = 'MAP-IN') THEN
          BEGIN
               { the real thing! }
               P:=Pos (' ',Regel);

               Val (Copy (Regel,1,P-1),Map1,Nop);

               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);

               Val (Regel,Map2,Nop);
               IF (Nop <> 0) THEN
                  Map2:=Ord (Regel[1]);

               CharSetPtrs[SetNr]^.MapIn[Map1]:=Map2;

               Continue;
          END;

          IF (Keyword = 'MAP-OUT') THEN
          BEGIN
               { the real thing! }
               P:=Pos (' ',Regel);

               Val (Copy (Regel,1,P-1),Map1,Nop);

               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);

               Val (Regel,Map2,Nop);
               IF (Nop <> 0) THEN
                  Map2:=Ord (Regel[1]);

               CharSetPtrs[SetNr]^.MapOut[Map1]:=Map2;

               Continue;
          END;

          { ignore other keywords }

     END; { while }

     Close (InFile);

     IF (SetNr = 0) THEN
     BEGIN
          { als de karakterset niet gevonden is, dan maar de default nemen }
          LogExtraMessage ('Charset "'+SetName+'" not found; using default');
          Exit; { default set nr already filled in }
     END;

     LoadCharSet:=SetNr;
END;


{--------------------------------------------------------------------------}
{ InitCharSets                                                             }
{                                                                          }
{ Deze routine initialiseert de multi-character set handling. It checks    }
{ CHARSETS.TDB file syntax and loads the two default character sets.       }
{ Returns FALSE if something fatal occured and WaterGate should be halted. }
{                                                                          }
FUNCTION InitCharSets : BOOLEAN;

VAR SetCount : BYTE;
    SetPtrs  : ARRAY[1..255] OF StringPtr;
    SetTypes : ARRAY[1..255] OF CharSetType;
    LineNr   : WORD;
    Path     : STRING;

    { check if we know a certain set name }
    FUNCTION FindSet (SetName : STRING; SetType : CharSetType) : BOOLEAN;

    VAR Lp : BYTE;

    BEGIN
         FindSet:=TRUE; { assume found }

         FOR Lp:=1 TO SetCount DO
             IF (SetType = SetTypes[Lp]) AND (SetName = SetPtrs[Lp]^) THEN
                Exit; { true }

         WriteLn (' Cannot find ',SetDescr[SetType],' set "'+SetName+'" referenced in line ',LineNr);
         WriteLn ('  of ',Path);

         FindSet:=FALSE; { not found }
    END;

    { Add a found set name to the table of found sets }
    FUNCTION AddSet (SetName : STRING; SetType : CharSetType) : BOOLEAN;

    VAR Lp : BYTE;

    BEGIN
         AddSet:=FALSE; { assume error }

         IF (Length (SetName) > MAXLEN_SETNAME) THEN
         BEGIN
              WriteLn (' Character set name is too long (max ',MAXLEN_SETNAME,')');
              WriteLn ('  in line ',LineNr,' of ',Path);
              Exit;
         END;

         IF (SetCount = 255) THEN
         BEGIN
              WriteLn (' Maximum is 255 sets in line ',LineNr,' of ',Path);
              Exit;
         END;

         SetName:=UpCaseString (SetName);

         Inc (SetCount);
         GetMem (SetPtrs[SetCount],Length (SetName)+1);
         SetPtrs[SetCount]^:=SetName;
         SetTypes[SetCount]:=SetType;

         AddSet:=TRUE; { no problem }
    END;

    { free up memory used by the set storage }
    PROCEDURE FreeSetNames;
    BEGIN
         WHILE (SetCount > 0) DO
         BEGIN
              FreeMem (SetPtrs[SetCount],Length (SetPtrs[SetCount]^)+1);
              Dec (SetCount);
         END; { while }
    END;

{ InitCharSets }

VAR InFile      : TEXT;
    IORes       : BYTE;
    Keyword,
    Regel       : STRING;
    P           : BYTE;
    InSet       : BOOLEAN;
    Map1,Map2   : INTEGER;
    Nop         : ValNop;

LABEL Abort;

BEGIN
     InitCharSets:=FALSE; { assume problems }

     Path:=Config.SystemDir+CHARSETS_FILENAME;

     Assign (InFile,Path);
     {$I-} Reset (InFile); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          IF (IORes = 2) OR (IORES = 3) THEN
             WriteLn (' Cannot find '+Path)
          ELSE
              WriteLn (' Error opening '+Path);

          LogDiskIOError (IORes,'Error open '+Path);

          Exit; { FALSE }
     END;

     { doorloop de hele file en doe een syntax check }

     LineNr:=0;
     InSet:=FALSE;
     SetCount:=0;

     FtnIn:='';
     FtnOut:='';
     MimeIn:='';
     MimeOut:='';

     WHILE (NOT Eof (InFile)) DO
     BEGIN
          ReadLn (InFile,Regel);
          Inc (LineNr);

          IF (Regel = '') OR (Regel[1] = ';') THEN
             Continue;

          { tabs omzetten in spaties en alle voorloop en aanvul spaties verwijderen }
          Regel:=DeleteFrontAndBackSpaces (CleanTabs (Regel,1));

          IF (Regel = '') OR (Regel[1] = ';') THEN
             Continue;

          { commentaar verwijderen }
          P:=Pos (';',Regel);
          IF (P > 0) THEN
             Regel:=DeleteBackSpaces (Copy (Regel,1,P-1));

          IF (Regel = '') THEN
             Continue;

          IF (Regel = 'SET-END') THEN
          BEGIN
               { keyword without argument }
               IF (NOT InSet) THEN
               BEGIN
                    WriteLn (' Need FTN-SET / MIME-SET keyword before '+Keyword+' in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               InSet:=FALSE;

               Continue;
          END;

          { nu moet er <keyword><space><rest> staan }
          P:=Pos (' ',Regel);

          IF (P = 0) THEN
          BEGIN
               WriteLn (' Syntax error in line ',LineNr,' in ',Path);
               GOTO Abort;
          END;

          { haal het keyword uit de regel }
          Keyword:=UpCaseString (Copy (Regel,1,P-1));

          Delete (Regel,1,P);
          Regel:=DeleteFrontSpaces (Regel);

          { --- default set indicator? }

          IF (Keyword = 'FTN-IN') THEN
          BEGIN
               FtnIn:=Regel;
               Continue;
          END;

          IF (Keyword = 'FTN-OUT') THEN
          BEGIN
               FtnOut:=Regel;
               Continue;
          END;

          IF (Keyword = 'MIME-IN') THEN
          BEGIN
               MimeIn:=Regel; { case is important! }
               Continue;
          END;

          IF (Keyword = 'MIME-OUT') THEN
          BEGIN
               MimeOut:=Regel; { case is important! }
               Continue;
          END;

          { --- character set definition? }

          IF (Keyword = 'FTN-SET') THEN
          BEGIN
               { nieuwe set definitie }
               IF (NOT AddSet (UpCaseString (Regel),stFtn)) THEN
                  GOTO Abort;

               InSet:=TRUE;
               Continue;
          END;

          IF (Keyword = 'MIME-SET') THEN
          BEGIN
               IF (NOT AddSet (UpCaseString (Regel),stMime)) THEN
                  GOTO Abort;

               InSet:=TRUE;
               Continue;
          END;

          { als we nu niet in een set zitten, dan niet mappen }
          IF (NOT InSet) THEN
          BEGIN
               WriteLn (' Need FTN-SET / MIME-SET keyword before '+Keyword+' in line ',LineNr);
               WriteLn ('  of ',Path);
               GOTO Abort;
          END;

          IF (Keyword = 'MAP-IN') THEN
          BEGIN
               { check spelling }
               P:=Pos (' ',Regel);

               IF (P = 0) THEN
               BEGIN
                    WriteLn (' Second argument for MAP-IN not present in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Val (Copy (Regel,1,P-1),Map1,Nop);
               IF (Nop <> 0) THEN
               BEGIN
                    WriteLn (' Error in first value of MAP-IN statement in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               IF (Map1 < 128) OR (Map1 > 255) THEN
               BEGIN
                    WriteLn (' First value out of range (128..255) in MAP-IN statement in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);

               IF (Regel = '') THEN
               BEGIN
                    WriteLn (' Second value missing in MAP-IN statement in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Val (Regel,Map2,Nop);

               IF (Nop <> 0) THEN
               BEGIN
                    Map2:=Ord (Regel[1]);
                    Nop:=0;
               END;

               IF (Nop <> 0) THEN
               BEGIN
                    WriteLn (' Error in second value in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               IF (Map2 < 32) OR (Map2 > 255) THEN
               BEGIN
                    WriteLn (' Second value out of range (32..255) in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Continue;
          END;

          IF (Keyword = 'MAP-OUT') THEN
          BEGIN
               { check spelling }
               P:=Pos (' ',Regel);

               IF (P = 0) THEN
               BEGIN
                    WriteLn (' Second argument for MAP-OUT not present in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Val (Copy (Regel,1,P-1),Map1,Nop);
               IF (Nop <> 0) THEN
               BEGIN
                    WriteLn (' Error in first value of MAP-OUT statement in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               IF (Map1 < 128) OR (Map1 > LAST_INTERNAL_CODE) THEN
               BEGIN
                    WriteLn (' First value out of range (128..',LAST_INTERNAL_CODE,') in MAP-OUT statement in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);

               IF (Regel = '') THEN
               BEGIN
                    WriteLn (' Second value missing in MAP-OUT statement in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Val (Regel,Map2,Nop);

               IF (Nop <> 0) THEN
               BEGIN
                    Map2:=Ord (Regel[1]);
                    Nop:=0;
               END;

               IF (Nop <> 0) THEN
               BEGIN
                    WriteLn (' Error in second value in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               IF (Map2 < 1) OR (Map2 > 255) THEN
               BEGIN
                    WriteLn (' Second value out of range (1..255) in line ',LineNr);
                    WriteLn ('  of ',Path);
                    GOTO Abort;
               END;

               Continue;
          END;

          { unknown keyword }

          WriteLn (' Unknown keyword "'+Keyword+'" in line ',LineNr);
          WriteLn ('  of ',Path);

          GOTO Abort;

     END; { while }

     { controleer of de default sets voorkwamen }
     IF (FtnIn= '') THEN
        WriteLn (' FTN-IN set not defined in '+Path);

     IF (FtnOut = '') THEN
        WriteLn (' FTN-OUT set not defined in '+Path);

     IF (MimeIn = '') THEN
        WriteLn (' MIME-IN set not defined in '+Path);

     IF (MimeOut = '') THEN
        WriteLn (' MIME-OUT set not defined in '+Path);

     IF (MimeIn = '') OR (MimeOut = '') OR (FtnIn = '') OR (FtnOut = '') THEN
        GOTO Abort;

     IF (NOT FindSet (UpCaseString (FtnIn),stFtn)) THEN
     BEGIN
          WriteLn ('  (the FTN-IN set!!)');
          GOTO Abort;
     END;

     IF (NOT FindSet (UpCaseString (FtnOut),stFtn)) THEN
     BEGIN
          WriteLn ('  (the FTN-OUT set!!)');
          GOTO Abort;
     END;

     IF (NOT FindSet (UpCaseString (MimeIn),stMime)) THEN
     BEGIN
          WriteLn ('  (the MIME-IN set!!)');
          GOTO Abort;
     END;

     IF (NOT FindSet (UpCaseString (MimeOut),stMime)) THEN
     BEGIN
          WriteLn ('  (the MIME-OUT set!!)');
          GOTO Abort;
     END;

     InitCharSets:=TRUE; { no problems }

Abort:
     Close (InFile);

     FreeSetNames;
END;


{--------------------------------------------------------------------------}
{ LoadDefaultCharSets                                                      }
{                                                                          }
{ Deze routine laadt de default karakter sets in nadat het systeem         }
{ overgeschakeld is naar full-screen mode.                                 }
{                                                                          }
PROCEDURE LoadDefaultCharSets;
BEGIN
     FtnInSetNr:=LoadCharSet (FtnIn,stFtn,FALSE{in});
     FtnOutSetNr:=LoadCharSet (FtnOut,stFtn,TRUE{out});

     MimeInSetNr:=LoadCharSet (MimeIn,stMime,FALSE{in});
     MimeOutSetNr:=LoadCharSet (MimeOut,stMime,FALSE{out});

     { stop het hoogste nummer in FixedSets, zodat deze sets nooit }
     { vrijgegeven worden.                                         }

     FixedSets:=FtnInSetNr;

     IF (FtnOutSetNr > FixedSets) THEN
        FixedSets:=FtnOutSetNr;

     IF (MimeInSetNr > FixedSets) THEN
        FixedSets:=MimeInSetNr;

     IF (MimeOutSetNr > FixedSets) THEN
        FixedSets:=MimeOutSetNr;
END;


{--------------------------------------------------------------------------}
{ JunkCharSets                                                             }
{                                                                          }
{ Frees up all memory used by the multiple character set support.          }
{                                                                          }
PROCEDURE JunkCharSets;
BEGIN
     WHILE (CharSetCount > 0) DO
     BEGIN
          FreeMem (CharSetPtrs[CharSetCount],SizeOf (CharSetRecord));
          CharSetPtrs[CharSetCount]:=NIL;
          Dec (CharSetCount);
     END; { while }
END;


{--------------------------------------------------------------------------}
{ CheckForMimeHeaders                                                      }
{                                                                          }
{ Deze routine kijkt of de regel een MIME header of boundary indicatie     }
{ bevat. Zoja, dan worden de stuur vlaggen aangepast.                      }
{                                                                          }
PROCEDURE CheckForMimeHeaders (VAR Regel : STRING); FAR;

VAR UpRegel : STRING;
    P       : BYTE;

BEGIN
     { boundary check hier toevoegen }


     { niet alles onnodig gaan upcasen
     { RWI 970204: check al uitgevoerd vOOr aanroep..
     IF (Pos (': ',Regel) < 12) THEN
        Exit;
     }

     UpRegel:=UpCaseString (Regel);

     IF (Copy (UpRegel,1,14) = 'CONTENT-TYPE: ') THEN
     BEGIN
          Delete (UpRegel,1,14);

          IF (Pos ('MULTIPART',UpRegel) > 0) THEN
             MimeIsMulti:=TRUE;

          { voorkom het vertalen van 8-bit encoded plaatjes! }
          { dus controleer op text. Anders niet eens verwerken }
          MimeIsText:=(Copy (UpRegel,1,4) = 'TEXT') OR (Copy (UpRegel,1,7) = 'MESSAGE');

          { kijk of er een charset in voorkomt }
          { zoniet, neem dan de default }
          P:=Pos ('CHARSET',UpRegel);
          IF (P > 1) AND (UpRegel[P-1] IN [' ',#9,';',',']) THEN
          BEGIN
               { alles tot en met de charset verwijderen }
               Delete (UpRegel,1,P+6);

               { andere variabelen strippen }
               P:=Pos (';',UpRegel);
               IF (P > 0) THEN
                  UpRegel:=Copy (UpRegel,1,P-1);

               { spaties tot en met de = verwijderen }
               IF (Pos ('=',UpRegel) > 0) THEN
               BEGIN
                    WHILE (UpRegel[1] <> '=') DO
                          Delete (UpRegel,1,1);

                    { = zelf verwijderen }
                    Delete (UpRegel,1,1);

                    { nog meer spaties overslaan }
                    WHILE (UpRegel[1] = ' ') DO
                          Delete (UpRegel,1,1);

                    { CR aan het einde verwijderen }
                    P:=Length (UpRegel);
                    IF (UpRegel[P] = #13) THEN
                       Delete (UpRegel,P,1);

                    { kijk of er aanhalingstekens omheen staan }
                    IF (UpRegel[1] = '"') THEN
                       Delete (UpRegel,1,1);

                    P:=Length (UpRegel);
                    IF (UpRegel[P] = '"') THEN
                       Delete (UpRegel,P,1);

                    { laadt de karakterset }
                    MimeCharSetNr:=LoadCharSet (UpRegel,stMime,FALSE{in});
               END;
          END;

          { hoeven niet verder te zoeken }
          Exit;
     END;

     IF (Copy (UpRegel,1,27) = 'CONTENT-TRANSFER-ENCODING: ') THEN
     BEGIN
          { kijk of we quoted-printable decoding moeten doen }
          MimeIsQP:=(Pos ('QUOTED-PRINTABLE',UpRegel) > 0);
          Exit;
     END;
END;


{--------------------------------------------------------------------------}
{ Mime_TranslateBodyLineToFtn                                              }
{                                                                          }
{ Deze routine verwerkt e'e'n regel van de body van een MIME bericht.      }
{                                                                          }
PROCEDURE Mime_TranslateBodyLineToFtn (VAR OrigRegel : STRING); FAR;

VAR P     : BYTE;
    Regel : STRING;
    Digit : CHAR;
    Getal : BYTE;

BEGIN
     IF (Pos (': ',OrigRegel) >= 12) THEN
        CheckForMimeHeaders (OrigRegel);

     { deze check zorgt ervoor de berichten met 8-bit
       karakters niet vertaald worden!

     IF (NOT MimeIsQP) THEN
     BEGIN
          MsgsAddLineToNoEOL (Body,OrigRegel);
          Exit;
     END;
     }

     { niet direct de originele regel veranderen, want die is opgeslagen }
     { en dat geeft problememen bij het geheugen vrijgeven.              }
     Regel:=OrigRegel;

     IF MimeIsQP THEN
     BEGIN
          { translate soft returns }
          REPEAT
                P:=Pos ('='+#13,Regel);
                IF (P > 0) THEN
                   Delete (Regel,P,2);
          UNTIL (P = 0);
     END;

     { kan niet met Pos doorzoeken ivm gewone = tekens (=3D) }
     P:=1;
     WHILE (P <= Length (Regel)) DO
     BEGIN
          IF MimeIsQP AND (Regel[P] = '=') AND
             (Regel[P+1] IN ['0'..'9','A'..'F','a'..'f']) AND
             (Regel[P+2] IN ['0'..'9','A'..'F','a'..'f']) THEN
          BEGIN
               { lokale hex vertaling voor hoge snelheid }
               Digit:=UpCase (Regel[P+1]);

               IF (Digit IN ['0'..'9']) THEN
                  Getal:=Ord (Digit)-Ord ('0')
               ELSE
                   IF (Digit IN ['A'..'F']) THEN
                      Getal:=Ord (Digit)-Ord ('A')+10;

               Getal:=Getal SHL 4;

               Digit:=UpCase (Regel[P+2]);

               IF (Digit IN ['0'..'9']) THEN
                  Getal:=Getal OR (Ord (Digit)-Ord ('0'))
               ELSE
                   IF (Digit IN ['A'..'F']) THEN
                      Getal:=Getal OR (Ord (Digit)-Ord ('A')+10);

               { opslaan als 8-bit data }
               Regel[P]:=Char (Getal);
               Delete (Regel,P+1,2);
          END;

          { als het een 8-bit teken is, dan vertalen we de karakterset code }
          IF (MimeCharSetNr > 0) THEN
          BEGIN
               Getal:=Byte (Regel[P]);

               { Getal nu in MIME character set }
               IF (Getal >= 128) THEN
               BEGIN
                    { MAP Getal naar interne set }
                    { dit kan ook gewone ASCII zijn!! }
                    Getal:=CharSetPtrs[MimeCharSetNr]^.MapIn[Getal];

                    { MAP Getal nu naar de FTN set }
                    IF (Getal >= 128) THEN
                       Getal:=CharSetPtrs[FtnOutSetNr]^.MapOut[Getal];

                    { RWI 970112: prevent 00's in the body because of errors }
                    IF (Getal = 0) THEN
                       Getal:=Ord ('?');

                    { Getal nu in de FTN set }
                    Regel[P]:=Char (Getal);
               END;
          END;

          Inc (P);
          IF (P = 0) THEN
             Break;
     END; { while }

     MsgsAddLineToNoEOL (Body,Regel);
END;


VAR FoundEncodingHeader : BOOLEAN;

{--------------------------------------------------------------------------}
{ CopyMimeContentHeadersToBody                                             }
{                                                                          }
{ Deze routine kijkt of een header een MIME regel is die aan geeft dat de  }
{ body een encoded file bevat. Zoja, dan wordt deze regel in de body       }
{ gezet.                                                                   }
{                                                                          }
PROCEDURE CopyMimeContentHeadersToBody (VAR OrigRegel : STRING); FAR;

VAR UpRegel : STRING;

BEGIN
     UpRegel:=UpCaseString (OrigRegel);

     IF (Copy (UpRegel,1,8) = 'CONTENT-') THEN
     BEGIN
          MsgsAddLineToNoEOL (Body,OrigRegel);

          { check for known methods that will trigger the decoder }
          IF (Pos ('APPLICATION/',UpRegel) > 0) OR
             (Pos ('UNKNOWN/',UpRegel) > 0) OR
             (Pos ('IMAGE/',UpRegel) > 0) OR
             (Pos ('BASE64',UpRegel) > 0)
          THEN
              FoundEncodingHeader:=TRUE;
     END;
END;


{--------------------------------------------------------------------------}
{ MimeBodyToFtn                                                            }
{                                                                          }
{ Deze routine scant de body van een MIME bericht en zet alle =XX en =<CR> }
{ codes om in 8-bit codes. Als de body al 8-bit was, dan blijft dat zo.    }
{ Na die vertaling wordt de karakterset omgezet naar een FTN karakter set. }
{                                                                          }
PROCEDURE MimeBodyToFtn;

VAR OldBody : TopRegelRecordPtr;

BEGIN
     IF Config.LogDebug THEN
        LogMessage ('Translating MIME body to FTN');

     { initialiseer default variabelen }
     MimeIsText:=TRUE;
     MimeIsQP:=FALSE;
     MimeCharSetNr:=MimeInSetNr;
     MimeIsMulti:=FALSE;

     { doorloop de RFC headers voor MIME headers }
     MsgsForEach (Msg.HeaderTop_U,CheckForMimeHeaders);

     OldBody:=Msg.BodyTop;
     Msg.BodyTop:=NIL;

     IF (NOT MimeIsMulti) THEN
     BEGIN
          FoundEncodingHeader:=FALSE;

          MsgsForEach (Msg.HeaderTop_U,CopyMimeContentHeadersToBody);

          { lege regel tussen headers en encoded block }
          IF FoundEncodingHeader THEN
             MsgsAddLineTo (Body,'') { empty line between headers and encoded block }
          ELSE
              MsgsReleaseLines (Msg.BodyTop);
     END;

     { doorloop nu de body en doe het vertaalwerk }
     MsgsForEachKill (OldBody,Mime_TranslateBodyLineToFtn);
END;


{--------------------------------------------------------------------------}
{ Ftn_CheckIfNeedQuotedPrintable                                           }
{                                                                          }
{ Deze routine doorloopt iedere regel van de body om te kijken of we       }
{ Quoted-printable encoding nodig hebben. Als de body geen 8-bit tekens    }
{ bevat, dan gebruiken we geen quoted-printable encoding. Ook = tekens     }
{ blijven dan gewone = tekens.                                             }
{ Quoted printable betekent ook automatisch soft line breaks. Dit kan      }
{ geforceerd worden van de config.                                         }
{                                                                          }
PROCEDURE Ftn_CheckIfNeedQuotedPrintable (TopRegelPtr : TopRegelRecordPtr);

VAR EenRegelPtr : EenRegelRecordPtr;
    RegelLength : BYTE;
    Regel       : STRING; { voor swapped regels }
    Lp          : BYTE;
    Teken       : BYTE;

BEGIN
     FtnDoQuotePrn:=FALSE; { do not use quoted printable encoding }

     IF (TopRegelPtr = NIL) THEN
        Exit;

     EenRegelPtr:=TopRegelPtr^.FirstRegelRecordPtr;
     MsgsNewSeek (EenRegelPtr);

     WHILE (EenRegelPtr <> NIL) DO
     BEGIN
          CASE EenRegelPtr^.Waar OF
               wMem :
                   BEGIN
                        Regel:=EenRegelPtr^.RegelPtr^;
                        EenRegelPtr:=EenRegelPtr^.NextRegelRecordPtr;
                        MsgsNewSeek (EenRegelPtr);
                   END;

               wSwapped :
                   BEGIN
                        { lees de lengte van de regel in }
                        BlockRead (SwapFile,RegelLength,1);

                        { einde van het swapped blok? }
                        IF (RegelLength = 0) THEN
                        BEGIN
                             { ja, ga naar de volgende regel }
                             EenRegelPtr:=EenRegelPtr^.NextRegelRecordPtr;
                             MsgsNewSeek (EenRegelPtr);
                             Continue;
                        END;

                        { lees de regel zelf in }
                        BlockRead (SwapFile,Regel[1],RegelLength);
                        Regel[0]:=Char (RegelLength);
                   END;
          END; { case }

          FOR Lp:=1 TO Length (Regel) DO
              IF NOT (Byte (Regel[Lp]) IN [13,32..127]) THEN
              BEGIN
                   { first do the character set translation before   }
                   { we are really sure. This is simple, because an  }
                   { FTN message has only one character set. So far. }

                   Teken:=Byte (Regel[Lp]);

                   { voorkom array access voor [0..12,14..31] }
                   IF (Teken >= 128) THEN
                   BEGIN
                        { vertaal naar interne set }
                        Teken:=CharSetPtrs[FtnCharSetNr]^.MapIn[Teken];

                        { nu een interne code, of een gewone ascii code }
                        IF (Teken >= 128) THEN
                           Teken:=CharSetPtrs[MimeOutSetNr]^.MapOut[Teken];
                   END;

                   IF NOT (Teken IN [13,32..127]) THEN
                   BEGIN
                        FtnDoQuotePrn:=TRUE;
                        Exit;
                   END;
              END;

     END; { while }
END;


{---------------------------------------------------------------------------}
{ Ftn_UnparagraphBody                                                       }
{                                                                           }
{ Deze routine kijkt of de regel te lang is en zoja, breekt em af op de     }
{ opgegeven wrap positie. De regels worden aan de body toegevoegd.          }
{                                                                           }
{ Werkt als volgt: we tellen het aantal tekens sinds de laatste CR en als   }
{ dat meer dan Config.LineWrapLen is, dan zoeken we de woord grens en       }
{ voegen een extra CR toe.                                                  }
{                                                                           }
PROCEDURE Ftn_UnparagraphBody (VAR OrigRegel : STRING); FAR;

VAR P     : BYTE;
    Regel : STRING;

BEGIN
     { snelle controle of deze regel zo mee kan }
     P:=Pos (#13,OrigRegel);

     IF (P = Length (OrigRegel)) AND (FtnSinceCR+P <= Config.WrapLineLen) THEN
     BEGIN
          { 90% case: <regel>#13 }
          MsgsAddLineToNoEOL (Body,OrigRegel);
          Inc (FtnLines); { RWI 970130: toegevoegd }
          FtnSinceCR:=0; { weer 0 toegevoegd }
          Exit;
     END;

     { eventueel <regel> zonder CR testen, als ie vaak voorkomt tenminste }

     { nu moeten we toch eerste kopieren voordat we iets kunnen uithalen }
     { want anders vernietigen we OrigRegel en gaat de FreeMem mis.      }
     Regel:=OrigRegel;

     WHILE (Regel <> '') DO
     BEGIN
          { ### kunnen we nu precies Config.WrapLineLen tekens toegevoegd hebben? }

          P:=Pos (#13,Regel);

          { als dit hele stuk zonder CR nog past, dan gewoon doen en klaar }
          IF (P = 0) AND (FtnSinceCR+Length (Regel) < Config.WrapLineLen) THEN
          BEGIN
               Inc (FtnSinceCR,Length (Regel));
               MsgsAddLineTo (Body,Regel);
               Exit;
          END;

          { als het stuk tot de volgende enter nog past, dan gewoon doen }
          IF (P <> 0) AND (FtnSinceCR+P <= Config.WrapLineLen) THEN
          BEGIN
               { RWI 970130: P is inclusief CR, changed to NoEOL }
               MsgsAddLineToNoEOL (Body,Copy (Regel,1,P{inclusive CR}));
               Inc (FtnLines); { RWI 970130: toegevoegd }
               Delete (Regel,1,P);
               FtnSinceCR:=0;
               Continue;
          END;

          { stuk is te lang. Wrappen! }

          { hoeveel mogen we nog toevoegen? }
          { moet altijd een valid offset in Regel zijn!! }
          P:=Config.WrapLineLen-FtnSinceCR;

          { ga op zoek naar de spatie voor wordwrap }
          WHILE (P > 0) AND (Regel[P] <> ' ') DO
                Dec (P);

          IF (P = 0) THEN
          BEGIN
               { geen spatie gevonden. Dat wordt dus keihard afbreken }
               P:=Config.WrapLineLen-FtnSinceCR;
               MsgsAddLineToNoEOL (Body,Copy (Regel,1,P)+#13);
               Delete (Regel,1,P);
          END ELSE
          BEGIN
               { braaf wordwrappen }
               MsgsAddLineToNoEOL (Body,Copy (Regel,1,P-1{spatie weg})+#13);
               Delete (Regel,1,P); { incluis spatie }
               { pech als er twee spaties achter elkaar }
               { staan.                                 }
          END;

          { we hebben nu altijd een CR aan het einde gehad }
          FtnSinceCR:=0;
          Inc (FtnLines); { RWI 970130: toegevoegd }

     END; { while Regel<>'' }
END;


{--------------------------------------------------------------------------}
{ Ftn_TranslateBodyLineToMimeQuotedPrintable                               }
{                                                                          }
{ Deze routine vertaald een regel tekst van FTN formaat naar MIME formaat. }
{ We gebruiken 7BIT content-transfer-encoding en moeten dus hex codering   }
{ gebruiken (=E7 enzo). Karakterset wordt ook omgezet.                     }
{                                                                          }
PROCEDURE Ftn_TranslateBodyLineToMimeQuotedPrintable (VAR OrigRegel : STRING); FAR;

VAR Regel : STRING;

    {----------------------------------------------------------------------}
    { SoftWrappedOutput                                                    }
    {                                                                      }
    { Schrijf een stuk van Regel weg door af te kappen op het laatste      }
    { woord. Voeg een = teken aan het einde toe om een MIME soft break     }
    { aan te geven. Verwijder het weggeschreven stuk uit Regel.            }
    {                                                                      }
    PROCEDURE SoftWrappedOutput;

    VAR P : BYTE;

    BEGIN
         P:=Length (Regel);
         WHILE (P > 30) AND (Regel[P] <> ' ') DO
               Dec (P);

         IF (P = 30) THEN
            P:=Length (Regel);

         MsgsAddLineTo (Body,Copy (Regel,1,P)+'=');
         Inc (FtnLines);

         Delete (Regel,1,P);
    END;

{ Ftn_TranslateBodyLineToMimeQuotedPrintable }

VAR Lp    : BYTE;
    Teken : BYTE;

BEGIN
     Regel:='';

     { zolang er geen 8-bit in voorkomt zijn we veilig }
     FOR Lp:=1 TO Length (OrigRegel) DO
     BEGIN
          Teken:=Byte (OrigRegel[Lp]);

          { RWI 970110: fast processing of valid characters }
          IF (Teken >= 32) AND (Teken < 128) AND (Teken <> Ord ('=')) THEN
          BEGIN
               IF (Length (Regel) >= 76{maxlen}) THEN
                  SoftWrappedOutput;

               Regel:=Regel+Char (Teken);
               Continue;
          END;

          IF (Teken = 13{line break}) THEN
          BEGIN
               MsgsAddLineTo (Body,Regel);  { geen softbreak! }
               Inc (FtnLines);
               Regel:='';
               Continue;
          END;

          IF (Teken >= 128) THEN
          BEGIN
               { teken nu in FTN set }

               { vertaal naar interne set }
               Teken:=CharSetPtrs[FtnCharSetNr]^.MapIn[Teken];

               { nu een interne code, of een gewone ascii code }
               IF (Teken >= 128) THEN
                  Teken:=CharSetPtrs[MimeOutSetNr]^.MapOut[Teken];
          END;

          IF (Teken < 32) OR (Teken >= 128) OR (Teken = Ord ('=')) THEN
          BEGIN
               IF (Length (Regel) > 73{maxlen 76}) THEN
                  SoftWrappedOutput;

               { quoted-printable toevoegen }
               Regel:=Regel+'='+Byte2HexString (Teken);
               FtnHaveHigh:=TRUE;
          END ELSE
          BEGIN
               IF (Length (Regel) >= 76{maxlen}) THEN
                  SoftWrappedOutput;

               Regel:=Regel+Char (Teken);
          END;
     END; { for }

     IF (Regel <> '') THEN
     BEGIN
          MsgsAddLineTo (Body,Regel+'='{MIME soft break});
          Inc (FtnLines);
     END;
END;


{--------------------------------------------------------------------------}
{ FtnBodyToMime                                                            }
{                                                                          }
{ Deze routine zet de body om van een 8-bits FTN karakter set (nu in       }
{ Msg.Chr_F en anders de default set) in een MIME encoded body van het     }
{ gewenste type.                                                           }
{                                                                          }
FUNCTION FtnBodyToMime : LONGINT;

VAR OldBody : TopRegelRecordPtr;

BEGIN
     IF Config.LogDebug THEN
        LogMessage ('Translating FTN body to MIME');

     { initialiseer default variabelen }
     IF (Msg.Chrs_F = '') THEN
        FtnCharSetNr:=FtnInSetNr
     ELSE
         FtnCharSetNr:=LoadCharSet (Msg.Chrs_F,stFtn,FALSE{in});

     FtnLines:=0;
     FtnHaveHigh:=FALSE; { alleen maar 7bit data }

     { doorloop nu de body en doe het vertaalwerk }
     { dit kost dus even twee keer zoveel geheugen/swapfile! }
     OldBody:=Msg.BodyTop;
     Msg.BodyTop:=NIL;

     { RWI 970115: if the sysop doesn't like quoted printable, then }
     {             check if we really need to use it. This avoids   }
     {             encoded equal signs..                            }
     IF Config.AlwaysMimeQuotePrint THEN
        FtnDoQuotePrn:=TRUE
     ELSE
         Ftn_CheckIfNeedQuotedPrintable (OldBody);

     { nu _of_ quoted-printable coderen, of gewoon paragrafen vernietigen }
     IF FtnDoQuotePrn THEN
     BEGIN
          MsgsForEachKill (OldBody,Ftn_TranslateBodyLineToMimeQuotedPrintable);
     END ELSE
     BEGIN
          { afkappen gebeurt op MAXLENREGEL afstand vanaf de laatste CR, }
          { wat door SinceCR geteld wordt.                               }
          FtnSinceCR:=0; { nul tekens sinds laatste CR }

          MsgsForEachKill (OldBody,Ftn_UnparagraphBody);
     END;

     FtnBodyToMime:=FtnLines; { voor Lines: header in news }
END;


{--------------------------------------------------------------------------}
{ BuildCHRSKludge                                                          }
{                                                                          }
{ Deze routine geeft een "^ACHRS: <name> 2" kludge terug.                  }
{                                                                          }
FUNCTION BuildCHRSKludge : STRING;
BEGIN
     BuildCHRSKludge:=#1'CHRS: '+CharSetPtrs[FtnOutSetNr]^.SetName+' 2';
END;


{--------------------------------------------------------------------------}
{ AddStandardMimeHeaders                                                   }
{                                                                          }
{ Deze routine voegt de drie MIME headers toe: version, content and        }
{ encoding.                                                                }
{                                                                          }
PROCEDURE AddStandardMimeHeaders;
BEGIN
     IF FtnHaveHigh OR FtnDoQuotePrn THEN
     BEGIN
          MsgsAddLineTo (Header_U,'MIME-version: 1.0');
          Msg.IsMime:=TRUE; { direct internal RFC -> FTN doesn't work otherwise! }

          IF FtnHaveHigh THEN
             MsgsAddLineTo (Header_U,'Content-type: text/plain; charset='+CharSetPtrs[MimeOutSetNr]^.SetName);

          IF FtnDoQuotePrn THEN
             MsgsAddLineTo (Header_U,'Content-transfer-encoding: Quoted-printable');
     END;
END;


{--------------------------------------------------------------------------}
{ MimeHeaderToFtn                                                          }
{                                                                          }
{ This routine extract the real data from MIME encoded headers in the form }
{ of "=?iso-8859-1?Q?Sen=F5r_David?=" and translates it to an FTN set.     }
{ See RFC1342 for details.                                                 }
{                                                                          }
FUNCTION MimeHeaderToFtn (Line : STRING) : STRING;

VAR P        : BYTE;
    Encoding : CHAR;
    Digit    : CHAR;
    Getal    : BYTE;

BEGIN
     MimeHeaderToFTN:=Line;     { safe copy, just in case }

     { remove the =? and ?= parts }
     IF (Copy (Line,1,2) <> '=?') OR (Copy (Line,Length (Line)-1,2) <> '?=') THEN
        Exit; { keep original }

     Line:=Copy (Line,3,Length (Line)-4);

     { load the character set involved }
     P:=Pos ('?',Line);
     MimeCharSetNr:=LoadCharSet (Copy (Line,1,P-1),stMime,FALSE{in});
     Delete (Line,1,P);

     Encoding:=UpCase (Line[1]);
     IF (Line[2] <> '?') OR (NOT (Encoding IN ['B','Q'])) THEN
        Exit; { keep original }

     Delete (Line,1,2);
     IF (Pos ('?',Line) > 0) THEN
        Exit;

     IF (Encoding = 'Q') THEN
     BEGIN
          { Quoted-printable encoding }

          { Start removing QP encoding }
          { "_" represents a space!    }

          P:=1;
          WHILE (P <= Length (Line)) DO
          BEGIN
               { according to RFC1342!! }
               { do this before decoding possibly encoded underscores! }
               IF (Line[P] = '_') THEN
                  Line[P]:=' ';

               IF (Line[P] = '=') AND (P < Length (Line)-1) AND
                  (Line[P+1] IN ['0'..'9','A'..'F','a'..'f']) AND
                  (Line[P+2] IN ['0'..'9','A'..'F','a'..'f']) THEN
               BEGIN
                    { lokale hex vertaling voor hoge snelheid }
                    Digit:=UpCase (Line[P+1]);

                    IF (Digit IN ['0'..'9']) THEN
                       Getal:=Ord (Digit)-Ord ('0')
                    ELSE
                        IF (Digit IN ['A'..'F']) THEN
                           Getal:=Ord (Digit)-Ord ('A')+10;

                    Getal:=Getal SHL 4;

                    Digit:=UpCase (Line[P+2]);

                    IF (Digit IN ['0'..'9']) THEN
                       Getal:=Getal OR (Ord (Digit)-Ord ('0'))
                    ELSE
                        IF (Digit IN ['A'..'F']) THEN
                           Getal:=Getal OR (Ord (Digit)-Ord ('A')+10);

                    { opslaan als 8-bit data }
                    Line[P]:=Char (Getal);
                    Delete (Line,P+1,2);
               END;

               Inc (P);
               IF (P = 0) THEN
                  Break;
          END; { while }

     END ELSE
         IF (Encoding = 'B') THEN
         BEGIN
              { BASE64 encoding }
              Line:=DecodeBase64Header (Line);

              { needs post-processing }
         END;

     { post-processing MIME->FTN: translate all 8 bit characters }
     IF (MimeCharSetNr > 0) THEN
        FOR P:=1 TO Length (Line) DO
        BEGIN
             Getal:=Byte (Line[P]);

             { Getal nu in MIME character set }
             IF (Getal >= 128) THEN
             BEGIN
                  { MAP Getal naar interne set }
                  { dit kan ook gewone ASCII zijn!! }
                  Getal:=CharSetPtrs[MimeCharSetNr]^.MapIn[Getal];

                  { MAP Getal nu naar de FTN set }
                  IF (Getal >= 128) THEN
                     Getal:=CharSetPtrs[FtnOutSetNr]^.MapOut[Getal];

                  { RWI 970112: prevent 00's in the body because of errors }
                  IF (Getal = 0) THEN
                     Getal:=Ord ('?');

                  { Getal nu in de FTN set }
                  Line[P]:=Char (Getal);
             END;
        END; { for, if }

     MimeHeaderToFTN:=Line; { return the translated version }
END;


{--------------------------------------------------------------------------}
{ Unit Initialization                                                      }
{                                                                          }
BEGIN
     CharSetCount:=0;
END.
