/**********************************************************************
  Copyright (c) 2000-2003,2009 Michael Dillon
  All rights reserved.

  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions 
  are met:

       Redistributions of source code must retain the above copyright 
       notice, this list of conditions and the following disclaimer. 

       Redistributions in binary form must reproduce the above 
       copyright notice, this list of conditions and the following 
       disclaimer in the documentation and/or other materials provided 
       with the distribution. 

       Neither the name of Crystalline Realms, Midnight's Hour BBS
       nor the names of their contributors may be used to endorse or
	   promote products derived from this software without specific
	   prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
  OF THE POSSIBILITY OF SUCH DAMAGE.
 *********************************************************************/
/*********************************************************************
 tinytic.cpp - Master program for TinyTIC.  This houses the heart of
               TinyTIC, handles most of the startup information 
               including config file parsing and invoking of the 
               TIC processor.
 *********************************************************************/
#include "tinytic.h"
#include "parse.h"
#include "ticlog.h"
#include "tprocess.h"

string szHomeDir       = "";
string szWorkDir       = "";

void IdentifyHomeDirectory(const char *arg);
bool ParseCommandLine(int argc, char *argv[]);
void ShowUsage(void);

int main(int argc, char *argv[])
{
        signed short int _MainReturn = 0;

        // Find the current working directory
        szWorkDir = getcwd(NULL, 4096);

        cout << PROGNAME << ": Quick TIC File Processor v" << VERSION_STR << endl;
        cout << COPYR << AUTHOR << ". All Rights Reserved." << endl;
        cout << "Created on " << __DATE__ << " at " << __TIME__ << endl;
        cout << endl;
        cout << endl;
        
#if defined(__TINYTIC_WINDOWS)
        IdentifyHomeDirectory(NULL);
#elif defined(__TINYTIC_POSIX) || defined(__TINYTIC_OS2)
        IdentifyHomeDirectory(argv[0]);
#endif

        if (ParseCommandLine(argc, argv) == false)
        {
                return ( -1 );
        }
        else
        {
                cout << "Reading configuration file..." << endl;
        }

        if (!ReadConfigFile())
        {
                PutLog("Unable to load the configuration file.", FATAL);
                _MainReturn = 1;
        }
        else
        {
                PutLog("Started...", INFO, 1);

                if (!ProcessTICFiles())
                {
                        PutLog("Aborted...", FATAL);
                        _MainReturn = 2;
                }
                else
                {
                        PutLog("Stopped Successfully...", INFO, 1);
                }
        }

        // Go back to the current working directory
        chdir(szWorkDir.c_str());

        // Return
        return (_MainReturn);
}

void IdentifyHomeDirectory(const char *arg)
{
        string::iterator _ident_idx;
#if defined(__TINYTIC_WINDOWS)
        char szBuffer[4096];

        GetModuleFileName(NULL, szBuffer, 4096);
        szHomeDir = szBuffer;
        _ident_idx = strrchr(szHomeDir.c_str(), '\\');
        szHomeDir.erase(++_ident_idx, szHomeDir.end());
#elif defined( __TINYTIC_POSIX )
        string szTmpString = arg;
        string::size_type offset = 0;
        offset = szTmpString.find('/');
        if (offset == string::npos)
        {
                szHomeDir = "./";
        }
        else
        {
                _ident_idx = szTmpString.begin() + offset;
                szTmpString.erase(++_ident_idx, szTmpString.end());
                szHomeDir = szTmpString.c_str();
        }
#elif defined(__TINYTIC_OS2)
        // Behaves just as POSIX does
        string szTmpString = arg;
        if ((_ident_idx = strrchr(szTmpString.c_str(), '\\')) == NULL)
        {
                szHomeDir = ".\\";
        }
        else
        {
                szTmpString.erase(++_ident_idx, szTmpString.end());
                szHomeDir = szTmpString.c_str();
        }
#endif
}

bool ParseCommandLine(int argc, char *argv[])
{
        string szCmdLine;
        bool _ParseCommandLineReturn = true;

        if (argc > 1)
        {
                for (int z = 1; z < argc; z++)
                {

                        szCmdLine = argv[z];

                        if ((stricmp(szCmdLine.c_str(), "--help") == 0) ||
                            (stricmp(szCmdLine.c_str(), "-?") == 0)     ||
                            (stricmp(szCmdLine.c_str(), "-h") == 0))
                        {
                                ShowUsage();
                                _ParseCommandLineReturn = false;
                        }
                        else if ((stricmp(szCmdLine.c_str(), "--config-file") == 0) ||
                                 (stricmp(szCmdLine.c_str(), "-c") == 0))
                        {
						        z++;
                                if (VerifyFileUsable(argv[z], 1) == false)
                                {
                                        cout << "Unable to use the specified config file, ignoring." << endl;
                                }
                                else
                                {
                                          szConfigFile = argv[z];
										  cout << "Using config file \'"
                                                << szConfigFile.c_str()
                                                << "\'." << endl;

                                }
                        }
                        else if ((stricmp( szCmdLine.c_str(), "--data-file") == 0) ||
                                 (stricmp( szCmdLine.c_str(), "-d") == 0))
                        {
                                z++;
                                if (VerifyFileUsable(argv[z]) == false)
                                {
                                        // Default is defined in parse.cpp:ReadConfigFile()
                                        cout << "Unable to use the specified data file, ignoring." << endl;
                                }
                                else
                                {
                                        DataFile = argv[z];
                                        cout << "Using datafile \'"
                                                << DataFile.c_str()
                                                << "\'." << endl;
                                }
                        }
                        else if ((stricmp(szCmdLine.c_str(), "--log-file") == 0) ||
                                 (stricmp(szCmdLine.c_str(), "-l") == 0))
                        {
                                z++;
                                if (VerifyFileUsable(argv[z]) == false)
                                {
                                        cout << "Unable to use the specified log file, ignoring." << endl;
                                }
                                else
                                {
								        szLogFile = argv[z];
                                        cout << "Using logfile \'"
                                                << szLogFile.c_str()
                                                << "\'." << endl;
                                }
                        }
                        else
                        {
                                cout << "Unknown parameter "
                                        << argv[ z ] << endl;
                        }
                }
        }

        return (_ParseCommandLineReturn);
}

void ShowUsage(void)
{
        cout << "Usage: " << PROGNAME << endl;
        cout << "Command Line Options: " << endl;
        cout << endl;
        cout << "     -c,--config-file  : Specify a config file to use other than the default" << endl;
        cout << "                         (e.g. --config-file d:\\tinytic.cfg )" << endl;
        cout << "     -d,--data-file    : Location where the datafile will be stored, Overrides" << endl;
        cout << "                         the one specified in the configuration file." << endl;
        cout << "                         (e.g. --data-file d:\\tinytic.dat )" << endl;
        cout << "     -l,--log-file     : Location where the log file will be stored, Overrides" << endl;
        cout << "                         the one specified in the configuration file." << endl;
        cout << "                         (e.g. --log-file d:\\tinytic.log )" << endl;
        cout << "     -h,-?,--help      : This help screen." << endl;
        cout << endl;
}
        
bool VerifyFileUsable(const char *src, int io)
{
        ofstream        hOFile;
        ifstream        hIFile;
        bool            _VFUReturn = false;

        if (src != NULL && *src != '\0')
        {
                if (io == 0)
                {
                         hOFile.open(src, ios::app);
                }
                else
                {
                         hIFile.open(src, ios::in);
                }

                if (io == 0 && hOFile.fail() == false)
				{
                         _VFUReturn = true;
                         hOFile.close();
                }
                else if (io != 0 && hIFile.fail() == false)
				{
                          _VFUReturn = true;
                          hIFile.close();
				}
        }

        return (_VFUReturn);
}
