/**********************************************************************
  Copyright (c) 2000-2003,2009 Michael Dillon
  All rights reserved.

  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions 
  are met:

       Redistributions of source code must retain the above copyright 
       notice, this list of conditions and the following disclaimer. 

       Redistributions in binary form must reproduce the above 
       copyright notice, this list of conditions and the following 
       disclaimer in the documentation and/or other materials provided 
       with the distribution. 

       Neither the name of Crystalline Realms, Midnight's Hour BBS
       nor the names of their contributors may be used to endorse or
	   promote products derived from this software without specific
	   prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
  OF THE POSSIBILITY OF SUCH DAMAGE.
 *********************************************************************/
/*********************************************************************
 CDirect.cpp - Cross-platform directory searching class.
 *********************************************************************/
#include "CDirect.h"

// Default constructor, looks in current directory and searches for all files
CDirect::CDirect()
{
        // Default to searching for all files
        _pattern = "*.*";
        // Default to current directory
        _directory = ".";

        // Flag the directory as "Not-in-use"
        _open = false;

        // Call the FixPath to make sure the directory has a trailing slash
        FixPath();

        // Set the Max count to 0
        _MaxFiles = 0L;

        // Get the max count
        GetMaxFiles();
}

// Secondary constructor, allows for specified directory and match pattern
CDirect::CDirect( const char *pattern, const char *directory )
{
        // Use the given pattern
        _pattern = pattern;
        // Use the given directory
        _directory = directory;

        // flag the directory as "Not-In-Use"
        _open = false;

        // Call FixPath to make sure the directory has a trailing slash
        FixPath();

        // Set the Max count to 0
        _MaxFiles = 0L;

        // Get the max count
        GetMaxFiles();
}

// Destructor, closes anything that was opened
CDirect::~CDirect()
{
        // Check to see if the directory was in use or was opened.
        if (_open == true)
        {
#if defined(__DIRECT__FILE_SEARCH_WINDOWS) || defined(__DIRECT__FILE_SEARCH_OS2)
// -----------------------
// Windows & OS/2 Specific
// -----------------------
                // Close the find search
                _findclose(hFileSearch);
#elif defined(__DIRECT_FILE_SEARCH_POSIX)
// -----------------------
// POSIX Specific
// -----------------------
                // Close the open directory
                closedir(hDir);
#endif
        }
}

// Find:
//      Searches for entries of files matching the pattern
//
bool    CDirect::Find( void )
{
        // Return value for this function
        bool _FindReturn = false;

        // Has the directory been accessed already ?
        if (_open == false)
        {
                // No, find the first file entry matching the pattern.
                _open = FindFirst();
                // Set the return to that of FindFirst();
                _FindReturn = _open;
        }
        else
        {
                // Yes, Find the next file entry
                _FindReturn = FindNext();
        }

        // Return
        return ( _FindReturn );
}
                                                
// FindFirst: Searches for the first entry matching the pattern
bool CDirect::FindFirst(void)
{
        // Return value for FindFirst()
        bool  _FindFirstReturn = false;

#if defined(__DIRECT__FILE_SEARCH_WINDOWS) || defined(__DIRECT__FILE_SEARCH_OS2)
// ------------------------
// Windows & OS/2 Specific
// ------------------------
        // Create a temporary buffer
        string  szTmpBuf;

        // Copy the given directory to the temporary buffer
        szTmpBuf = _directory.c_str();

        // Copy the pattern to the buffer
        szTmpBuf += _pattern.c_str();

        // Search for the first entry matching the pattern
        if ((hFileSearch = _findfirst(szTmpBuf.c_str(), &hFindData)) > -1L)
        {
                // Entry found, update the hFileData structure
                ValidateData();
                // Set the return to true
                _FindFirstReturn = true;
        }
#elif defined(__DIRECT__FILE_SEARCH_POSIX)
// -------------------------
// POSIX Specific
// -------------------------
        // Attempt to open the specified directory
        if ((hDir = opendir(_directory.c_str())) != NULL)
        {
                // Read each entry of the directory until a match was found
                // or the directory has no more entries.
                while ((hDirent = readdir(hDir)) != NULL)
                {
                        // Check to see if the file's name matches the given
                        // pattern.
#if defined(DJGPP)
                        if (fnmatch(_pattern.c_str(), hDirent->d_name, FNM_NOESCAPE | FNM_NOCASE) == 0)
#else
                        if (fnmatch(_pattern.c_str(), hDirent->d_name, 0) == 0)
#endif
                        {
                                // Yes, update the hFileData structure
                                ValidateData();
                                // Set the return to true                                
                                _FindFirstReturn = true;
                                // Get out of the loop
                                break;
                        }
                }
        }
#endif

        // Return
        return (_FindFirstReturn);
}

// FindNext: Searches for the next entry in the directory matching the pattern
bool CDirect::FindNext(void)
{
        // Clear the value of hFileData
        DestroyValues();
        // Return value for FindNext
        bool    _FindNextReturn = false;

#if defined(__DIRECT__FILE_SEARCH_WINDOWS) || defined(__DIRECT__FILE_SEARCH_OS2)
// ------------------------
// Windows & OS/2 Specific
// ------------------------
        // Search for the next entry
        if (_findnext(hFileSearch, &hFindData) == 0)
        {
                // Found, Update the hFileData structure
                ValidateData();
                // Set the return to true
                _FindNextReturn = true;
        }
#elif defined(__DIRECT__FILE_SEARCH_POSIX)
// -------------------------
// POSIX Specific
// -------------------------
        // Attempt to read the directory for more entries or
        // until there are no more entries to be found.
        while ( ( hDirent = readdir( hDir ) ) != NULL )
        {
                // Check to see if the file name matches that of the pattern
#if defined(DJGPP)
                if (fnmatch(_pattern.c_str(), hDirent->d_name, FNM_NOESCAPE | FNM_NOCASE) == 0)
#else
                if (fnmatch(_pattern.c_str(), hDirent->d_name, 0) == 0)
#endif
                {
                        // Yes, update the hFileData structure
                        ValidateData();
                        // Set the return to true
                        _FindNextReturn = true;
                        // Get Out of the loop
                        break;
                }
        }
#endif

        // Return
        return (_FindNextReturn);
}

// ValidateData: Updates the values for the hFileData structure
void CDirect::ValidateData(void)
{
        // Temporary string
        string szTmpBuffer;

        // Copy the directory to the TempBuffer
        szTmpBuffer = _directory.c_str();

#if defined(__DIRECT__FILE_SEARCH_WINDOWS) || defined(__DIRECT__FILE_SEARCH_OS2)
// -----------------------
// Windows & OS/2 Specific
// -----------------------
        // Copy the file name found in the search to the hFileData structure
        strcpy(hFileData.Name, hFindData.name);
#elif defined(__DIRECT__FILE_SEARCH_POSIX)
// -----------------------
// POSIX Specific
// -----------------------
        // Copy the file name found with readdir() to the hFileData structure
        strcpy(hFileData.Name, hDirent->d_name);
#endif

        // Copy the file name onto the directory
        szTmpBuffer += hFileData.Name;

        // Call stat() to get information about the file.
        if (stat(szTmpBuffer.c_str(), &hFStat) == 0)
        {
                // Set the last access time
                hFileData.AccessTime = ( time_t )hFStat.st_atime;
                // Set the creation time
                hFileData.CreateTime = ( time_t )hFStat.st_ctime;
                // Set the last write time
                hFileData.WriteTime  = ( time_t )hFStat.st_mtime;
                // Set the size
                hFileData.Size       =           hFStat.st_size;
                // Set the mode
                hFileData.Mode       =           hFStat.st_mode;
        }
        else
        {
                // Stat failed, destroy all values except the file's name
                hFileData.AccessTime = ( time_t )0;
                hFileData.CreateTime = ( time_t )0;
                hFileData.WriteTime  = ( time_t )0;
                hFileData.Size       =           0L;
                hFileData.Mode       =           0;
        }
}

// GetInfo: Return's the current entry for a matching file (if any)
FileData CDirect::GetInfo(void)
{
        // Return the hFileData structure
        return (hFileData);
}

// Clears the values for the hFileData structure
void CDirect::DestroyValues(void)
{
        // Set all values to that equivalent to NULL
        hFileData.Mode       =           0;
        hFileData.AccessTime = ( time_t )0;
        hFileData.CreateTime = ( time_t )0;
        hFileData.WriteTime  = ( time_t )0;
        hFileData.Size       =           0L;
		memset(hFileData.Name, 0, sizeof(hFileData.Name));
}

// FixPath: Ensure there's a trailing slash on the directory name
void CDirect::FixPath(void)
{
        // Temporary buffer
        string  szTmpBuffer;

        // Temporary iterator
        string::iterator idx;

        // Copy the directory into the buffer
        szTmpBuffer = _directory.c_str();

        // Set the iterator to the end of the buffer
        idx = szTmpBuffer.end();

#if defined(__DIRECT__FILE_SEARCH_WINDOWS) || defined(__DIRECT__FILE_SEARCH_OS2)
// -----------------------
// Windows & OS/2 Specific
// -----------------------
        // Check to see if there's a trailing backslash
        if (*idx != '\\')
        {
                // No, add one.
                szTmpBuffer += '\\';
        }
#elif defined(__DIRECT__FILE_SEARCH_POSIX)
// -----------------------
// POSIX Specific
// -----------------------
        // Check for a trailing slash
        if (*idx != '/')
        {
                // No, add one.
                szTmpBuffer += '/';
        }
#endif

        // Save the changes
        _directory = szTmpBuffer;
}

void CDirect::GetMaxFiles(void)
{
#if defined(__DIRECT__FILE_SEARCH_WINDOWS) || defined(__DIRECT_FILE_SEARCH_OS2)
// ------------------------
// Windows & OS/2 Specific
// ------------------------
        // Create a search handle
        long                    hSearch = -1L;
        // Create a find data structure to hold information
        struct _finddata_t      hFData;
        // Temporary string to hold the full path & pattern
        string                  szTmp;

        // Combine the directory and pattern into a full path
        szTmp  = _directory.c_str();
        szTmp += _pattern.c_str();
        // Create a search handle and try to find the first instance of
        // a file matching the pattern, if none then MaxFiles will remain
        // zero
        if ((hSearch = _findfirst(szTmp.c_str(), &hFData)) > -1L)
        {
                // increment the max files value
                _MaxFiles++;
                // Search for other files after the first one until
                // there are no more files to search through.
                while (_findnext(hSearch, &hFData) == 0)
                {
                        // increment, file found.
                        _MaxFiles++;
                }
                // Close the search, finish.
                _findclose(hSearch);
        }
#elif defined(__DIRECT__FILE_SEARCH_POSIX)
// -------------------------
// POSIX Specific
// -------------------------
        // Create a DIRectory structure
        DIR             *hDirec;
        // Create a local instance to hold information obtained
        struct dirent   *hDirentLocal;
        // Attempt to open the specified directory
        if ((hDirec = opendir( _directory.c_str())) != NULL)
        {
                // Read each entry of the directory until a match was found
                // or the directory has no more entries.
                while ((hDirentLocal = readdir(hDirec)) != NULL)
                {
                        // Check to see if the file's name matches the given
                        // pattern.
#if defined(DJGPP)
                        if (fnmatch(_pattern.c_str(), hDirentLocal->d_name, FNM_NOESCAPE | FNM_NOCASE) == 0)
#else
                        if (fnmatch(_pattern.c_str(), hDirentLocal->d_name, 0) == 0)
#endif
                        {
                                // increment the file count.
                                _MaxFiles++;
                        }
                }
                // Terminate the search
                closedir(hDirec);
        }
#endif
}

