/* --------------------------------------------------------------------------
 *
 * $Workfile: LCmdSelect.cpp $
 * $Revision: 18 $
 *   $Author: Lel $
 *     $Date: 06.01.24 16:15 $
 *
 *   Project: Larsen Commander
 *   Created: June 10, 1996.
 * Copyright: Leif Erik Larsen. 1996. Ski, Norway.
 *            This is unpublished property, all rights reserved.
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdSelect.h"
#include "lcmd/LCmdFilePanel.h"

LCmdSelect::LCmdSelect ( LCmdFilePanel& fpanel )
           :fpanel(fpanel)
{
}

LCmdSelect::~LCmdSelect ()
{
}

int LCmdSelect::select ( int index )
{
   switch (index)
   {
      case NAVIGATE_UP:
         return selectIfUp();

      case NAVIGATE_DOWN:
         return selectIfDown();

      case NAVIGATE_LEFT:
         return selectIfLeft();

      case NAVIGATE_RIGHT:
         return selectIfRight();

      case NAVIGATE_PAGEUP:
         return selectIfPageUp();

      case NAVIGATE_PAGEDOWN:
         return selectIfPageDown();

      case NAVIGATE_HOME:
         return selectIfHome();

      case NAVIGATE_END:
         return selectIfEnd();

      default:
         return selectIfRandom(index);
   }
}

LCmdSelectBrief::LCmdSelectBrief ( LCmdFilePanel& fpanel )
                :LCmdSelect(fpanel)
{
}

int LCmdSelectBrief::selectIfDown ()
{
   int scrollWidth;
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex + 1;

   if (newIndex >= fpanel.items.getCount())
      return -1;

   fpanel.curSelectedItem = newIndex;

   // Unselect previously selected item
   fpanel.drawItem(prevIndex);

   // Calculate index of column to which newly selected item belongs
   int newColumn = newIndex / fpanel.itemsInListVer;
   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   GRectangle rScroll(fpanel.filesRect);

   // -----------------------------------------------------------------------
   // If requested item is in leftmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   if (newColumn == fpanel.leftMostColumn &&
       fpanel.leftMostColumnXPos < 0)
   {
      scrollWidth = -fpanel.leftMostColumnXPos; // Convert to absolute value
      fpanel.leftMostColumnXPos = 0;
   }

   // -----------------------------------------------------------------------
   // If requested item is already within visible area

   else
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // -----------------------------------------------------------------------
   // If requested item is in rightmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   else
   if (newIndex > lastVisibleNC)
   {
      int winWidth = rScroll.width;
      int xpos = fpanel.calcColumnXPos(newColumn);
      int columnWidth = fpanel.columns.get(newColumn).width;
      bool visible = xpos > winWidth ? false : true;

      scrollWidth = winWidth - xpos - columnWidth;

      int i = newColumn;
      for (xpos = winWidth - columnWidth;
           xpos > 0 && i >= 0;
           xpos -= fpanel.columns.get(--i).width);

      fpanel.leftMostColumnXPos = (columnWidth < winWidth) ? xpos : 0;
      fpanel.leftMostColumn = (i > 0) ? i : 0;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;

      if (!visible)
      {
         // Previous selected item was completely outside visible area, so
         // request whole file list window to be redrawn (just in case).
         fpanel.invalidateRect();
         return newIndex;
      }
   }

   // --------------------------------------------------------------------------
   // The requested column is completely outside visible area, so activate
   // it as when random selection

   else
   if (newColumn < fpanel.leftMostColumn)
   {
      return selectIfRandom(newIndex);
   }

   // --------------------------------------------------------------------------
   // If requested item is in the column next to the right of currently
   // rightmost visible column, then perform a normal right scrolling to
   // activate the next column to the right

   else
   {
      fpanel.leftMostColumn += 1;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;
      scrollWidth = fpanel.columns.get(fpanel.leftMostColumn).width;
   }

   // Perform the scrolling
   fpanel.scrollWindow(scrollWidth, 0, true, &rScroll);

   // Redraw the newly activated column
   fpanel.drawColumn(newColumn);

   return newIndex;
}

int LCmdSelectBrief::selectIfUp ()
{
   int i;
   int scrollWidth;
   int lastVisibleNC;
   int newColumn;
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex - 1;

   if (newIndex < 0)
      return -1;

   fpanel.curSelectedItem = newIndex;

   // Unselect previously active item
   fpanel.drawItem(prevIndex);

   // Calculate index of column to which newly selected item belongs
   newColumn = newIndex / fpanel.itemsInListVer;
   lastVisibleNC = fpanel.calcLastVisibleItemNC ();
   GRectangle rScroll(fpanel.filesRect);

   // -----------------------------------------------------------------------
   // If requested item is in leftmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   if (newColumn == fpanel.leftMostColumn &&
       fpanel.leftMostColumnXPos < 0)
   {
      scrollWidth = -fpanel.leftMostColumnXPos; // Convert to absolute value
      fpanel.leftMostColumnXPos = 0;
   }

   // -----------------------------------------------------------------------
   // If requested item is already within visible area

   else
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // -----------------------------------------------------------------------
   // If requested item is in rightmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   else
   if (newIndex > lastVisibleNC)
   {
      int winWidth = rScroll.width;
      int xpos = fpanel.calcColumnXPos(newColumn);
      int columnWidth = fpanel.columns.get(newColumn).width;

      scrollWidth = winWidth - xpos - columnWidth;

      for (xpos = winWidth - columnWidth, i = newColumn;
           xpos > 0 && i >= 0;
           xpos -= fpanel.columns.get(--i).width);

      fpanel.leftMostColumnXPos = xpos;
      fpanel.leftMostColumn = (i > 0) ? i : 0;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;
   }

   // --------------------------------------------------------------------------
   // If requested item is in the column next to the left of currently
   // leftmost visible column, then perform a normal left scrolling to activate
   // the next column to the left

   else
   if (newColumn == fpanel.leftMostColumn - 1)
   {
      fpanel.leftMostColumn -= 1;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;
      scrollWidth = fpanel.columns.get(fpanel.leftMostColumn).width;
   }

   // --------------------------------------------------------------------------
   // The requested column is completely outside visible area, so activate
   // it as when random selection

   else
   {
      return selectIfRandom(newIndex);
   }

   // Perform the scrolling
   fpanel.scrollWindow(scrollWidth, 0, true, &rScroll);

   // Redraw the newly activated column
   fpanel.drawColumn(newColumn);

   return newIndex;
}

int LCmdSelectBrief::selectIfEnd ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = fpanel.items.getCount() - 1;

   if (newIndex == prevIndex)
      return newIndex;

   fpanel.curSelectedItem = newIndex;

   // Unselect previously selected item
   fpanel.drawItem(prevIndex);

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= fpanel.calcLastVisibleItemNC())
   {
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------

   int i;
   int newColumn = newIndex / fpanel.itemsInListVer;
   int winWidth = fpanel.filesRect.width;

   // If the new column is to wide to show in window, then just activate and
   // repaint it before return
   if (fpanel.columns.get(newColumn).width >= winWidth)
   {
      fpanel.leftMostColumnXPos = 0;
      fpanel.leftMostColumn = newColumn;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;
      fpanel.invalidateRect();
      return newIndex;
   }

   // Find index and position of new leftmost visible column
   i = newColumn;
   fpanel.leftMostColumnXPos = winWidth;
   do
   {
      fpanel.leftMostColumnXPos -= fpanel.columns.get(i).width;
      if (fpanel.leftMostColumnXPos < 0)
         break;
   }
   while (--i >= 0);

   fpanel.leftMostColumn = (i >= 0) ? i : 0;
   fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;

   fpanel.drawItem(newIndex);
   fpanel.invalidateRect();

   return newIndex;
}

int LCmdSelectBrief::selectIfHome ()
{
   if (fpanel.curSelectedItem == 0)
      return 0;

   int prevIndex = fpanel.curSelectedItem;
   int iPrev1Visible = fpanel.firstVisibleItem;
   int iPrevLeftMostColumnXPos = fpanel.leftMostColumnXPos;

   fpanel.leftMostColumn = 0;
   fpanel.leftMostColumnXPos = 0;
   fpanel.curSelectedItem = 0;
   fpanel.firstVisibleItem = 0;

   if (iPrev1Visible == 0 &&
       iPrevLeftMostColumnXPos == 0)
   {
      // Very first item is already at upper left corner
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(0);
   }
   else
   {
      fpanel.invalidateRect();
   }

   return 0;
}

int LCmdSelectBrief::selectIfLeft ()
{
   int scrollWidth;
   int newColumn;

   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex - fpanel.itemsInListVer;
   if (newIndex < 0)
      newIndex = 0;

   if (newIndex == prevIndex)
      return newIndex;

   fpanel.curSelectedItem = newIndex;

   // Unselect previously selected item
   fpanel.drawItem(prevIndex);

   // -----------------------------------------------------------------------
   // If requested item is in current leftmost visible column, and that
   // column is currently clipped (part of it is outside visible screen
   // area)

   newColumn = newIndex / fpanel.itemsInListVer;

   if (newColumn == fpanel.leftMostColumn &&
       fpanel.leftMostColumnXPos < 0)
   {
      scrollWidth = -fpanel.leftMostColumnXPos; // Convert to absolute
      fpanel.leftMostColumnXPos = 0;
   }

   // -----------------------------------------------------------------------
   // If requested item is already within visible area

   else
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= fpanel.calcLastVisibleItemNC())
   {
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // Activate the left neightbor column of previos leftmost column

   else
   if (newColumn == fpanel.leftMostColumn - 1)
   {
      fpanel.leftMostColumn -= 1;
      fpanel.leftMostColumnXPos = 0;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;
      scrollWidth = fpanel.columns.get(fpanel.leftMostColumn).width;
   }

   // --------------------------------------------------------------------------
   // Previous item was completeley outside visible area, so activate
   // it as when random selection

   else
   {
      return selectIfRandom(newIndex);
   }

   // Perform the scrolling
   GRectangle rScroll(fpanel.filesRect);
   fpanel.scrollWindow(scrollWidth, 0, true, &rScroll);

   // Redraw the new leftmost column
   fpanel.drawColumn(fpanel.leftMostColumn);

   return newIndex;
}

int LCmdSelectBrief::selectIfRight ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex + fpanel.itemsInListVer;
   if (newIndex >= fpanel.items.getCount())
      newIndex = fpanel.items.getCount() - 1;

   if (newIndex == prevIndex)
      return newIndex;

   fpanel.curSelectedItem = newIndex;

   // Unselect previously selected item
   fpanel.drawItem(prevIndex);

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= fpanel.calcLastVisibleItemNC())
   {
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // Activate the right neightbor column of previos rightmost column

   GRectangle rScroll(fpanel.filesRect);
   int winWidth = rScroll.width;

   // Calculate index of column to which newly selected item belongs
   int newColumn = newIndex / fpanel.itemsInListVer;

   // Calculate width of area to scroll
   int i;
   int scrWidth = fpanel.leftMostColumnXPos;
   int nrOfColumns = fpanel.columns.getCount();
   for (i = fpanel.leftMostColumn;
        i < nrOfColumns;
        i++)
   {
      scrWidth += fpanel.columns.get(i).width;
      if (scrWidth > winWidth)
      {
         if (i == fpanel.leftMostColumn)
         {
            // Previous active column was to wide to show in window, so just
            // force the new column to be visible and repaint before return
            int winWidth = rScroll.width;
            int columnWidth = fpanel.columns.get(newColumn).width;

            int xpos;
            for (xpos = winWidth - columnWidth, i = newColumn;
                 xpos > 0 && i >= 0;
                 xpos -= fpanel.columns.get(--i).width);

            fpanel.leftMostColumnXPos = (columnWidth < winWidth) ? xpos : 0;
            fpanel.leftMostColumn = (i > 0) ? i : 0;
            fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;

            fpanel.invalidateRect();
            return newIndex;
         }

         scrWidth -= winWidth;
         break;
      }
   }

   // If the new column is to wide to show in window, then just activate and
   // repaint it before return
   if (fpanel.columns.get(newColumn).width >= winWidth ||
       fpanel.calcColumnXPos(newColumn) > fpanel.calcColumnXPos(i) ||
       newColumn < fpanel.leftMostColumn)
   {
      fpanel.leftMostColumnXPos = 0;
      fpanel.leftMostColumn = newColumn;
      fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;
      fpanel.invalidateRect();
      return newIndex;
   }

   // Perform the scrolling
   fpanel.scrollWindow(-scrWidth, 0, true, &rScroll);

   // Find index and position of new leftmost visible column
   fpanel.leftMostColumnXPos = winWidth;
   do
   {
      fpanel.leftMostColumnXPos -= fpanel.columns.get(i).width;
      if (fpanel.leftMostColumnXPos < 0)
      {
         fpanel.leftMostColumn = i;
         break;
      }
   }
   while (--i >= 0);

   fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;

   // Redraw the newly activated column
   fpanel.drawColumn(newColumn);

   return newIndex;
}

int LCmdSelectBrief::selectIfPageDown ()
{
   return -1; // ___?___
}

int LCmdSelectBrief::selectIfPageUp ()
{
   return -1; // ___?___
}

int LCmdSelectBrief::selectIfRandom  ( int index )
{
   if (index < 0)
      index = 0;

   int count = fpanel.items.getCount();
   if (count <= 0)
      return -1;

   if (index >= count)
      index = count - 1;

   int prevIndex = fpanel.curSelectedItem;
   int lastVisibleNC = fpanel.calcLastVisibleItemNC();

   // Test if the requested item is completely visible, without being
   // clipped at all.
   bool visible = fpanel.leftMostColumnXPos == 0 && // Is clipped if != 0
                  index >= fpanel.firstVisibleItem &&
                  index <= lastVisibleNC;

   if (visible && index == prevIndex)
   {
      // Redraw the item, in case marking of it has been toggled on/off
      fpanel.drawItem(index);
      return index;
   }

   fpanel.curSelectedItem = index;

   // Unselect previously selected item
   fpanel.drawItem(prevIndex);

   // --------------------------------------------------------------------------
   // If requested item is already within none-clipped visible area

   if (visible)
   {
      fpanel.drawItem(index);
      return index;
   }

   // --------------------------------------------------------------------------
   // Make requested item visible

   int winWidth = fpanel.filesRect.width;
   int newColumn = index / fpanel.itemsInListVer;

   int i = newColumn;
   for (int xpos = winWidth - fpanel.columns.get(i).width;
        i > 0 && xpos - fpanel.columns.get(i-1).width > 0;
        xpos -= fpanel.columns.get(--i).width);

   fpanel.leftMostColumn = i;
   fpanel.leftMostColumnXPos = 0;
   fpanel.firstVisibleItem = fpanel.leftMostColumn * fpanel.itemsInListVer;

   // Repaint whole list of filenames in fpanel
   fpanel.invalidateRect();

   return index;
}

LCmdSelectWide::LCmdSelectWide ( LCmdFilePanel& fpanel )
                :LCmdSelect(fpanel)
{
}

int LCmdSelectWide::selectIfDown ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex + 1;
   if (newIndex >= fpanel.items.getCount())
      return -1;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.curSelectedItem = newIndex;
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If requested item is the item just below current downmost visible item

   if (newIndex == fpanel.firstVisibleItem + fpanel.itemsInListVer)
   {
      fpanel.curSelectedItem = newIndex;
      GRectangle rScroll(fpanel.filesRect);
      rScroll.y = (rScroll.y + rScroll.height - 1) - (fpanel.itemHeight * (fpanel.itemsInListVer - 1));
      fpanel.scrollWindow(0, fpanel.itemHeight, true, &rScroll);
      fpanel.firstVisibleItem += 1;

      if (newIndex - 1 >= fpanel.firstVisibleItem &&
          newIndex - 1 <= fpanel.firstVisibleItem + fpanel.itemsInListVer - 1)
      {
         fpanel.drawItem(newIndex - 1);
      }

      fpanel.drawItem(newIndex);

      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return fpanel.selectItem(newIndex);
}

int LCmdSelectWide::selectIfUp ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex - 1;
   if (newIndex < 0)
      return -1;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.curSelectedItem = newIndex;
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If requested item is the item just above current uppermost visible item

   if (newIndex == fpanel.firstVisibleItem - 1)
   {
      fpanel.curSelectedItem = newIndex;
      GRectangle rScroll(fpanel.filesRect);
      rScroll.y = (rScroll.y + rScroll.height - 1) - (fpanel.itemHeight * fpanel.itemsInListVer);
      rScroll.height -= fpanel.itemHeight + 1;
      fpanel.scrollWindow(0, -fpanel.itemHeight, true, &rScroll);
      fpanel.firstVisibleItem -= 1;

      if (newIndex + 1 >= fpanel.firstVisibleItem &&
          newIndex + 1 <= fpanel.firstVisibleItem + fpanel.itemsInListVer - 1)
      {
         fpanel.drawItem(newIndex + 1);
      }

      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return fpanel.selectItem(newIndex);
}

int LCmdSelectWide::selectIfEnd ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = fpanel.items.getCount() - 1;
   if (newIndex < 0)
      return -1;

   fpanel.curSelectedItem = newIndex;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------

   fpanel.firstVisibleItem = fpanel.items.getCount() - fpanel.itemsInListVer;
   if (fpanel.firstVisibleItem < 0)
      fpanel.firstVisibleItem = 0;

   fpanel.invalidateRect();

   return newIndex;
}

int LCmdSelectWide::selectIfHome ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = 0;
   if (newIndex >= fpanel.items.getCount())
      return -1;

   fpanel.curSelectedItem = newIndex;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------

   fpanel.firstVisibleItem = 0;

   fpanel.invalidateRect();

   return newIndex;
}

int LCmdSelectWide::selectIfLeft ()
{
   return -1; // ___?___
}

int LCmdSelectWide::selectIfRight ()
{
   return -1; // ___?___
}

int LCmdSelectWide::selectIfPageDown ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex + fpanel.itemsInListVer;
   if (newIndex >= fpanel.items.getCount())
      newIndex = fpanel.items.getCount() - 1;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.curSelectedItem = newIndex;
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If previously select item was vithin visible area

   if (prevIndex >= fpanel.firstVisibleItem &&
       prevIndex <= lastVisibleNC)
   {
      fpanel.curSelectedItem = newIndex;

      fpanel.firstVisibleItem += fpanel.itemsInListVer;

      if (fpanel.firstVisibleItem > fpanel.items.getCount() - fpanel.itemsInListVer)
         fpanel.firstVisibleItem = fpanel.items.getCount() - fpanel.itemsInListVer;

      if (fpanel.firstVisibleItem < 0)
         fpanel.firstVisibleItem = 0;

      fpanel.invalidateRect();
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return fpanel.selectItem(newIndex);
}

int LCmdSelectWide::selectIfPageUp ()
{
   int prevIndex = fpanel.curSelectedItem;
   int newIndex = prevIndex - fpanel.itemsInListVer;
   if (newIndex < 0)
      newIndex = 0;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC ();
   if (newIndex >= fpanel.firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      fpanel.curSelectedItem = newIndex;
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If previously select item was vithin visible area

   if (prevIndex >= fpanel.firstVisibleItem &&
       prevIndex <= lastVisibleNC)
   {
      fpanel.curSelectedItem = newIndex;

      fpanel.firstVisibleItem -= fpanel.itemsInListVer;
      if (fpanel.firstVisibleItem < 0)
         fpanel.firstVisibleItem = 0;

      fpanel.invalidateRect();
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return fpanel.selectItem(newIndex);
}

int LCmdSelectWide::selectIfRandom  ( int index )
{
   int prevIndex = fpanel.curSelectedItem;
   if (index < 0 || index >= fpanel.items.getCount())
   {
      return -1;
   }

   fpanel.curSelectedItem = index;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = fpanel.calcLastVisibleItemNC();
   if (index >= fpanel.firstVisibleItem && index <= lastVisibleNC)
   {
      fpanel.drawItem(prevIndex);
      fpanel.drawItem(index);
      return index;
   }

   // --------------------------------------------------------------------------
   // Place newly selected item in the center of the fpanel

   fpanel.firstVisibleItem = index - (fpanel.itemsInListVer / 2);

   if (fpanel.items.getCount() - index < fpanel.itemsInListVer / 2)
      fpanel.firstVisibleItem = fpanel.items.getCount() - fpanel.itemsInListVer;

   if (fpanel.firstVisibleItem < 0)
      fpanel.firstVisibleItem = 0;

   fpanel.invalidateRect();

   return index;
}

LCmdSelectFull::LCmdSelectFull ( LCmdFilePanel& fpanel )
               :LCmdSelectWide(fpanel)
{
}

int LCmdSelectFull::selectIfRight ()
{
   if (fpanel.columns.get(0).width > fpanel.filesRect.width)
      fpanel.onHScrollLineDown();
   return fpanel.curSelectedItem;
}

int LCmdSelectFull::selectIfLeft ()
{
   if (fpanel.columns.get(0).width > fpanel.filesRect.width)
      fpanel.onHScrollLineUp();
   return fpanel.curSelectedItem;
}

LCmdSelectInfo::LCmdSelectInfo ( LCmdFilePanel& fpanel )
               :LCmdSelect(fpanel)
{
}

int LCmdSelectInfo::selectIfDown ()
{
   fpanel.onVScrollLineDown();
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfUp ()
{
   fpanel.onVScrollLineUp();
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfEnd ()
{
   int sumh = fpanel.info.sumHeightOfItems();
   GDimension dim = fpanel.getWindowSize();
   int pos = sumh - dim.height;
   if (pos > 0)
      fpanel.onVScrollSliderTrack(pos);
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfHome ()
{
   fpanel.onVScrollSliderTrack(0);
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfLeft ()
{
   fpanel.onHScrollLineUp();
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfRight ()
{
   fpanel.onHScrollLineDown();
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfPageDown ()
{
   fpanel.onVScrollPageDown();
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfPageUp ()
{
   fpanel.onVScrollPageUp();
   return fpanel.curSelectedItem;
}

int LCmdSelectInfo::selectIfRandom  ( int /*index*/ )
{
   return fpanel.curSelectedItem;
}

