/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdProcess.h"
#include "lcmd/LCmd.h"
#include "lcmd/LCmdConsoleMonitor.h"
#include "lcmd/LCmdCmdLine.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/gui/event/GUserMessage.h"
#include "glib/util/GSearchPath.h"
#include "glib/sys/GSystem.h"

LCmdProcessLauncher::LCmdProcessLauncher ( LCmdCmdLineEntry& owner, 
                                           const GString& origCommand, 
                                           const GString& workingDir, 
                                           const GString& prgName, 
                                           const GString& paramStr, 
                                           bool forceNewSession, 
                                           bool isInternalCmd, 
                                           bool runAsObject, 
                                           bool closeOnExit, 
                                           bool forceRunViaShell )
                    :GProcessLauncher(workingDir, 
                                      prgName, 
                                      paramStr, 
                                      forceNewSession, 
                                      closeOnExit, 
                                      lcmd->options.conVarious.startConFullScreen, 
                                      lcmd->options.conVarious.startDosFullScreen),
                     owner(owner),
                     finished(false),
                     isInternalCmd(isInternalCmd),
                     runAsObject(runAsObject),
                     forceCmd(forceRunViaShell),
                     printPIDInfo(false),
                     origCommand(origCommand),
                     detachRequested(false)
{
}

LCmdProcessLauncher::~LCmdProcessLauncher ()
{
}

void LCmdProcessLauncher::detachChildProg ()
{
   detachRequested = true;
}

void LCmdProcessLauncher::waitForTheChildProcessToFinish ()
{
   // Reduce the priority of this thread somewhat, to prevent the 
   // GUI-thread from being CPU-hogged.
   setPriority(GThread::PRIORITY_NORMAL_MINUS);

   // Read pipe data and append to the console monitor.
   const int maxText = 1024;
   GString text(maxText);
   bool anyData = true;
   for (;;)
   {
      bool stillRunning = isRunningChild();
      if (detachRequested)
         break;

      if (!(anyData || stillRunning))
      {
         if (childStdOut->peekBytesAvailable() <= 0 &&
             childStdErr->peekBytesAvailable() <= 0)
         {
            break;
         }
         anyData = true;
      }

      if (!anyData)
         GThread::Sleep(50);

      anyData = false; // Until the opposite has been proven.

      // Read text from childs STDOUT and append it to the console monitor.
      childStdOut->readUnblocked(text, maxText);
      if (text.length() > 0)
      {
         anyData = true;
         lcmd->conmon.appendTextFromSecondaryThread(*this, text);
      }

      // Read text from childs STDERR and append it to the console monitor.
      childStdErr->readUnblocked(text, maxText);
      if (text.length() > 0)
      {
         anyData = true;
         lcmd->conmon.appendTextFromSecondaryThread(*this, text);
      }

      // Check if the child process waits for some data on its STDIN. 
      // If it does then post a "UmChildWaitsForStdInData" user message 
      // to the command line window. Then the command line will change its 
      // background color (to yellow by default) so that the user will see 
      // that the next time he or she press enter the text in the command 
      // line will actually be written to the waiting child process's STDIN.
      if (stillRunning && !anyData && isChildWaitingForDataOnItsStdIn())
      {
         GString txt = getNextChildStdInText();
         if (txt == "")
         {
            // Request the GUI to ask the user for some input to give
            // to the waiting child process.
            GUserMessage um("UmChildWaitsForStdInData", this);
            sendGuiUserMessage(owner, um, false);
            // Wait for at least some data to give to the waiting child.
            waitForSomeDataOnStdIn();
            txt = getNextChildStdInText();
         }
         // Copy to the console what the user has "typed".
         lcmd->conmon.appendTextFromSecondaryThread(*this, txt);
         // Then, write it to the pipe so that the child can read it.
         childStdIn->print(txt);
         childStdIn->flush();
         // Make sure that we loop at least once more, even if the 
         // child process actually has exited by now. This is to read 
         // any data written by the program just before it exited.
         anyData = true; 
      }
   }
}

void LCmdProcessLauncher::printHelp ()
{
   // Print the name of each command that will be transferred to the shell.
   GString comSpec = lcmd->getEnvironmentVar(COMSPEC, DEFCOMSPEC);
   GStringl header1("%Txt_HlpShellCommands", GVArgs(comSpec));
   GString shellTxt(512);
   shellTxt += '\n';
   shellTxt += header1;
   shellTxt += '\n';
   shellTxt.append('~', header1.length());
   shellTxt += '\n';
   int header1Len = header1.length();
   int num = owner.internalCommands.getCount();
   for (int i=0, xpos=0; i<num; i++)
   {
      InternalCommand& cmd = owner.internalCommands.getIndexedItem(i);
      if (cmd.isTransferToShell() && !cmd.isDisabled())
      {
         const GString& str = cmd.getCommandStr();
         if ((xpos += str.length() + 2) >= (header1Len - 2))
         {
            xpos = 0;
            shellTxt += '\n';
         }
         shellTxt += str;
         shellTxt += ", ";
      }
   }

   // Print some brief info about each of the currently enabled
   // internal commands.
   GStringl header2("%Txt_HlpInternalCommands");
   GString internalTxt(512);
   internalTxt += "\n\n";
   internalTxt += header2;
   internalTxt += '\n';
   internalTxt.append('~', header2.length());
   internalTxt += '\n';
   internalTxt += GStringl("%TxtInternalCommandHelp_X");
   internalTxt += '\n';
   for (int i=0; i<num; i++)
   {
      InternalCommand& cmd = owner.internalCommands.getIndexedItem(i);
      if (!cmd.isTransferToShell() && !cmd.isDisabled())
      {
         internalTxt += cmd.getShortHelp();
         internalTxt += '\n';
      }
   }

   // ---
   lcmd->conmon.appendTextFromSecondaryThread(*this, shellTxt);
   lcmd->conmon.appendTextFromSecondaryThread(*this, internalTxt);
}

const GString& LCmdProcessLauncher::getOriginalCommand () const 
{ 
   return origCommand; 
}

bool LCmdProcessLauncher::hasFinished () const 
{ 
   return finished; 
}

void LCmdProcessLauncher::internalCmd_Help ()
{
   printHelp();
}

void LCmdProcessLauncher::internalCmd_HelpGeneral ()
{
   GStringl txt("%TxtCmdLineHelp1");
   lcmd->conmon.appendTextFromSecondaryThread(*this, txt);
}

void LCmdProcessLauncher::internalCmd_Hist ()
{
   CommandHistory& h = lcmd->cmdLine.entry.cmdHist; // Make this a fast one

   int num = 10;
   if (paramStr != "")
   {
      try {
         num = GInteger::ParseInt(paramStr);
      } catch (GNumberFormatException& /*e*/) {
         num = 10;
      }
   }
   if (num < 0)
      num = 10;
   if (num > h.getStringCount())
      num = h.getStringCount();

   GString txtOut(512);
   for (int i=num, idx=h.getStringCount()-num; i>0; i--, idx++)
      txtOut += GString("%03d: %s\n", GVArgs(i).add(h.getIndexedString(idx)));
   lcmd->conmon.appendTextFromSecondaryThread(*this, txtOut);
}

void LCmdProcessLauncher::internalCmd_PopD ()
{
   GString txtOut(512);
   GArray<GString>& s = owner.dirStack;
   for (int i=0, num=s.getCount(); i<num; i++)
      txtOut += GString("%03d: %s\n", GVArgs(num-i).add(s.get(i)));
   lcmd->conmon.appendTextFromSecondaryThread(*this, txtOut);
}

void LCmdProcessLauncher::internalCmd_Alias ()
{
   GString txtOut(512);
   LCmdAliases& aliases = owner.aliases; // Make this a fast one
   for (int i=0, num=aliases.getCount(); i<num; i++)
   {
      const GString& name = aliases.getAliasName(i);
      const GString& val = aliases.getAliasStr(i);
      txtOut += GString("%03d: %s=%s\n", GVArgs(i+1).add(name).add(val));
   }
   lcmd->conmon.appendTextFromSecondaryThread(*this, txtOut);
}

void LCmdProcessLauncher::internalCmd_Info ()
{
   bool recentHadLF = true;
   LCmdFilePanel& panel = lcmd->curPanel->getOppositePanel();
   GArray<LCmdInfoPanel::AbstractItem>& items = panel.info.items;
   lcmd->sysInfo.update();
   GString txtOut(512);
   int num = items.getCount();
   for (int i=0, infoIdx=1; i<num; i++)
   {
      const GString& infoStr = items.get(i).getItemString();
      if (infoStr != "")
      {
         if (recentHadLF)
            txtOut += GString("%02d: %s", GVArgs(infoIdx).add(infoStr));
         else
            txtOut += infoStr;
         recentHadLF = (infoStr.lastChar() == '\n');
         if (recentHadLF)
            infoIdx++;
      }
   }
   lcmd->conmon.appendTextFromSecondaryThread(*this, txtOut);
}

void LCmdProcessLauncher::internalCmd_Which ()
{
   GEnvironment& env = lcmd->getEnvironmentVars();
   GString fullPath = env.which(paramStr);
   GString str;
   if (fullPath == "")
      str = GStringl("%Txt_Cmd_WHICH_CantFindX", GVArgs(paramStr));
   else
      str = fullPath;
   lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n", GVArgs(str));
}

void LCmdProcessLauncher::internalCmd_FindDup ()
{
   GVfsLocal vfs; // Work on the local file system.

   // Get the name of which search path environment to search through
   int idx = paramStr.indexOf(' ');
   GString envName = "";
   if (idx > 0)
   {
      envName = GString(paramStr, 0, idx);
   }
   else
   {
      idx = paramStr.length();
      envName =  paramStr;
   }

   envName.trim();

   if (envName == "")
      envName = "PATH"; // This is the default PATH-name

   GString path = lcmd->getEnvironmentVar(envName);
   if (path == "")
   {
      GStringl msg("%Txt_Cmd_FINDDUP_UndefinedEnvVarX", GVArgs(envName));
      lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n\n", GVArgs(msg));
   }
   else
   {
      GString fileName = GString(paramStr, idx); // The remaining part of paramStr
      fileName.trim();
      if (fileName == "")
      {
         // Since the filename was not specified, use the default
         if (envName.equalsIgnoreCase("PATH"))
            fileName = "*.exe";
         else
         if (envName.equalsIgnoreCase("LIBPATH"))
            fileName = "*.dll";
         else
         if (envName.equalsIgnoreCase("HELP"))
            fileName = "*.hlp";
         else
         if (envName.equalsIgnoreCase("BOOKSHELF"))
            fileName = "*.inf";
         else
         if (envName.equalsIgnoreCase("LOCPATH"))
            fileName = "*.dll";
         else
         if (envName.equalsIgnoreCase("INCLUDE"))
            fileName = "*.h*";
         else
         if (envName.equalsIgnoreCase("DPATH"))
            fileName = "*.*";
         else
            fileName = "*.exe";
      }

      GStringl msg("%Txt_Cmd_FINDDUP_LookingForXInY", GVArgs(fileName).add(envName));
      lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n\n", GVArgs(msg));

      // Find all duplicate filenames and print them to the console monitor.
      GSearchPath schPath(path);
      GSectionBag sbag;

      // Loop through all directories in the search path and find all files
      // that match the specified filename.
      int numDirs = schPath.getCount();
      for (int i=0; i<numDirs; i++)
      {
         const GString& nextDir = schPath.getIndexedDir(i);
         const GString& slash = nextDir.endsWith(GFile::SlashChar) ? GString::Empty : GFile::SlashStr;
         GString fullDir("%s%s%s", GVArgs(nextDir).add(slash).add(fileName));
         GStringl msg1("%Txt_Cmd_FINDDUP_ScanningForX", GVArgs(fullDir));
         lcmd->conmon.appendTextFromSecondaryThread(*this, "%s ... ", GVArgs(msg1));

         int count = 0;
         GVfs::List list;
         vfs.fillList(list, fullDir, true, false, true, true);
         for (int num=list.size(); count<num; count++)
         {
            const GVfs::List::Item& item = list[count];
            const GString& fname = item.getFName();
            sbag.putString(fname, nextDir, "");
         }

         GStringl msg2("%Txt_Cmd_FINDDUP_FoundNFiles", GVArgs(count));
         lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n", GVArgs(msg2));
      }

      // Print all duplicate filenames
      int num = sbag.getNumberOfSections();
      for (int idx=0; idx<num; idx++)
      {
         GKeyBag<GString>& bag = sbag.getIndexedSectionBag(idx);
         int numDup = bag.getCount();
         if (numDup > 1)
         {
            GString fileName = sbag.getIndexedSectionName(idx);
            GStringl msg("%Txt_Cmd_FINDDUP_XExistInDirs", GVArgs(fileName));
            lcmd->conmon.appendTextFromSecondaryThread(*this, "\n%s\n", GVArgs(msg));
            for (int i=0; i<numDup; i++)
            {
               const GString& dir = bag.getKey(i);
               lcmd->conmon.appendTextFromSecondaryThread(*this, "\t%s\n", GVArgs(dir));
            }
         }
      }
   }
}

void LCmdProcessLauncher::runAsInternalCommand ()
{
   if (prgName.equalsIgnoreCase("help"))
      internalCmd_HelpGeneral();
   else
   if (prgName.equalsIgnoreCase("?"))
      internalCmd_Help();
   else
   if (prgName.equalsIgnoreCase("h") || prgName.equalsIgnoreCase("hist"))
      internalCmd_Hist();
   else
   if (prgName.equalsIgnoreCase("popd"))
      internalCmd_PopD();
   else
   if (prgName.equalsIgnoreCase("alias"))
      internalCmd_Alias();
   else
   if (prgName.equalsIgnoreCase("info"))
      internalCmd_Info();
   else
   if (prgName.equalsIgnoreCase("which"))
      internalCmd_Which();
   else
   if (prgName.equalsIgnoreCase("finddup"))
      internalCmd_FindDup();
   else
   {
      GStringl msg("%Txt_Cmd_Err_UnknownCmdX", GVArgs(prgName));
      lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n", GVArgs(msg));
   }
}

GString LCmdProcessLauncher::getWorkingDir ( const GString& defaultDir ) const
{
   GString homeDir = getProgramName(); // Full path of program that is about to be launched
   GFile::CutFileName(homeDir);
   if (GFile::IsSlash(homeDir.lastChar()))
      homeDir.removeLastChar();
   LCmdAliases& al = owner.aliases;
   GProgram& prg = GProgram::GetProgram();
   GEnvironment& env = prg.getEnvironmentVars();
   GString ret = al.processString(*lcmd->curPanel, defaultDir, &env, homeDir);
   // Remove enclosing quotes, if any. Caller needs a "clean" directory name.
   if (ret.length() > 2 && ret.beginsWith("\"") && ret.endsWith("\""))
   {
      ret.removeFirstChar();
      ret.removeLastChar();
   }
   return ret;
}

void LCmdProcessLauncher::programPathHasBeenValidated ( const GString& prgName, const GString& paramStr )
{
   if (!forceCmd)
      if (owner.isNoneConsoleCommand(prgName, paramStr))
         forceNewSession = true;
}

void LCmdProcessLauncher::processHasBeenLaunched ( bool asChild )
{
   if (asChild)
   {
      lcmd->mainWin.setStatusbarText(GString::Empty);

      // Print the command that is actually executed, so that the
      // user can know what is going on.
      if (printPIDInfo)
      {
         int pid = getPidOfRunningChildProcess();
         GString prgName = getProgramName();
         GString paramStr = getParamString();
         GStringl msg("%Txt_Cmd_ChildProcessLaunchedOK", GVArgs(pid).add(prgName).add(paramStr));
         lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n", GVArgs(msg));
      }

      // CMD.EXE (and possibly other programs as well) automatically changes
      // the WPS icon of its parent program to equal the default icon of the
      // child program when it launches a child program. So change our icon
      // back to the Larsen Commander icon. This must be done asynchronosuly,
      // however, because its must be done by the GUI thread. We are not
      // the GUI thread.
      GUserMessage um("UM_CHILD_PROCESS_HAS_LAUNCHED", this);
      sendGuiUserMessage(owner, um);
   }
   else
   {
      // The process was launched as a new session.
      // Print the command that is actually executed, so that the
      // user can know what is going on. Since the process was launched as
      // a new session, the PID is not valid. So don't print it.
      GString pn;
      if (prgName.contains(' '))
         pn = GString("\"%s\"", GVArgs(prgName));
      else
         pn = prgName;
      GStringl msg("%Txt_Cmd_SessionLaunchedOK", GVArgs(pn).add(paramStr));
      lcmd->conmon.appendTextFromSecondaryThread(*this, "%s\n", GVArgs(msg));
   }
}

void LCmdProcessLauncher::processHasFinished ( bool asChild, 
                                               int result, 
                                               bool normalExit )
{
   GString statusTxt;
   if (printPIDInfo && asChild && !detachRequested)
   {
      GString textToPrint;
      if (normalExit)
      {
         GString cmd = getOriginalCommand();
         int pid = getPidOfRunningChildProcess();
         statusTxt = GStringl("%Txt_Cmd_ChildProcessFinished", GVArgs(pid).add(result).add(cmd));
         int num = owner.getRunningChildrenCount();
         if (num >= 2)
            textToPrint = statusTxt;
      }
      textToPrint += "\n";
      lcmd->conmon.appendTextFromSecondaryThread(*this, textToPrint);
   }

   // Show the exit code of the process into the main cell of statusbar.
   if (asChild)
      lcmd->mainWin.setStatusbarText(statusTxt);
}

void LCmdProcessLauncher::runAsExternalCommand ()
{
   bool preventReread = false; // True if we should not reread file panels when child process has finsihed

   printPIDInfo = true;

   // Is it any other special command that has to be executed by COMSPEC?
   InternalCommand *ic = owner.internalCommands.get(prgName);
   if (ic != null && !ic->isDisabled())
   {
      GString tmp = prgName;
      tmp.toLowerCase();
      if (tmp == "call" ||
          tmp == "chcp" ||
          tmp == "copy" ||
          tmp == "date" ||
          tmp == "del" ||
          tmp == "detach" ||
          tmp == "dir" ||
          tmp == "echo" ||
          tmp == "erase" ||
          tmp == "for" ||
          tmp == "goto" ||
          tmp == "help" ||
          tmp == "if" ||
          tmp == "md" ||
          tmp == "mkdir" ||
          tmp == "move" ||
          tmp == "path" ||
          tmp == "pause" ||
          tmp == "rem"  ||
          tmp == "ren" ||
          tmp == "rename" ||
          tmp == "rd" ||
          tmp == "rmdir" ||
          tmp == "run" ||
          tmp == "set" ||
          tmp == "setlocal" ||
          tmp == "start" ||
          tmp == "time" ||
          tmp == "type" ||
          tmp == "ver" ||
          tmp == "vol")
      {
         forceCmd = true;
         printPIDInfo = false;
         if (tmp == "start")
         {
            forceNewSession = true;
         }
         else
         if (tmp == "dir" ||
             tmp == "set" ||
             tmp == "ver" ||
             tmp == "vol" ||
             tmp == "path" ||
             tmp == "pause" ||
             tmp == "type" ||
             tmp == "start" ||
             tmp == "echo")
         {
            preventReread = true;
         }
      }
   }

   if (!forceNewSession)
      forceNewSession = owner.isNoneConsoleCommand(prgName, paramStr, forceCmd);

   if (forceCmd)
   {
      GString pstr = GProcessLauncher::SysShellPrefixArgs + prgName;
      if (paramStr != "")
         pstr += GString::Blank + paramStr;
      paramStr = pstr;
      GProgram& prg = GProgram::GetProgram();
      GString temp = prg.getEnvironmentVar(COMSPEC, DEFCOMSPEC);
      GEnvironment& env = prg.getEnvironmentVars();
      prgName = env.which(temp);
   }

   // ---
   GProcessLauncher::run();

   // ---
   GString msg; // The locale specific error message.
   switch (getErrorCode())
   {
      case EC_NoError: 
         msg = ""; 
         if (startedAsChild) // If command was run (by our super class) as a child process.
         {
            // Signal that the child process has fisnished.
            GUserMessage um("UM_CHILD_PROCESS_HAS_FINISHED", GBoolean::GetBooleanStr(preventReread));
            sendGuiUserMessage(owner, um);
         }
         break;

      case EC_ErrActDirX: 
         // "Error activating directory: '%s'\n"
         msg = GStringl("%Txt_Cmd_Err_ActDir", GVArgs(workingDir)); 
         break;
      
      case EC_ErrNotAProgX:
         // "Not a program file: '%s'\n"
         msg = GStringl("%Txt_Cmd_Err_NotAProg", GVArgs(prgName)); 
         break;
      
      case EC_ErrPrgNotFoundX:
         // "Program file not found: '%s'\n"
         msg = GStringl("%Txt_Cmd_Err_ProgNotFound", GVArgs(prgName)); 
         break;

      case EC_ErrDllXReqByPrgY:
         // "Error loading module '%s' required by program '%s'.\n"
         msg = GStringl("%Txt_Cmd_Err_MissingModule", GVArgs(nameOfMissingDll).add(prgName)); 
         break;

      case EC_ErrLauchCmdXErrCodeY:
      default:
         // "Could not launch command '%s'. Error code: %d\n"
         msg = GStringl("%Txt_Cmd_Err_LaunchCmd", GVArgs(prgName).add(systemErrorCode)); 
         break;
   }

   if (msg != "")
      lcmd->conmon.appendTextFromSecondaryThread(*this, msg);
}

void LCmdProcessLauncher::run ()
{
   owner.runningCounter++;

   bool um_xxxstarted_wasSent = true;
   if (runAsObject || forceNewSession)
   {
      um_xxxstarted_wasSent = false;
   }
   else
   {
      GUserMessage um("UM_PROCESSLAUNCHERTHREADHASSTARTED", this);
      sendGuiUserMessage(owner, um);
   }

   if (runAsObject)
   {
      // The program a to be launched by the system, as a system dependent 
      // shell object such as e.g. an OS/2 WPS Object. This is to be done 
      // by the GUI-thread, however. In order to get the object window 
      // to occur on top Z-order. At least on OS/2.
      GUserMessage um("UmRunAsShellObject", prgName, this);
      sendGuiUserMessage(owner, um);
   }
   else
   if (isInternalCmd)
   {
      // The program is some internal Larsen Commander command,
      // such as e.g. FINDDUP or WHICH.
      runAsInternalCommand();
   }
   else
   {
      // The program is to be launched as an external application,
      // probably in the form of some executable file on the file system.
      // It doesn't necessarily has to be a child process, but if it is
      // then this method will not return until the child process has 
      // finished and exited.
      runAsExternalCommand();
   }

   owner.runningCounter--;

   if (um_xxxstarted_wasSent)
   {
      GUserMessage um("UM_PROCESSLAUNCHERTHREADHASFINISHED", this);
      sendGuiUserMessage(owner, um);
   }

   finished = true;
}
