/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdInternalViewer.h"
#include "lcmd/LCmdDlgGotoLine.h"
#include "lcmd/LCmd.h"

#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GStatusbar.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/io/GFileInputStream.h"

// Since most file systems on Windows and OS/2 are not case significant, 
// don't use case significant keys for this keybag.
GKeyBag<LCmdInternalViewer::FileOptions> LCmdInternalViewer::FileStore(16, -3, true);

DEFINE_COMMAND_TABLE(LCmdInternalViewer);
   ADD_COMMAND("cmdFileOpen",               cmdFileOpen);
   ADD_COMMAND("cmdExit",                   cmdExit);
   ADD_COMMAND("cmdCopy",                   cmdCopy);
   ADD_COMMAND("cmdSearch",                 cmdSearch);
   ADD_COMMAND("cmdSearchNext",             cmdSearchNext);
   ADD_COMMAND("cmdGotoLineNum",            cmdGotoLineNum);
   ADD_COMMAND("cmdSelectAll",              cmdSelectAll);
   ADD_COMMAND("cmdUnselect",               cmdUnselect);
   ADD_COMMAND("cmdShowContextMenu",        cmdShowContextMenu);
   ADD_COMMAND("cmdToggleToolbar",          cmdToggleToolbar);
   ADD_COMMAND("cmdToggleKeybar",           cmdToggleKeybar);
   ADD_COMMAND("cmdToggleStatusbar",        cmdToggleStatusbar);
   ADD_COMMAND("cmdToggleShowLineFeeds",    cmdToggleShowLineFeeds);
   ADD_COMMAND("cmdToggleShowSpaces",       cmdToggleShowSpaces);
   ADD_COMMAND("cmdToggleShowTabs",         cmdToggleShowTabs);
   ADD_COMMAND("cmdToggleQuickScrollMode",  cmdToggleQuickScrollMode);
   ADD_COMMAND("cmdToggleWordWrap",         cmdToggleWordWrap);
   ADD_COMMAND("cmdOptionTextViewerFont",   cmdOptionTextViewerFont);
   ADD_COMMAND("cmdNavigateUp",             cmdNavigateUp);
   ADD_COMMAND("cmdNavigateDown",           cmdNavigateDown);
   ADD_COMMAND("cmdNavigateLeft",           cmdNavigateLeft);
   ADD_COMMAND("cmdNavigateRight",          cmdNavigateRight);
   ADD_COMMAND("cmdNavigateHome",           cmdNavigateHome);
   ADD_COMMAND("cmdNavigateEnd",            cmdNavigateEnd);
   ADD_COMMAND("cmdNavigatePrevWord",       cmdNavigatePrevWord);
   ADD_COMMAND("cmdNavigateNextWord",       cmdNavigateNextWord);
   ADD_COMMAND("cmdNavigatePageUp",         cmdNavigatePageUp);
   ADD_COMMAND("cmdNavigatePageDown",       cmdNavigatePageDown);
   ADD_COMMAND("cmdNavigateTop",            cmdNavigateTop);
   ADD_COMMAND("cmdNavigateBottom",         cmdNavigateBottom);
   ADD_COMMAND("cmdNavigateSelectUp",       cmdNavigateSelectUp);
   ADD_COMMAND("cmdNavigateSelectDown",     cmdNavigateSelectDown);
   ADD_COMMAND("cmdNavigateSelectLeft",     cmdNavigateSelectLeft);
   ADD_COMMAND("cmdNavigateSelectRight",    cmdNavigateSelectRight);
   ADD_COMMAND("cmdNavigateSelectHome",     cmdNavigateSelectHome);
   ADD_COMMAND("cmdNavigateSelectEnd",      cmdNavigateSelectEnd);
   ADD_COMMAND("cmdNavigateSelectPrevWord", cmdNavigateSelectPrevWord);
   ADD_COMMAND("cmdNavigateSelectNextWord", cmdNavigateSelectNextWord);
   ADD_COMMAND("cmdNavigateSelectPageUp",   cmdNavigateSelectPageUp);
   ADD_COMMAND("cmdNavigateSelectPageDown", cmdNavigateSelectPageDown);
   ADD_COMMAND("cmdNavigateSelectTop",      cmdNavigateSelectTop);
   ADD_COMMAND("cmdNavigateSelectBottom",   cmdNavigateSelectBottom);
END_COMMAND_TABLE;

LCmdInternalViewer::Viewer::Viewer ( GTextViewer::SearchParams& sp, 
                                     LCmdInternalViewer& pwin )
                           :GTextViewer(&sp, 
                                        "Viewer", 
                                        GString::Empty, 
                                        pwin.wrapperWin, 
                                        false, 
                                        false),
                            pwin(pwin)
{
}

LCmdInternalViewer::Viewer::~Viewer ()
{
}

void LCmdInternalViewer::Viewer::onCaretPositionChanged ()
{
   if (pwin.fileLoader.isExecuting())
      return;
   int x = getCurrentPosX() + 1; // Make 1-based index, for the user.
   int y = getCurrentPosY() + 1; // Ditto.
   int lineCount = getLinesCount() + 1; // Count the last (non-linefeed-terminated) line also.
   GStringl columnStr("%StrTVStatusbar_CurColumn", GVArgs(x));
   GStringl lineStr("%StrTVStatusbar_CurLine", GVArgs(y));
   GStringl lineCountStr("%StrTVStatusbar_LineCount", GVArgs(lineCount));
   GStatusbar* sbar = pwin.getStatusbar();
   if (sbar != null)
   {
      sbar->setCellValue("CurrentColumn", columnStr);
      sbar->setCellValue("CurrentLine", lineStr);
      sbar->setCellValue("LineCount", lineCountStr);
   }
}

bool LCmdInternalViewer::Viewer::onButton2Down ( int xpos, int ypos, 
                                                 const GWindowMessage::InputFlags& flags )
{
   GTextViewer::onButton2Down(xpos, ypos, flags);

   if (!isMouseCapture())
   {
      // Show the popup-menu of our parent window.
      if (pwin.isVisiblePopupMenu())
         setVisiblePopupMenu(false);
      else
         setVisiblePopupMenu(true, xpos, ypos);
   }

   return true;
}

bool LCmdInternalViewer::Viewer::onPaint ( GGraphics& g, const GRectangle& rect )
{
   // Don't paint the text while the text-loader is in progress.
   if (pwin.fileLoader.isExecuting())
   {
      GColor bck = getBackgroundColor();
      g.drawFilledRectangle(rect, bck);
      return true;
   }
   else
   {
      return GTextViewer::onPaint(g, rect);
   }
}

LCmdInternalViewer::LCmdInternalViewer ( const GString& filePath )
                   :GFrameWindow("TextViewer",
                                 filePath,
                                 null,
                                 0,
                                 WS2_IGNORE_COLORS_AND_FONT_PROFILE | WS2_OS2Y),
                    filePath(filePath),
                    MSGID_APPENDLINES("appendLines"),
                    dlgSearch("TextViewer.DlgTextSearch"),
                    wrapperWin("Wrapper", 
                               GString::Empty, 
                               this, this, 
                               WS_VISIBLE, 
                               WS2_OS2Y | WS2_IGNORE_COLORS_AND_FONT_PROFILE | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT),
                    viewerWin(dlgSearch, *this),
                    fileLoader(*this)
{
   wrapperWin.setInsets(new GInsets(2 ,2, 2, 2), true);
   wrapperWin.setBorder(new GLineBorder(), true);
   wrapperWin.setLayoutManager(new GBorderLayout(), true);

   // ---
   setAccelTable("ViewerAccel");
   setMenubar("ViewerMenu", lcmd->isUseFancyMenubars());
   setToolbar("ViewerToolbar", "ToolBar");
   viewerWin.setPopupMenu("ViewerContextMenu", lcmd->isUseFancyPopupMenues());
   setKeybarVisible(true);
   setStatusbarVisible(true);
   setAutoLayoutClientWin(&wrapperWin);

   // Add statusbar cells.
   GStatusbar* sbar = getStatusbar();
   sbar->addTextCell("CurrentColumn");
   sbar->addTextCell("CurrentLine");
   sbar->addTextCell("LineCount");
   viewerWin.onCaretPositionChanged(); // Initially set cell values.

   // Activate the window size and position, and make it visible.
   LCmdInternalViewer::FileOptions* opts = FileStore.get(filePath);
   setResizable(true);
   if (opts == null)
   {
      // This file has not been loaded by a recent file viewer instance,
      // so use the default window positions, as well as other options.
      GString sectionName = getName();
      queryProfile(sectionName);
   }
   else
   {
      // Restore all options except window pos/size/state.
      setAutoManageWindowPos(false); 
      GString sectionName = getName();
      queryProfile(sectionName); // Load all options except size/pos/state.
      setAutoManageWindowPos(true); 

      // Position the window as close to the recent instance as possible.
      const GPoint& pos = opts->windowPos;
      const GDimension& size = opts->windowSize;
      GFrameWindow::State state = opts->windowState;
      if (state == GFrameWindow::Minimized)
         state = GFrameWindow::Normalized;
      restoreWindowPosImpl(pos, size, state);
   }

   // Add our self to LCmd's list of viewer instances, so that
   // we can be destroyed in a clean way.
   lcmd->openViewers.add(this);

   // Start the thread that loads the file.
   fileLoader.loadFile(filePath);

   // Position the caret as close to the recent instance as possible.
   if (opts != null)
   {
      const GTextViewer::Pos& pos = opts->caretPos;
      int x = opts->firstVisibleColumn;
      int y = opts->firstVisibleLine;
      viewerWin.restoreCaretPositioningAndView(pos, y, x);
   }

   // Make the caret initially visible in the text area.
   ensureFocusOK();
}

LCmdInternalViewer::~LCmdInternalViewer ()
{
   // Remember caret position, etc. in case the same file is re-opened later.
   GPoint windowPos;
   GDimension windowSize;
   GFrameWindow::State windowState;
   storeWindowPosImpl(windowPos, windowSize, windowState);
   GTextViewer::Pos caretPos = viewerWin.getCurrentPos();
   int firstVisibleLine = viewerWin.getIndexOfFirstVisibleLine();
   int firstVisibleColumn = viewerWin.getIndexOfFirstVisibleColumn();
   LCmdInternalViewer::FileOptions* opts = new LCmdInternalViewer::FileOptions(caretPos, windowSize, windowPos, windowState, firstVisibleLine, firstVisibleColumn);
   if (!FileStore.update(filePath, opts))
      FileStore.put(filePath, opts);

   // Remove our self from LCmd's list of viewer instances.
   // But don't destroy it, since we are already about to be destroyed.
   bool destroy = false;
   int index = lcmd->openViewers.indexOf(*this);
   if (index >= 0)
      lcmd->openViewers.remove(index, 1, destroy);
}

GTextViewer& LCmdInternalViewer::getTextViewer ()
{
   return viewerWin;
}

LCmdDlgTextSearch& LCmdInternalViewer::getSearchParamsEditor ()
{
   return dlgSearch;
}

void LCmdInternalViewer::queryProfile ( const GString& sectName )
{
   GSectionBag& ini = lcmd->getIniProperties();

   // ---
   viewerWin.setShowLinefeeds(ini.getBool(sectName, "showLinefeeds", viewerWin.isShowLinefeeds()));
   viewerWin.setShowSpaces(ini.getBool(sectName, "showSpaces", viewerWin.isShowSpaces()));
   viewerWin.setShowTabs(ini.getBool(sectName, "showTabs", viewerWin.isShowTabs()));
   viewerWin.setChrShowLinefeed(ini.getChar(sectName, "chrShowLinefeed", viewerWin.getChrShowLinefeed()));
   viewerWin.setChrShowSpace(ini.getChar(sectName, "chrShowSpace", viewerWin.getChrShowSpace()));
   viewerWin.setChrShowTab(ini.getChar(sectName, "chrShowTab", viewerWin.getChrShowTab()));
   viewerWin.setQuickScrollMode(ini.getBool(sectName, "quickScrollMode", viewerWin.isQuickScrollMode()));
   viewerWin.setWordWrap(ini.getBool(sectName, "wordWrap", viewerWin.isWordWrap()));

   // Reactivate the stored window size/pos/state, and recurse through all 
   // of our child windows.
   GFrameWindow::queryProfile(sectName);
}

void LCmdInternalViewer::writeProfile ( const GString& sectName, bool force )
{
   GSectionBag& ini = lcmd->getIniProperties();

   // Store window size/pos/state, and recurse through all child windows.
   GFrameWindow::writeProfile(sectName, force);

   // ---
   if (force || true)
      storeWindowPos(sectName, "Pos");
   ini.putBool(sectName, "showLinefeeds", viewerWin.isShowLinefeeds(), force || true);
   ini.putBool(sectName, "showSpaces", viewerWin.isShowSpaces(), force || true);
   ini.putBool(sectName, "showTabs", viewerWin.isShowTabs(), force || true);
   ini.putChar(sectName, "chrShowLinefeed", viewerWin.getChrShowLinefeed(), force || true);
   ini.putChar(sectName, "chrShowSpace", viewerWin.getChrShowSpace(), force || true);
   ini.putChar(sectName, "chrShowTab", viewerWin.getChrShowTab(), force || true);
   ini.putBool(sectName, "quickScrollMode", viewerWin.isQuickScrollMode(), force || true);
   ini.putBool(sectName, "wordWrap", viewerWin.isWordWrap(), force || true);
}

LCmdInternalViewer::FileOptions::FileOptions ( const GTextViewer::Pos& caretPos,
                                               const GDimension& windowSize,
                                               const GPoint& windowPos,
                                               GFrameWindow::State windowState,
                                               int firstVisibleLine,
                                               int firstVisibleColumn )
                                :caretPos(caretPos),
                                 windowSize(windowSize),
                                 windowPos(windowPos),
                                 windowState(windowState),
                                 firstVisibleLine(firstVisibleLine),
                                 firstVisibleColumn(firstVisibleColumn)
{
}

LCmdInternalViewer::FileOptions::~FileOptions ()
{
}

LCmdInternalViewer::FileLoader::FileLoader ( LCmdInternalViewer& owner )
                               :GWorkerThread("DlgTextViewerFileLoadProgress", 200),
                                owner(owner)
{
}

LCmdInternalViewer::FileLoader::~FileLoader ()
{
}

void LCmdInternalViewer::FileLoader::loadFile ( const GString& path )
{
   GStringl loadingStr("%StrTVStatusbar_Status_Loading", GVArgs(path));
   owner.pushStatusbarText(loadingStr);
   owner.setText(path);
   owner.viewerWin.removeAllText();
   owner.filePath = path;
   workModal(owner, 350);
   owner.viewerWin.invalidateAll(true); // Force everything to be updated.
   owner.viewerWin.updateGuiAfterContentChange();
   owner.viewerWin.onCaretPositionChanged();
   owner.popStatusbarText();
}

void LCmdInternalViewer::FileLoader::runTheWorkerThread ( GWorkerThread& worker )
{
   try {
      GVfsLocal vfs;
      const bool textMode = true;
      GFileInputStream s(vfs, owner.filePath, textMode);
      GString line(1024);
      const int BlockSize = 128; // Number of lines to read before adding them to the text viewer window.
      GArray<GString> arr(BlockSize);
      while (!isStopRequested())
      {
         // ---
         for (int i=0; i<BlockSize && !isStopRequested(); i++)
         {
            int count = arr.getCount();
            if (s.readString(line) == "")
            {
               arr.remove(i, count-i);
               break;
            }

            // The string array should not delete the added string, 
            // because its ownership is transferred to the text viewer
            // GUI upon handling of MSGID_APPENDLINES.
            const bool autoDelete = false;
            GString* str = new GString(line);
            if (i < count)
               arr.set(i, str, autoDelete);
            else
               arr.add(str, autoDelete);
         }

         if (arr.getCount() <= 0)
            break;

         sendUserMessageToMonitor(owner.MSGID_APPENDLINES, &arr);

         // Remove all text line elements from the array.
         // This will not delete the contained text lines, because we 
         // gave a false "autoDelete" argument for each and every line
         // that we added to it. This is part of the algorithm, because 
         // the ownership of the text lines belongs to the GUI text viewer.
         arr.removeAll(); 
      }
   } catch (GIOException& e) {
      GStringl msg("%VIEWER_ERR_LOADFILE", GVArgs(owner.filePath).add(e.toString()));
      GString stackTrace = e.getStackTrace(e.toString());
      GWorkerThread::showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Od", GString::Empty, false, stackTrace);
   }
}

void LCmdInternalViewer::FileLoader::onWorkerThreadUserEvent ( GWorkerThread& worker, 
                                                               GDialogPanel& /*monitor*/, 
                                                               const GString& msgID, 
                                                               GObject* userParam )
{
   if (msgID == owner.MSGID_APPENDLINES)
   {
      GArray<GString>* arr = dynamic_cast<GArray<GString>*>(userParam);
      if (arr != null)
      {
         for (int i=0, num=arr->getCount(); i<num; i++)
         {
            GString* line = &arr->get(i);
            owner.viewerWin.appendLine(line, true, false);
         }
      }
   }
}

void LCmdInternalViewer::FileLoader::onWorkerThreadUpdateMonitor ( GWorkerThread& worker, 
                                                                   GDialogPanel& monitor )
{
   int count = owner.viewerWin.getLinesCount();
   monitor.setComponentValue("LINE_COUNT", count);
}

void LCmdInternalViewer::FileLoader::onWorkerThreadCommand ( GWorkerThread& worker, 
                                                             GDialogPanel& monitor, 
                                                             const GString& cmd )
{
   if (cmd == "DLG_CANCEL")
   {
      monitor.setComponentEnabled(cmd, false);
      requestStop();
   }
}

void LCmdInternalViewer::ensureFocusOK ()
{
   viewerWin.grabFocus();
}

void LCmdInternalViewer::updateCommandStates ()
{
   bool anySelected = viewerWin.isAnySelectedText();
   bool moreThanOneLine = (viewerWin.getLinesCount() > 1);
   setCommandEnableState("cmdCopy", anySelected);
   setCommandEnableState("cmdSearch", moreThanOneLine);
   setCommandEnableState("cmdSearchNext", moreThanOneLine);
   setCommandEnableState("cmdGotoLineNum", moreThanOneLine);
   setCommandEnableState("cmdSelectAll", moreThanOneLine);

   // ---
   setCommandToggleState("cmdToggleToolbar", isToolbarVisible());
   setCommandToggleState("cmdToggleKeybar", isKeybarVisible());
   setCommandToggleState("cmdToggleStatusbar", isStatusbarVisible());
   setCommandToggleState("cmdToggleShowLineFeeds", viewerWin.isShowLinefeeds());
   setCommandToggleState("cmdToggleShowSpaces", viewerWin.isShowSpaces());
   setCommandToggleState("cmdToggleShowTabs", viewerWin.isShowTabs());
   setCommandToggleState("cmdToggleQuickScrollMode", viewerWin.isQuickScrollMode());
   setCommandToggleState("cmdToggleWordWrap", viewerWin.isWordWrap());
}

bool LCmdInternalViewer::onActivate ()
{
   GFrameWindow::onActivate();
   viewerWin.grabFocus();
   return true;
}

bool LCmdInternalViewer::onClose ()
{
   cmdExit();
   return true;
}

void LCmdInternalViewer::cmdFileOpen ( GAbstractCommand* /*cmd*/ )
{
   GString initialPath = filePath;
   if (initialPath == "")
   {
      try {
         initialPath = GFile::GetCurrentDir();
      } catch (APIRET& /*rc*/) {
         // Not critical. Just ignore this.
      }
   }
   GString path = userChooseFile("%cmdFileOpen", initialPath);
   if (path != "")
   {
      GFile f(path);
      fileLoader.loadFile(f);
   }
}

void LCmdInternalViewer::cmdExit ( GAbstractCommand* /*cmd*/ )
{
   pushStatusbarText("%StrTVStatusbar_Status_Closing");
   lcmd->mainWin.sendUserMessage("EXIT_TEXTVIEWER", this);
}

void LCmdInternalViewer::cmdCopy ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdCopy();
}

void LCmdInternalViewer::cmdSearch ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdSearch();
}

void LCmdInternalViewer::cmdSearchNext ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdSearchNext();
}

void LCmdInternalViewer::cmdGotoLineNum ( GAbstractCommand* /*cmd*/ )
{
   int max = viewerWin.getLinesCount();
   int curLine = viewerWin.getCurrentPosY();
   LCmdDlgGotoLine dlg(1, max, curLine + 1, 0);
   int line = dlg.execute(this);
   if (line >= 1 && line <= max)
   {
      int column = viewerWin.getIndexOfFirstVisibleColumn();
      viewerWin.gotoPos(line - 1, column);
   }
}

void LCmdInternalViewer::cmdSelectAll ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdSelectAll();
}

void LCmdInternalViewer::cmdUnselect ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.unselectAll();
}

void LCmdInternalViewer::cmdShowContextMenu ( GAbstractCommand* /*cmd*/ )
{
   int x = viewerWin.getCaretXPos() + 5;
   int y = viewerWin.getCaretYPos();
   viewerWin.setVisiblePopupMenu(true, x, y);
}

void LCmdInternalViewer::cmdToggleToolbar ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isToolbarVisible();
   setToolbarVisible(show);
}

void LCmdInternalViewer::cmdToggleKeybar ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isKeybarVisible();
   setKeybarVisible(show);
}

void LCmdInternalViewer::cmdToggleStatusbar ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isStatusbarVisible();
   setStatusbarVisible(show);
}

void LCmdInternalViewer::cmdToggleShowLineFeeds ( GAbstractCommand* /*cmd*/ )
{
   bool flag = viewerWin.isShowLinefeeds();
   viewerWin.setShowLinefeeds(!flag);
}

void LCmdInternalViewer::cmdToggleShowSpaces ( GAbstractCommand* /*cmd*/ )
{
   bool flag = viewerWin.isShowSpaces();
   viewerWin.setShowSpaces(!flag);
}

void LCmdInternalViewer::cmdToggleShowTabs ( GAbstractCommand* /*cmd*/ )
{
   bool flag = viewerWin.isShowTabs();
   viewerWin.setShowTabs(!flag);
}

void LCmdInternalViewer::cmdToggleQuickScrollMode ( GAbstractCommand* /*cmd*/ )
{
   bool flag = viewerWin.isQuickScrollMode();
   viewerWin.setQuickScrollMode(!flag);
}

void LCmdInternalViewer::cmdToggleWordWrap ( GAbstractCommand* /*cmd*/ )
{
   bool flag = viewerWin.isWordWrap();
   viewerWin.setWordWrap(!flag);
}

void LCmdInternalViewer::cmdOptionTextViewerFont ( GAbstractCommand* /*cmd*/ )
{
   GString titleText = "%DlgOptFontCmn_TitleChooseTextViewer";
   viewerWin.userChooseFont(titleText, true, this, GString::Empty, true);
}

void LCmdInternalViewer::cmdNavigateUp ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateUp();
}

void LCmdInternalViewer::cmdNavigateDown ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateDown();
}

void LCmdInternalViewer::cmdNavigateLeft ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateLeft();
}

void LCmdInternalViewer::cmdNavigateRight ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateRight();
}

void LCmdInternalViewer::cmdNavigateHome ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateHome();
}

void LCmdInternalViewer::cmdNavigateEnd ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateEnd();
}

void LCmdInternalViewer::cmdNavigatePrevWord ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigatePrevWord();
}

void LCmdInternalViewer::cmdNavigateNextWord ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateNextWord();
}

void LCmdInternalViewer::cmdNavigatePageUp ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigatePageUp();
}

void LCmdInternalViewer::cmdNavigatePageDown ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigatePageDown();
}

void LCmdInternalViewer::cmdNavigateTop ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateTop();
}

void LCmdInternalViewer::cmdNavigateBottom ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateBottom();
}

void LCmdInternalViewer::cmdNavigateSelectUp ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectUp();
}

void LCmdInternalViewer::cmdNavigateSelectDown ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectDown();
}

void LCmdInternalViewer::cmdNavigateSelectLeft ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectLeft();
}

void LCmdInternalViewer::cmdNavigateSelectRight ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectRight();
}

void LCmdInternalViewer::cmdNavigateSelectHome ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectHome();
}

void LCmdInternalViewer::cmdNavigateSelectEnd ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectEnd();
}

void LCmdInternalViewer::cmdNavigateSelectPrevWord ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectPrevWord();
}

void LCmdInternalViewer::cmdNavigateSelectNextWord ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectNextWord();
}

void LCmdInternalViewer::cmdNavigateSelectPageUp ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectPageUp();
}

void LCmdInternalViewer::cmdNavigateSelectPageDown ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectPageDown();
}

void LCmdInternalViewer::cmdNavigateSelectTop ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectTop();
}

void LCmdInternalViewer::cmdNavigateSelectBottom ( GAbstractCommand* /*cmd*/ )
{
   viewerWin.cmdNavigateSelectBottom();
}
