/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelModeFull.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelHeader.h"
#include "lcmd/LCmdFilePanelInfoBar.h"
#include "lcmd/LCmdFilePanelFrame.h"
#include "lcmd/LCmd.h"

#include "glib/gui/layout/GBorderLayout.h"

LCmdFilePanelModeFull::LCmdFilePanelModeFull ( LCmdFilePanel& fpanel,
                                               const GString& winName )
                      :GWindow(&fpanel.tabs, winName, GString::Empty, WS2_IGNORE_COLORS_AND_FONT_PROFILE | WS2_OS2Y),
                       LCmdFilePanelModeAbstract(fpanel),
                       fpanel(fpanel),
                       columnsBar(*this, GBorderLayout::NORTH),
                       list(*this)
{
   setLayoutManager(new GBorderLayout(), true);
}

LCmdFilePanelModeFull::~LCmdFilePanelModeFull ()
{
}

LCmdFilePanelModeFull::List::List ( LCmdFilePanelModeFull& parentWin )
                      :GDecoratedWindow("List",
                                        GBorderLayout::CENTER, 
                                        &parentWin,
                                        GString::Empty,
                                        WS_VISIBLE,
                                        WS2_OS2Y),
                       parentWin(parentWin),
                       columnsBar(parentWin.columnsBar),
                       fpanel(parentWin.fpanel),
                       totalWidthOfAllColumns(0),
                       curSelectedItem(-1), // Initially no selected item.
                       firstVisibleItem(-1), // Initially no items in list.
                       itemsInListVer(1), // Prevent initial 'divide by zero'.
                       leftMostColumnXPos(0),
                       itemHeight(0),
                       iconSize(0),
                       fullModeIconColumnWidth(0),
                       wideModeFSizeColumnWidth(0),
                       fullModeFDateColumnWidth(0),
                       fullModeFTimeColumnWidth(0),
                       widthOfWidestItemFName(0),
                       widthOfWidestItemFExt(0)
{
   parentWin.fpanel.items.addFileItemContainerListener(this);

   setHScrollVisible(true);
   setVScrollVisible(true);

   // Activate the popup menu of which to display when user
   // right-click on a file item or on some other space of this window.
   GProgram& prg = GProgram::GetProgram();
   setPopupMenu("FileItemContextMenu", prg.isUseFancyPopupMenues());

   // Let the file panel object handle the drag-n-drop events.
   setDragDropHandler(&fpanel, false);
}

LCmdFilePanelModeFull::List::~List()
{
   parentWin.fpanel.items.removeFileItemContainerListener(this);
}

void LCmdFilePanelModeFull::invalidateAll ( bool inclChildren ) const
{
   GWindow::invalidateAll(false);
   list.invalidateAll(inclChildren);
   columnsBar.invalidateAll(inclChildren);
}

void LCmdFilePanelModeFull::invalidateRect ( const GRectangle& rect ) const
{
   GWindow::invalidateRect(rect);
   list.invalidateRect(rect);
   columnsBar.invalidateRect(rect);
}

bool LCmdFilePanelModeFull::isHorizontallyScrollable () const
{
   return list.isHScrollbarEnabled();
}

void LCmdFilePanelModeFull::List::allItemsHaveBeenRemoved ()
{
   leftMostColumnXPos = 0;
   curSelectedItem = -1;
   firstVisibleItem = -1;
   
   // The columnsbar must be updated when the filename list has been reread.
   calcDefaultColumnWidths();
   if (fpanel.view.showColumnsBar)
      columnsBar.layout();
}

void LCmdFilePanelModeFull::onViewHasBeenActivated ()
{
   int curSel = list.curSelectedItem;
   list.leftMostColumnXPos = 0;
   layout();
   fpanel.frameWin.layout();
   fpanel.frameWin.headerWin.layout();
   list.updateScrollBarRange(); // Will call updateScrollBarPos() as well
   list.curSelectedItem = -1; // Just to force reselection of current item
   fpanel.selectItem(curSel);
}

int LCmdFilePanelModeFull::getCurrentSelectedIndex () const
{
   return list.curSelectedItem;
}

int LCmdFilePanelModeFull::getFirstVisibleIndex () const
{
   return list.firstVisibleItem;
}

int LCmdFilePanelModeFull::List::calcLastVisibleItemNC () const
{
   int lastVisibleIdx = firstVisibleItem + itemsInListVer - 1;
   if (lastVisibleIdx >= fpanel.items.getCount())
      return fpanel.items.getCount() - 1;
   if (lastVisibleIdx < firstVisibleItem)
      return firstVisibleItem;
   return lastVisibleIdx;
}

int LCmdFilePanelModeFull::List::calcItemIdxFromPos ( int xpos, int ypos ) const
{
   GRectangle rect(filesRect); // Get a working copy of current panel window rectangle

   if (xpos < 0 ||
       xpos >= rect.x + rect.width - 1 ||
       ypos < 0 ||
       ypos >= rect.y + rect.height - 1)
   {
      // Specified position is outside window, so we must return -1
      return -1;
   }

   // Current view mode has never more than one column.
   int idx = firstVisibleItem + ((rect.y + rect.height - 1 - ypos) / itemHeight);
   if (idx >= firstVisibleItem + itemsInListVer)
      idx = -1;

   if (idx < 0 || idx >= fpanel.items.getCount())
      return -1;

   return idx;
}

void LCmdFilePanelModeFull::List::calcDefaultColumnWidths ()
{
   fullModeIconColumnWidth = columnsBar.clmnIcon.calcDefaultWidth();
   wideModeFSizeColumnWidth = columnsBar.clmnSize.calcDefaultWidth();
   fullModeFDateColumnWidth = columnsBar.clmnDate.calcDefaultWidth();
   fullModeFTimeColumnWidth = columnsBar.clmnTime.calcDefaultWidth();

   // ---
   GGraphics g(this);
   widthOfWidestItemFName = 0;
   widthOfWidestItemFExt = 0;
   for (int i=0, num=fpanel.items.getCount(); i<num; i++)
   {
      LCmdFileItem& fitem = fpanel.items.get(i);
      const GString& fname = fitem.path.getName();
      const GString& fext = fitem.path.getExtension();

      int strw1 = g.getWidthOfString(fname);
      int strw2 = g.getWidthOfString(fext);
      if (fitem.isDirectory())
      {
         // Directories doesn't have extensions.
         strw1 += strw2;
         strw2 = 0;
      }
      else
      {
         if (strw2 > widthOfWidestItemFExt)
            widthOfWidestItemFExt = strw2;
      }

      if (strw1 > widthOfWidestItemFName)
         widthOfWidestItemFName = strw1;
   }
}

void LCmdFilePanelModeFull::List::updateHScrollBarPos ()
{
   int pos = -leftMostColumnXPos;
   setHScrollPos(pos);
}

void LCmdFilePanelModeFull::List::updateVScrollBarPos ()
{
   setVScrollPos(firstVisibleItem);
}

void LCmdFilePanelModeFull::List::updateScrollBarPos ()
{
   updateHScrollBarPos();
   updateVScrollBarPos();
}

void LCmdFilePanelModeFull::List::updateScrollBarRange ()
{
   // Calculate the horizontal scrollbar.
   int thumbLen  = filesRect.width;
   int scrollLen = totalWidthOfAllColumns - thumbLen;
   setHScrollRange(scrollLen, thumbLen);
   
   // Calculate the vertical scrollbar.
   thumbLen  = itemsInListVer;
   scrollLen = fpanel.items.getCount() - thumbLen;
   setVScrollRange(scrollLen, thumbLen);

   // Also update the current scrollbar(s) thumb position, so that they also
   // always are placed at the correct position. (E.g. when the file list
   // window has been resized.)
   updateScrollBarPos();
}

void LCmdFilePanelModeFull::List::calcAllColumns ()
{
   columnsBar.layout();

   leftMostColumnXPos = 0;

   // The width of the one and only column is the sum of the
   // width of all the visible buttons in the columnsbar.
   int totalWidth = 0;
   int num = columnsBar.getChildWindowCount();
   for (int i=0; i<num; i++)
   {
      GWindow& butt = columnsBar.getChildWindow(i);
      totalWidth += butt.getWindowSize().width;
   }

   // ---
   totalWidthOfAllColumns = totalWidth;

   // Update the scrollbars of the fileslist of this panel.
   updateScrollBarRange();
}

void LCmdFilePanelModeFull::List::calcItems ()
{
   const int addToItemHeight = lcmd->options.various.addToItemHeight;

   // Query pixels height of the font (the dummy text will not be painted)
   itemHeight = 2 + getHeightOfString("X") + addToItemHeight;
   iconSize = fpanel.calcIconSize();

   if (itemHeight < iconSize + 4 + addToItemHeight)
      itemHeight = iconSize + 4 + addToItemHeight;

   if (itemHeight < 6)
      itemHeight = 6;

   // Calculate how many items there are room for vertically
   itemsInListVer = filesRect.height / itemHeight;
   if (itemsInListVer < 1)
      itemsInListVer = 1;  // This item must never be less than one!

   calcDefaultColumnWidths();
   calcAllColumns(); // Will also call {@link #updateScrollBarRange}.
}

int LCmdFilePanelModeFull::calcItemIdxFromPos ( int xpos, int ypos ) const
{
   return list.calcItemIdxFromPos(xpos, ypos);
}

bool LCmdFilePanelModeFull::calcItemRect ( int itemIndex, GRectangle& rect ) const
{
   return list.calcItemRect(itemIndex, rect);
}

bool LCmdFilePanelModeFull::List::calcItemRect ( int itemIndex, GRectangle& rect ) const
{
   rect.clear();

   // Be sure not to use an illegal index (in case there are no items in list)
   if (itemIndex < 0 || itemIndex >= fpanel.items.getCount())
      return false;

   // Get a copy of current list window size.
   GRectangle r = getWindowRect();

   int winHeight = r.height;
   int winWidth = r.width;

   r.y = winHeight - (((itemIndex - firstVisibleItem) + 1) * itemHeight);

   // Calculate x-pos of the column, with respect to the left margine of window
   int xpos = leftMostColumnXPos;

   r.x = xpos;
   r.height = itemHeight;
   r.width = totalWidthOfAllColumns;
   if (r.width < winWidth)
      r.width = winWidth;

   // Don't paint items outside visible area
   if (r.x >= winWidth || r.y <= -itemHeight)
      return false;

   rect = r;
   return true;
}

void LCmdFilePanelModeFull::drawItem ( int itemIndex )
{
   list.drawItem(itemIndex);
}

void LCmdFilePanelModeFull::drawItem ( int itemIndex,
                                       GGraphics& g,
                                       const GRectangle& itemRect,
                                       bool isDragOver )
{
   list.drawItem(itemIndex, g, itemRect, isDragOver);
}

void LCmdFilePanelModeFull::List::drawItem ( int itemIndex )
{
   GRectangle rect;
   GGraphics g(*this);
   if (parentWin.calcItemRect(itemIndex, rect))
      drawItem(itemIndex, g, rect);
}

void LCmdFilePanelModeFull::List::drawItem ( int itemIndex,
                                             GGraphics& g,
                                             const GRectangle& itemRect,
                                             bool isDragOver )
{
   if (itemIndex < 0)
      return;

   LCmdFileItem& fitem = fpanel.items.get(itemIndex);

   if (isDragOver)
   {
      GRectangle r(itemRect);
      if (fitem.isUpDir() || (!fitem.isDirectory() && !fitem.isProgramFileName()))
         r.height = 2; // Draw a line rather than a box.
      g.drawEmphasiseBox(r, GColor::BLACK, 2, true);
      return;
   }

   bool iscurpanel = fpanel.isCurrentPanel();
   bool iscurselected = (itemIndex == curSelectedItem);
   GRectangle rect(itemRect);

   // Paint the thin border of the current selected item.
   if (iscurselected)
   {
      // Draw a thin border around current selected item.
      g.setLineType(GGraphics::LTSolid);
      if (iscurpanel)
         g.setColor(fpanel.colors.itemSelectedThinActive);
      else
         g.setColor(fpanel.colors.itemSelectedThinInactive);
      g.drawRectangle(rect);
      // Don't erase the thin frame just painted.
      rect.inflateRect(-1, -1);
   }

   // ---
   // Find background color.
   GColor bc;
   if (iscurselected)
   {
      if (iscurpanel)
      {
         if (fitem.isMarked())
            bc = fpanel.colors.itemSelectedBckActiveMarked;
         else
            bc = fpanel.colors.itemSelectedBckActive;
      }
      else
      {
         if (fitem.isMarked())
            bc = fpanel.colors.itemBckMarked;
         else
            bc = fpanel.colors.itemSelectedBckInactive;
      }
   }
   else
   {
      if (fitem.isMarked())
         bc = fpanel.colors.itemBckMarked;
      else
         bc = fpanel.colors.itemBck;
   }

   // ---
   // Find foreground (text) color.
   GColor fc;
   if (fitem.isDirectory())
      fc = fpanel.colors.itemDirTxt;
   else
   if (fitem.isZipOrArchiveFile())
      fc = fpanel.colors.itemArchiveTxt;
   else
   if (fitem.isReadOnly())
      fc = fpanel.colors.itemReadOnlyTxt;
   else
   if (fitem.isHidden() || fitem.isSystem())
      fc = fpanel.colors.itemSysHiddenTxt;
   else
      fc = fpanel.colors.itemFileTxt;

   // ---
   // Clear the background area of the item.
   g.drawFilledRectangle(rect, bc);

   // ---
   // Draw the icon and text, using current selected font and colors
   rect.x += 3 - iscurselected;

   int num = columnsBar.getElementCount();
   for (int i=0; i<num; i++)
   {
      GToolbarElement& elm = columnsBar.getElement(i);
      if (!elm.isVisible())
         continue;
      LCmdFilePanelColumnsBar::Button& butt = dynamic_cast<LCmdFilePanelColumnsBar::Button&>(elm);
      GPoint pt = butt.getWindowPos();
      GDimension dim = butt.getWindowSize();
      GRectangle r(pt.x + 2, rect.y, dim.width - 4, rect.height);
      butt.drawCell(g, fpanel, fitem, r, bc, fc, iscurselected);
   }
}

void LCmdFilePanelModeFull::itemsListHasBeenRefreshed ()
{
   layout();
}

void LCmdFilePanelModeFull::layout ()
{
   GWindow::layout();
   list.layout();
   columnsBar.clmnIcon.cmdColumnsbarButtonDefWidth(); // Always use default width on the icon column.
   columnsBar.layout();
}

void LCmdFilePanelModeFull::List::layout ()
{
   GDecoratedWindow::layout();

   // Keep a fresh copy of current window rectangle of the panel.
   filesRect = getWindowRect();

   // Calculate everything that has to do with column positions, widths,
   // vertical items count, etc.
   calcItems();

   if (curSelectedItem >= firstVisibleItem + itemsInListVer)
   {
      firstVisibleItem = curSelectedItem - (itemsInListVer / 2);
      if (firstVisibleItem < 0)
         firstVisibleItem = 0;
   }
}

bool LCmdFilePanelModeFull::List::onFontNameSizeChanged ( const GString& fontNameSize )
{
   layout();
   return true;
}

bool LCmdFilePanelModeFull::List::onPaint ( class GGraphics& g, const GRectangle& rect )
{
   GRectangle r = filesRect;
   if (fpanel.items.isExecuting())
   {
      // The filename reader thread is still working.
      GStringl msg("%Txt_FP_ReadingFilenames"); // Reading filenames...
      g.drawFilledRectangle(r, fpanel.colors.itemBck);
      g.drawText(msg, r, fpanel.colors.itemFileTxt, GGraphics::DUMMY_COLOR, GGraphics::HCENTER);
   }
   else
   {
      r.x = leftMostColumnXPos;
      r.width = totalWidthOfAllColumns;
      if (r.width < filesRect.width)
         r.width = filesRect.width;
      r.height = itemHeight;
      r.y = filesRect.y + filesRect.height - itemHeight;

      int idxItem = firstVisibleItem;
      if (idxItem >= 0)
      {
         // Loop to draw each item in the column
         int lastItem = idxItem + itemsInListVer - 1;
         int nrOfItems = fpanel.items.getCount();
         while (idxItem <= lastItem && idxItem < nrOfItems)
         {
            drawItem(idxItem, g, r);
            r.y -= itemHeight;
            idxItem++;
         }
      }

      // Clear the rest of the column area on screen
      r.height = r.y + itemHeight;
      r.y = 0;
      g.drawFilledRectangle(r, fpanel.colors.itemBck);
   }

   return true;
}

bool LCmdFilePanelModeFull::List::onButton1Up ( int /*xpos*/, int /*ypos*/, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
      captureMouse(false); // Release the mouse capture
   return true;
}

bool LCmdFilePanelModeFull::List::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   // Activate our frame window (in case LCmd isn't already active).
   setActive();

   // If the user is currently about doing a dynamic filename
   // text search then dismiss it now.
   if (fpanel.headerWin.isPerformingDynamicSearch())
      fpanel.headerWin.dismissDynamicSearch();

   fpanel.activatePanel(); // Activate this file panel.

   int prevSelected = curSelectedItem;
   curSelectedItem = calcItemIdxFromPos(xpos, ypos);
   if (curSelectedItem != prevSelected)
   {
      parentWin.drawItem(prevSelected);
      parentWin.drawItem(curSelectedItem);

      // Request the text area of the infobar to show the information
      // about newly selected item.
      fpanel.infoBar.updateAllFileItemInfoCells();

      // Let "everyone" know that the selection has changed.
      fpanel.onItemSelectionHasChanged();
   }

   if (curSelectedItem >= 0 && flags.isControlKeyDown())
      fpanel.toggleTag(curSelectedItem, true);

   captureMouse(true);

   return true;
}

bool LCmdFilePanelModeFull::List::onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (curSelectedItem >= 0 &&
       curSelectedItem == calcItemIdxFromPos(xpos, ypos))
   {
      if (flags.isAltKeyDown() && flags.isShiftKeyDown())
         fpanel.startSelectedProgram(curSelectedItem);
      else
         fpanel.doEnter(curSelectedItem, true);
   }
   return true;
}

bool LCmdFilePanelModeFull::List::onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   // Activate our frame window (in case LCmd isn't already active).
   setActive();

   if (isMouseCapture())
      return true;

   fpanel.activatePanel();
   setVisiblePopupMenu(true, xpos, ypos);

   return true;
}

bool LCmdFilePanelModeFull::List::onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
      return true;

   // If the user is currently about doing a dynamic filename
   // text search then dismiss it now.
   if (fpanel.headerWin.isPerformingDynamicSearch())
      fpanel.headerWin.dismissDynamicSearch();

   fpanel.activatePanel(); // Specified panel must be active

   int index = calcItemIdxFromPos(xpos, ypos);
   if (index == curSelectedItem)
      return true;

   int prevSelected = curSelectedItem;
   curSelectedItem = index;

   if (fpanel.markedFilesCount > 0)
   {
      if (index < 0 || !fpanel.items.get(index).isMarked())
      {
         // User has right-clicked on an untagged item even though there 
         // are other items currently being tagged. In order to prevent
         // confusion, untag all filename items so that if the user 
         // selects a command from the popup window the command will 
         // work only on the single selected (right clicked) item.
         // This is the same as in the Windows Explorer, which most 
         // users probably expect.
         fpanel.unselectAll();
      }
   }

   parentWin.drawItem(prevSelected);
   parentWin.drawItem(curSelectedItem);

   // Request the text area of the infobar to show the information
   // about newly selected item.
   fpanel.infoBar.updateAllFileItemInfoCells();

   // Let "everyone" know that the selection has changed.
   fpanel.onItemSelectionHasChanged();

   return true;
}

bool LCmdFilePanelModeFull::List::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (!isMouseCapture())
      return false;

   if (curSelectedItem >= 0)
   {
      int index = calcItemIdxFromPos(xpos, ypos);
      if (index >= 0)
      {
         LCmdFileItem& fitem = fpanel.items.get(curSelectedItem);
         bool prevIsTagged = fitem.isMarked();

         // Activate and repaint the calculated item
         int prevSelected = curSelectedItem;
         if (index != prevSelected)
         {
            curSelectedItem = index;
            parentWin.drawItem(prevSelected);
            parentWin.drawItem(index);

            // Request the text area of the infobar to show the information
            // about newly selected item
            fpanel.infoBar.updateAllFileItemInfoCells();

            // Let "everyone" know that the selection has changed.
            fpanel.onItemSelectionHasChanged();
         }

         if (flags.isControlKeyDown())
         {
            LCmdFileItem& itm = fpanel.items.get(index);
            if (itm.isMarked() != prevIsTagged)
               fpanel.toggleTag(index, true);
         }
      }
   }

   return false; // Return true if and only if we have actually set the mouse cursor.
}

bool LCmdFilePanelModeFull::List::onInitMenu ()
{
   lcmd->mainWin.updateCommandStates();
   return true;
}

bool LCmdFilePanelModeFull::List::onHScrollLineUp ()
{
   scrollHorizontal(-10);
   return true;
}

bool LCmdFilePanelModeFull::List::onHScrollLineDown ()
{
   scrollHorizontal(10);
   return true;
}

bool LCmdFilePanelModeFull::List::onHScrollPageUp ()
{
   scrollHorizontal(-(filesRect.x + filesRect.width - 1 - 10));
   return true;
}

bool LCmdFilePanelModeFull::List::onHScrollPageDown ()
{
   scrollHorizontal(filesRect.x + filesRect.width - 1 - 10);
   return true;
}

bool LCmdFilePanelModeFull::List::onHScrollSliderTrack ( int pos )
{
   int absXPos = -leftMostColumnXPos;
   scrollHorizontal(-absXPos + pos);
   return true;
}

bool LCmdFilePanelModeFull::List::onVScrollLineUp ()
{
   scrollVertical(-1);
   return true;
}

bool LCmdFilePanelModeFull::List::onVScrollLineDown ()
{
   scrollVertical(1);
   return true;
}

bool LCmdFilePanelModeFull::List::onVScrollPageUp ()
{
   int add = -(itemsInListVer - 1);
   if (add == 0)
      add = -1;
   scrollVertical(add);
   return true;
}

bool LCmdFilePanelModeFull::List::onVScrollPageDown ()
{
   int add = itemsInListVer - 1;
   if (add == 0)
      add = 1;
   scrollVertical(add);
   return true;
}

bool LCmdFilePanelModeFull::List::onVScrollSliderTrack ( int pos )
{
   scrollVertical(pos - firstVisibleItem);
   return true;
}

void LCmdFilePanelModeFull::List::scrollHorizontal ( int add )
{
   int absXPos = -leftMostColumnXPos;
   int prevAbsXPos = absXPos;

   absXPos += add;
   if (absXPos < 0)
   {
      absXPos = 0;
   }
   else
   {
      int x = totalWidthOfAllColumns - (filesRect.x + filesRect.width);
      if (absXPos > x)
         absXPos = x;
   }

   // Find the pixel X-position of the column, with respect to the left edge 
   // of the files list window, which is at pixel X-position 0.
   if (absXPos == 0)
      leftMostColumnXPos = 0;
   else
      leftMostColumnXPos = -absXPos;

   // Scroll the content of the files container.
   scrollWindow(prevAbsXPos - absXPos, 0, true, &filesRect);

   // Scroll the Columns Bar as well, so that it reflects the current
   // horizontal scrolling state of the files container.
   int movex = prevAbsXPos - absXPos;
   columnsBar.moveButtonsHor(movex);

   updateHScrollBarPos();
}

void LCmdFilePanelModeFull::List::scrollVertical ( int add )
{
   int old = firstVisibleItem;
   int newIndex = firstVisibleItem + add;
   if (newIndex <= 0)
      firstVisibleItem = 0;
   else
   if (newIndex >= fpanel.items.getCount())
      firstVisibleItem = fpanel.items.getCount() - 1;
   else
      firstVisibleItem = newIndex;
   int diff = firstVisibleItem - old;
   if (diff != 0)
   {
      GRectangle scroll(filesRect);
      scroll.y = scroll.y + scroll.height - (itemHeight * itemsInListVer);
      scrollWindow(0, add * itemHeight, true, &scroll);
   }
   updateVScrollBarPos();
}

int LCmdFilePanelModeFull::navigateDown ()
{
   return list.navigateDown();
}

int LCmdFilePanelModeFull::navigateUp ()
{
   return list.navigateUp();
}

int LCmdFilePanelModeFull::navigateEnd ()
{
   return list.navigateEnd();
}

int LCmdFilePanelModeFull::navigateHome ()
{
   return list.navigateHome();
}

int LCmdFilePanelModeFull::navigateLeft ()
{
   return list.navigateLeft();
}

int LCmdFilePanelModeFull::navigateRight ()
{
   return list.navigateRight();
}

int LCmdFilePanelModeFull::navigatePageDown ()
{
   return list.navigatePageDown();
}

int LCmdFilePanelModeFull::navigatePageUp ()
{
   return list.navigatePageUp();
}

int LCmdFilePanelModeFull::navigateRandom ( int index )
{
   return list.navigateRandom(index);
}

int LCmdFilePanelModeFull::List::navigateDown ()
{
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex + 1;
   if (newIndex >= fpanel.items.getCount())
      return -1;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC();
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      curSelectedItem = newIndex;
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If requested item is the item just below current downmost visible item

   if (newIndex == firstVisibleItem + itemsInListVer)
   {
      curSelectedItem = newIndex;
      GRectangle rScroll(filesRect);
      rScroll.y = (rScroll.y + rScroll.height - 1) - (itemHeight * (itemsInListVer - 1));
      scrollWindow(0, itemHeight, true, &rScroll);
      firstVisibleItem += 1;

      if (newIndex - 1 >= firstVisibleItem &&
          newIndex - 1 <= firstVisibleItem + itemsInListVer - 1)
      {
         parentWin.drawItem(newIndex - 1);
      }

      parentWin.drawItem(newIndex);
      updateVScrollBarPos();
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return navigateRandom(newIndex);
}

int LCmdFilePanelModeFull::List::navigateUp ()
{
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex - 1;
   if (newIndex < 0)
      return -1;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC();
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      curSelectedItem = newIndex;
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If requested item is the item just above current uppermost visible item

   if (newIndex == firstVisibleItem - 1)
   {
      curSelectedItem = newIndex;
      GRectangle rScroll(filesRect);
      rScroll.y = (rScroll.y + rScroll.height - 1) - (itemHeight * itemsInListVer);
      rScroll.height -= itemHeight + 1;
      scrollWindow(0, -itemHeight, true, &rScroll);
      firstVisibleItem -= 1;

      if (newIndex + 1 >= firstVisibleItem &&
          newIndex + 1 <= firstVisibleItem + itemsInListVer - 1)
      {
         parentWin.drawItem(newIndex + 1);
      }

      parentWin.drawItem(newIndex);
      updateVScrollBarPos();
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return navigateRandom(newIndex);
}

int LCmdFilePanelModeFull::List::navigateEnd ()
{
   int prevIndex = curSelectedItem;
   int newIndex = fpanel.items.getCount() - 1;
   if (newIndex < 0)
      return -1;

   curSelectedItem = newIndex;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC();
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------

   firstVisibleItem = fpanel.items.getCount() - itemsInListVer;
   if (firstVisibleItem < 0)
      firstVisibleItem = 0;

   invalidateAll(true);
   updateVScrollBarPos();
   return newIndex;
}

int LCmdFilePanelModeFull::List::navigateHome ()
{
   int prevIndex = curSelectedItem;
   int newIndex = 0;
   if (newIndex >= fpanel.items.getCount())
      return -1;

   curSelectedItem = newIndex;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC();
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------

   firstVisibleItem = 0;
   invalidateAll(true);
   updateVScrollBarPos();
   return newIndex;
}

int LCmdFilePanelModeFull::List::navigateLeft ()
{
   if (totalWidthOfAllColumns > filesRect.width)
      onHScrollLineUp();
   return curSelectedItem;
}

int LCmdFilePanelModeFull::List::navigateRight ()
{
   if (totalWidthOfAllColumns > filesRect.width)
      onHScrollLineDown();
   return curSelectedItem;
}

int LCmdFilePanelModeFull::List::navigatePageDown ()
{
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex + itemsInListVer;
   if (newIndex >= fpanel.items.getCount())
      newIndex = fpanel.items.getCount() - 1;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC();
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      curSelectedItem = newIndex;
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If previously select item was vithin visible area

   if (prevIndex >= firstVisibleItem &&
       prevIndex <= lastVisibleNC)
   {
      curSelectedItem = newIndex;

      firstVisibleItem += itemsInListVer;

      if (firstVisibleItem > fpanel.items.getCount() - itemsInListVer)
         firstVisibleItem = fpanel.items.getCount() - itemsInListVer;

      if (firstVisibleItem < 0)
         firstVisibleItem = 0;

      invalidateAll(true);
      updateVScrollBarPos();
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return navigateRandom(newIndex);
}

int LCmdFilePanelModeFull::List::navigatePageUp ()
{
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex - itemsInListVer;
   if (newIndex < 0)
      newIndex = 0;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC ();
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      curSelectedItem = newIndex;
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // If previously select item was vithin visible area

   if (prevIndex >= firstVisibleItem &&
       prevIndex <= lastVisibleNC)
   {
      curSelectedItem = newIndex;

      firstVisibleItem -= itemsInListVer;
      if (firstVisibleItem < 0)
         firstVisibleItem = 0;

      invalidateAll(true);
      updateVScrollBarPos();
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // The requested item is random, so simply let the general selection
   // function do the job for us

   return navigateRandom(newIndex);
}

int LCmdFilePanelModeFull::List::navigateRandom ( int index )
{
   int prevIndex = curSelectedItem;
   if (index < 0 || index >= fpanel.items.getCount())
      return -1;

   curSelectedItem = index;

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   int lastVisibleNC = calcLastVisibleItemNC();
   if (index >= firstVisibleItem && index <= lastVisibleNC)
   {
      parentWin.drawItem(prevIndex);
      parentWin.drawItem(index);
      return index;
   }

   // --------------------------------------------------------------------------
   // Place newly selected item in the center of the fpanel

   firstVisibleItem = index - (itemsInListVer / 2);

   if (fpanel.items.getCount() - index < itemsInListVer / 2)
      firstVisibleItem = fpanel.items.getCount() - itemsInListVer;

   if (firstVisibleItem < 0)
      firstVisibleItem = 0;

   invalidateAll(true);
   updateVScrollBarPos();
   return index;
}

GString LCmdFilePanelModeFull::userChooseFont ( const GString& titleText, 
                                                bool setFont, 
                                                GWindow* parentWin, 
                                                const GString& initialFont,
                                                bool fixedOnly )
{
   return list.userChooseFont(titleText, setFont, parentWin, initialFont, fixedOnly);
}

GString LCmdFilePanelModeFull::getFontNameSize () const
{
   return list.getFontNameSize();
}
