/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_FILEPANELMODEBRIEF
#define __LCMD_FILEPANELMODEBRIEF

#include "lcmd/LCmdFilePanelModeAbstract.h"
#include "lcmd/LCmdFileItemContainer.h"

#include "glib/gui/GDecoratedWindow.h"

/**
 * @author  Leif Erik Larsen
 * @since   2006.01.15
 */
class LCmdFilePanelModeBrief : public GDecoratedWindow,
                               public LCmdFilePanelModeAbstract, 
                               public LCmdFileItemContainer::Listener
{
   public:

      /** The file panel containing the filenames list that we show. */
      class LCmdFilePanel& fpanel;

      /** The dynamic array of information about all columns of file items. */
      GArray<class LCmdFileColumn> columns;

      /** Always contain current rectangle of the file names container area. */
      GRectangle filesRect;

      /** Index of current selected item, or -1 if no item is selected. */
      int curSelectedItem;

      /** Index of first (uppermost) visible item, or -1 if there is no items in the list. */
      int firstVisibleItem;

      /** Number of items in list vertically, at least 1. */
      int itemsInListVer;

      /** X-pos of left edge of leftmost visible column (may be negative). */
      int leftMostColumnXPos;

      /** Index of first visible column (may be clipped). */
      int leftMostColumn;

      /** Height (pixels) of a filename item. */
      int itemHeight;

      /** Pixels height and width of a file icon (depends on icon type). */
      int iconSize;

   private:
  
      /** Used to make sure that the columns are calculated correctly upon startup even if the file panel is initially hidden. */
      bool hasEverBeenVisible;

   public:

      explicit LCmdFilePanelModeBrief ( class LCmdFilePanel& fpanel,
                                        const GString& winName );

      virtual ~LCmdFilePanelModeBrief ();

   private:

      virtual void allItemsHaveBeenRemoved ();

      void calcAllColumns ();

      /**
       * Calculate the x-position of the indexed column, in case active view
       * mode is brief. The calculated x-position is with respect to the left
       * of the panel window, which is zero.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       */
      int calcColumnXPos ( int idx ) const;

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       */
      void calcItems ();

      /**
       * Calculate the horizontal position of the first leftmost visible
       * pixel of the leftmost visible column.
       *
       * The calculated x-position is with respect to the absolute
       * leftmost pixel of the very first column, which is alway zero.
       * The calculated value is used to calculate the thumb position of
       * the horizontal scrollbar of the brief view mode.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       */
      int calcHScrollPos ();

      /**
       * This function will calculate and return the index of the last 
       * visible (non-clipped) item. This index will never be less than
       * {@link #firstVisibleItem}, even if that item is clipped.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       */
      int calcLastVisibleItemNC ();

      /**
       * Draw indexed coulmn of filename items.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @return  Left margine position of next column (idxColumn + 1).
       */
      int drawColumn ( int idxColumn );
      int drawColumn ( class GGraphics& g, int idxColumn );

      /**
       * Draw one or more culumns. Typically called to draw all
       * visible columns to the right of {@link #iIdxFirstColumn}.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @param   g              Graphics of where to draw.
       * @param   idxFirstColumn Index of first column to draw.
       * @param   count          Number of columns to draw, or -1 if we shall
       *                         draw all visible columns to the right of
       *                         idxFirstColumn, starting with
       *                         idxFirstColumn.
       * @return Left margine position of next column (iIdxColumn + 1).
       */
      int drawColumns ( class GGraphics& g, int idxFirstColumn, int count );

      void scrollHorizontal ( int add );

      void updateHScrollBarPos ();
      void updateHScrollBarRange ();

   protected:

      virtual bool onButton1Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

      virtual bool onFontNameSizeChanged ( const GString& fontNameSize );

      virtual bool onHScrollLineUp ();
      virtual bool onHScrollLineDown ();
      virtual bool onHScrollPageUp ();
      virtual bool onHScrollPageDown ();
      virtual bool onHScrollSliderTrack ( int pos );

      virtual bool onInitMenu ();
      virtual bool onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onPaint ( class GGraphics& g, const class GRectangle& rect );

   public:

      virtual int calcItemIdxFromPos ( int xpos, int ypos ) const;

      virtual bool calcItemRect ( int itemIndex, GRectangle& rect ) const;

      void calculateWidthOfAllItemsAfterReloadFilenames ();

      virtual void drawItem ( int itemIndex );

      /**
       * Draw the single indexed filename item.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @param   itemIndex  Index (0..) of which file item to paint.
       *                     Caller must be sure this index is within
       *                     legal range.
       * @param   g          The actual PS of where to draw.
       * @param   itemRect   Rectangle area of where to paint the item.
       * @param   isDragOver True if an object being dragged is located
       *                     above the indexed item, or else false.
       */
      virtual void drawItem ( int itemIndex, 
                              class GGraphics& g, 
                              const class GRectangle& itemRect, 
                              bool isDragOver = false );

      virtual int getCurrentSelectedIndex () const;

      virtual int getFirstVisibleIndex () const;

      virtual void invalidateAll ( bool inclChildren ) const;

      virtual void invalidateRect ( const GRectangle& rect ) const;

      virtual bool isHorizontallyScrollable () const;

      virtual void itemsListHasBeenRefreshed ();

      virtual void layout ();

      virtual int navigateDown ();
      virtual int navigateUp ();
      virtual int navigateEnd ();
      virtual int navigateHome ();
      virtual int navigateLeft ();
      virtual int navigateRight ();
      virtual int navigatePageDown ();
      virtual int navigatePageUp ();
      virtual int navigateRandom ( int index );

      virtual void onViewHasBeenActivated ();
};

#endif // #ifndef __LCMD_FILEPANELMODEBRIEF
