/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelFrame.h"
#include "lcmd/LCmdPanelsParent.h"
#include "lcmd/LCmdSubClient.h"
#include "lcmd/LCmd.h"

#include "glib/sys/GSystem.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/gui/layout/GBorderLayout.h"

LCmdFilePanelFrame::LCmdFilePanelFrame ( const GString& name,
                                         const GString& constraints,
                                         LCmdPanelsParent& parentWin )
                   :GWindow(&parentWin, name, constraints, WS2_OS2Y),
                    panelsParent(parentWin),
                    headerWin(*this, GBorderLayout::NORTH),
                    fpanel(*this, GBorderLayout::CENTER),
                    infoBar(*this, GBorderLayout::SOUTH)
{
   setInsets(new GInsets(2, 2, 2, 2), true);
   setBorder(new GLineBorder(), true);
   setLayoutManager(new GBorderLayout(), true);
}

LCmdFilePanelFrame::~LCmdFilePanelFrame ()
{
}

GString LCmdFilePanelFrame::getName () const
{
   if (lcmd->fp1 == &lcmd->mainWin.subClient->panelsParent.frame1.fpanel)
      return GWindow::getName();

   // The file panels are currently swapped, so we must return
   // the name of the opposite file panel frame.
   if (this == &lcmd->mainWin.subClient->panelsParent.frame1)
      return panelsParent.FRAME2_NAME;
   else
      return panelsParent.FRAME1_NAME;
}

int LCmdFilePanelFrame::getPreferredWidth () const
{
   GDimension dim = panelsParent.getWindowSize();
   LCmdOptions& opt = LCmdOptions::GetOptions();
   int panel1Width = (dim.width * opt.panel1RelWidth) / 1000;
   if (&fpanel == lcmd->fp1)
      return panel1Width;
   else
      return dim.width - panel1Width;
}

bool LCmdFilePanelFrame::onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   return onButton1Down(xpos, ypos, flags);
}

bool LCmdFilePanelFrame::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   lcmd->mainWin.setActive();

   if (isPosOnVSplit(xpos, ypos))
   {
      if (&fpanel == lcmd->fp1)
      {
         int minX = 2;
         int maxX = panelsParent.getWindowSize().width;
         int startX = getWindowSize().width;
         int npos = panelsParent.trackVSplit(minX, maxX, startX, -1);
         if (npos != -1)
            setWidth(npos + 1);
      }
      else
      {
         lcmd->fp1->frameWin.onButton1Down(lcmd->fp1->frameWin.getWindowSize().width-1, ypos, flags);
      }
   }

   else
   if (isPosOnHSplit(xpos, ypos))
   {
      int subclientH = lcmd->mainWin.subClient->getWindowSize().height;
      int cmdcontH = lcmd->mainWin.subClient->cmdcont.getWindowSize().height;

      int minY = 0;
      int maxY = subclientH;
      int startY = cmdcontH - 1;
      int npos = lcmd->mainWin.subClient->trackHSplit(minY, maxY, startY, -1);
      if (npos != -1)
      {
         int cmdLineH = lcmd->mainWin.subClient->cmdcont.cmdline.getWindowSize().height;

         if (npos < cmdLineH)
         {
            lcmd->mainWin.subClient->setFullScreenMode(true);
            lcmd->mainWin.subClient->setCommandLineVisible(false);
            return true;
         }

         if (npos < (2 * cmdLineH))
         {
            lcmd->mainWin.subClient->setFullScreenMode(true);
            lcmd->mainWin.subClient->setCommandLineVisible(true);
            return true;
         }

         if (npos > subclientH)
         {
            lcmd->mainWin.subClient->setFilePanelsVisible(false);
            return true;
         }

         if (lcmd->options.panelsFullScreen)
         {
            // The panels are currently in "Full Screen" mode so we must
            // automatically toggle the mode off so that user can see the
            // size change immediately.
            lcmd->mainWin.subClient->setFullScreenMode(false);
            lcmd->mainWin.subClient->setCommandLineVisible(true);
         }

         setHeight(subclientH - npos - 4);
      }
   }

   return true;
}

bool LCmdFilePanelFrame::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isPosOnVSplit(xpos, ypos))
   {
      GSystem::SetMouseCursorShape(GSystem::MCSID_SIZEWE);
   }
   else
   if (isPosOnHSplit(xpos, ypos))
   {
      GSystem::SetMouseCursorShape(GSystem::MCSID_SIZENS);
   }

   return true; // Yes, we have set the mouse cursor, so don't return false!
}

bool LCmdFilePanelFrame::onPaint ( GGraphics& g, const GRectangle& /*rect*/ )
{
   GRectangle r = getWindowRect();

   // Clear the empty space inside that 3D-frame. In fact, this is not
   // actually neccessary except if both the vertical and the horizontal
   // scrollbar is visible at once, but it doesn't hurt anyway.
   // All this empty space of the frame window is normally completely hidden
   // underneath the child windows.
   g.drawFilledRectangle(r, fpanel.colors.itemBck);
   return true;
}

bool LCmdFilePanelFrame::isPosOnHSplit ( int /*xpos*/, int ypos )
{
   if (ypos > 3)
      return false;
   else
      return true;
}

bool LCmdFilePanelFrame::isPosOnVSplit ( int xpos, int /*ypos*/ )
{
   GRectangle r = getWindowRect();

   if (&fpanel == lcmd->fp1)
   {
      if (xpos < r.width - 3)
         return false;
   }
   else
   {
      if (xpos > 3)
         return false;
   }

   return true;
}

void LCmdFilePanelFrame::setWidth ( int width )
{
   GRectangle rClient = lcmd->mainWin.getWindowRect();
   bool isLeft = fpanel.isLeftPanel();
   int p1width = isLeft ? width : rClient.width - width - 3;
   int p1relWidth = (p1width * 1000) / rClient.width;
   if (p1relWidth < 0)
      p1relWidth = 0;
   else
   if (p1relWidth > 1000)
      p1relWidth = 1000;
   if (p1relWidth != lcmd->options.panel1RelWidth)
   {
      lcmd->options.panel1RelWidth = p1relWidth;

      // Update/recalculate windows due to vertical split bar movement
      panelsParent.layout();

      int idx1 = lcmd->fp1->getCurrentSelectedIndex();
      lcmd->fp1->selectItem(LCmdFilePanel::NavigateHome);
      lcmd->fp1->selectItem(idx1);

      int idx2 = lcmd->fp2->getCurrentSelectedIndex();
      lcmd->fp1->selectItem(LCmdFilePanel::NavigateHome);
      lcmd->fp2->selectItem(idx2);
   }
}

void LCmdFilePanelFrame::setHeight ( int height )
{
   int panelFrameH = lcmd->mainWin.subClient->getWindowSize().height;
   int consoleHeight = panelFrameH - height - 3;
   int consoleRelHeight = (consoleHeight * 1000) / panelFrameH;

   // ---
   if (consoleRelHeight < 0)
      consoleRelHeight = 0;
   else
   if (consoleRelHeight > 950)
   {
      lcmd->mainWin.subClient->setFullScreenMode(true);
      return;
   }

   // ---
   if (consoleRelHeight != lcmd->options.consoleRelHeight)
   {
      lcmd->options.consoleRelHeight = consoleRelHeight;

      // Update/recalculate windows due to vertical split bar movement
      lcmd->mainWin.subClient->recalcAllPartsOfLayout();
   }
}

void LCmdFilePanelFrame::setActive ( bool force )
{
   GWindow::setActive(force);
   fpanel.activatePanel();
}
