/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_FILEITEMCONTAINER
#define __LCMD_FILEITEMCONTAINER

#include "glib/gui/GWorkerThread.h"
#include "glib/util/GArray.h"
#include "lcmd/LCmdFileItem.h"

/**
 * This class implements the background thread that is used to read
 * the filename items from the file system and into our vector in
 * memory.
 *
 * The <i>GComparator</i> interface is implemented as part of the
 * sorting algorithm that sorts the filename items.
 *
 * @author  Leif Erik Larsen
 * @since   2001.03.16
 */
class LCmdFileItemContainer : public GWorkerThread,
                              public GComparator
{
   friend class LCmdFilePanel;

   public:

      /**
       * Interface class that must be implemented by all classes that want 
       * to listen for file item container events.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       * @see     LCmdFileItemContainer#addFileItemContainerListener
       */
      class Listener
      {
      public:
         virtual void allItemsHaveBeenRemoved () = 0;
      };

   private:

      class LCmdFilePanel& fpanel;

      /** The dynamic array of file items/objects. */
      GArray<LCmdFileItem> items;

      /** List of registered listeners. */
      GVector<LCmdFileItemContainer::Listener*> listeners;

      /** Remember which was the recent directory that we loaded items from. */
      GString previousDir;

      enum RunStep { FIND_FIRST, FIND_NEXT, SORT_ITEMS };
      RunStep runStep;
      bool cancelSem;
      int preLoadCounter;
      bool synchPaintPanel;

   public:

      LCmdFileItemContainer ( class LCmdFilePanel& fpanel );

      virtual ~LCmdFileItemContainer ();

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       * @see     #removeFileItemContainerListener
       */
      void addFileItemContainerListener ( LCmdFileItemContainer::Listener* l );

      /**
       * Get a reference to the indexed filename item,
       * with bounds checking.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.17
       * @throws  GArrayIndexOutOfBoundsException if the specified index is
       *                                          outside legal range.
       */
      LCmdFileItem& get ( int index ) const throw(GArrayIndexOutOfBoundsException);

      /**
       * Get a reference to the indexed filename item,
       * without any bounds checking.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.17
       */
      LCmdFileItem& operator[] ( int index ) const;

      /**
       * Get the current number of filename items that are contained
       * in the vector.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.17
       */
      int getCount () const;

      /**
       * Free all loaded file item icons from memory.
       */
      void freeItemIcons ();

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       * @see     #addFileItemContainerListener
       */
      void removeFileItemContainerListener ( LCmdFileItemContainer::Listener* l );

      /**
       * Sort the filename items using current sort options.
       */
      void sort ();

   private:

      // These methods overrides GWorkerThread, in order to implement
      // the filename reader background thread.
      virtual void onWorkerThreadCommand ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& cmd );
      virtual void onWorkerThreadInitDialog ( class GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadUpdateMonitor ( class GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& msgID, GObject* userParam );

      /**
       * This method will be called by the secondary thread when the
       * method fillList() has been called. The main purpose of the
       * secondary thread is to read filename items from the file system
       * and into our vector in memory.
       */
      virtual void runTheWorkerThread ( GWorkerThread& worker );

      /**
       * This method implements <i>GComparator::compare2Objects()</i>.
       * It is used to compare two file items as part of the sorting
       * algorithm that is initialized bu our <i>sortList()</i> method.
       */
      virtual int compare2Objects ( const GObject& obj1, const GObject& obj2 ) const;

      /**
       * Compare the two filename items alphabetically by using the
       * sorting options of the panel.
       *
       * @return Less than zero if item 1 is less than item 2.
       *         Zero if item 1 is equal to item 2.
       *         Larger than zero if item 1 is larger than item 2.
       */
      int compareItems ( const class LCmdFileItem& file1, const class LCmdFileItem& file2 ) const;

      /**
       * Kernel method of {@link #compareItems}.
       */
      int compareItems_ ( const class LCmdFileItem& file1, const class LCmdFileItem& file2, int fldIdx ) const;

      /**
       * Free all allocated memory and every loaded resource of every file
       * list item in specified file listbox panel. This method is called 
       * by the background thread, before starting to load new filename 
       * items.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.02
       */
      void freeItems ();
};

#endif // #ifndef __LCMD_FILEITEMCONTAINER
