/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFileItem.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdOptions.h"

#include "glib/gui/GIcon.h"
#include "glib/util/GStringUtil.h"

GString LCmdFileItem::TxtUpDir;
GString LCmdFileItem::TxtSubDir;

LCmdFileItem::LCmdFileItem ( const LCmdFileItem& src )
             :GFileItem(src),
              marked(src.marked),
              icon(null), // Don't copy the icon!
              iconMustBeDeleted(false),
              briefWidth(src.briefWidth),
              internalFlags(src.internalFlags),
              panel(src.panel),
              physicalFileRep(null)
{
}

LCmdFileItem::LCmdFileItem ( LCmdFilePanel& panel, const GFileItem& src )
             :GFileItem(src),
              marked(false),
              icon(null), // Don't copy the icon!
              iconMustBeDeleted(false),
              briefWidth(0),
              internalFlags(0),
              panel(&panel),
              physicalFileRep(null)
{
}

LCmdFileItem::LCmdFileItem ( GVfs& vfs, const GString& path )
             :GFileItem(vfs, path),
              marked(false),
              icon(null),
              iconMustBeDeleted(false),
              briefWidth(0),
              internalFlags(0),
              panel(null),
              physicalFileRep(null)
{
}

LCmdFileItem::~LCmdFileItem ()
{
   if (iconMustBeDeleted)
      delete icon;
   delete physicalFileRep;
}

GVfs::File* LCmdFileItem::getPhysicalFileRepresentation ()
{
   return physicalFileRep;
}

void LCmdFileItem::setPhysicalFileRepresentation ( GVfs::File* f )
{
   delete physicalFileRep;
   physicalFileRep = f;
}

bool LCmdFileItem::isMarked () const 
{ 
   return marked; 
}

void LCmdFileItem::setMarked ( bool marked )
{
   this->marked = marked;
}

int LCmdFileItem::getBriefWidth () const 
{ 
   return briefWidth; 
}

int LCmdFileItem::getInternalFlags () const 
{ 
   return internalFlags; 
}

void LCmdFileItem::setInternalFlags ( int flags )
{
   internalFlags = flags;
}

bool LCmdFileItem::isDocument ( LCmdFilePanel& /*fp*/ ) const
{
   return false; // TODO: Missing code here!
}

void LCmdFileItem::unloadIcon ()
{
   if (iconMustBeDeleted)
      delete icon;
   icon = null;
   iconMustBeDeleted = false;
}

const GIcon* LCmdFileItem::getIcon ( LCmdFilePanel& fp )
{
   if (icon != null)
      return icon;

   switch (fp.view.showIcon)
   {
      case LCmdFilePanelViewOptions::SHOWICON_SMALL:
      {
         GString path = getFullPath();
         iconMustBeDeleted = true;
         return icon = new GIcon(path, true);
      }

      case LCmdFilePanelViewOptions::SHOWICON_LARGE:
      {
         GString path = getFullPath();
         iconMustBeDeleted = true;
         return icon = new GIcon(path, false);
      }

      case LCmdFilePanelViewOptions::SHOWICON_INTERNAL:
      default:
      {
         // Use a shared internal icon object that is "static" in memory. 
         // That is; we should not destroy it when file item is destroyed.
         iconMustBeDeleted = false;
         if (isDirectory())
            return icon = (isUpDir() ? fp.iconDirUp : fp.iconDirDir);
         else
         if (isProgramFileName())
            return icon = fp.iconFileExe;
         else
         if (isZipOrArchiveFile())
            return icon = fp.iconDirZip;
         else
         if (isDocument(fp))
            return icon = fp.iconFileDoc;
         else
            return icon = fp.iconFileFile;
      }
   }
}

bool LCmdFileItem::doesItemMatchFilter ( LCmdDlgFileFilterProperties& filter ) const
{
   if (!filter.inclDirs && isDirectory())
      return false;
   if (!filter.inclFiles && !isDirectory())
      return false;
   if ((filter.flagArchive != 2) && (filter.flagArchive != int(isArchive())))
      return false;
   if ((filter.flagHidden != 2) && (filter.flagHidden != int(isHidden())))
      return false;
   if ((filter.flagSystem != 2) && (filter.flagSystem != int(isSystem())))
      return false;
   if ((filter.flagReadOnly != 2) && (filter.flagReadOnly != int(isReadOnly())))
      return false;
   if ((filter.useInclFilterStr && !doesItemMatchFilterList(filter.inclFilterStr)))
      return false;
   if ((filter.useExclFilterStr && !doesItemMatchFilterList(filter.exclFilterStr)))
      return false;
   return true;
}

bool LCmdFileItem::doesItemMatchFilterList ( const GString& filterList ) const
{
   GString name = getFileName();
   return GFile::IsFileNameInFilterList(name, filterList);
}

void LCmdFileItem::drawTheIcon ( GGraphics& g, 
                                 int xpos, 
                                 int ypos, 
                                 LCmdFilePanel& fp )
{
   bool issmall = false;
   if (fp.view.showIcon != LCmdFilePanelViewOptions::SHOWICON_INTERNAL)
      issmall = (fp.view.showIcon == LCmdFilePanelViewOptions::SHOWICON_SMALL);
   const GIcon* icon = getIcon(fp);
   if (icon != null)
      g.drawIcon(xpos, ypos, *icon, issmall);
}

void LCmdFileItem::drawTheFileSizeCell ( GGraphics& g, 
                                         const GRectangle& r, 
                                         const GColor& fc,
                                         bool alwaysLeftAlign )
{
   bool dir = isDirectory();
   if (dir && !(internalFlags & FINFO_SHOWSIZE))
   {
      LCmdFileItem::TxtUpDir;
      LCmdFileItem::TxtSubDir;

      if (isUpDir())
      {
         if (LCmdFileItem::TxtUpDir == "")
            LCmdFileItem::TxtUpDir = GStringl("%Txt_FP_UpDir");
         g.drawText(LCmdFileItem::TxtUpDir, r, fc);
      }
      else
      {
         if (LCmdFileItem::TxtSubDir == "")
            LCmdFileItem::TxtSubDir = GStringl("%Txt_FP_SubDir");
         g.drawText(LCmdFileItem::TxtSubDir, r, fc);
      }
   }
   else
   {
      GString txt = GStringUtil::ToByteCountString(fileSize, false, false, false);
      if (alwaysLeftAlign)
         g.drawText(txt, r, fc);
      else
         g.drawText(txt, r, fc, GGraphics::DUMMY_COLOR, GGraphics::RIGHT);
   }
}

bool LCmdFileItem::isInternalFlag ( int flag ) const 
{ 
   return (internalFlags & flag) != 0; 
}

GString LCmdFileItem::getFileAttributesDescrStr () const
{
   GString buff(32);
   if (isDirectory())
      buff.operator+=(GStringl("%TxtFileAttrDir")).operator+=(' ');
   if (isArchive())
      buff.operator+=(GStringl("%TxtFileAttrArchive")).operator+=(' ');
   if (isHidden())
      buff.operator+=(GStringl("%TxtFileAttrHidden")).operator+=(' ');
   if (isSystem())
      buff.operator+=(GStringl("%TxtFileAttrSystem")).operator+=(' ');
   if (isReadOnly())
      buff.operator+=(GStringl("%TxtFileAttrReadOnly")).operator+=(' ');
   if (buff != "") // Cut the trailing space
      buff.removeLastChar();
   return buff;
}

GString LCmdFileItem::getDirectory () const
{
   if (path.containsDirectory())
      return path.getDrive() + path.getDirectory();
   else
   if (panel != null)
      return panel->getCurrentSysDirectory(true);
   else // Should never happen, but in case...
      return GCharacter::ToString(GFile::SlashChar);
}

GString LCmdFileItem::getFullPath () const
{
   if (path.containsDrive() || path.containsDirectory())
      return path.getFullPath(); // The path is already fully qualified.
   if (panel == null) // Should never happen, but in case.
      return getFileName();
   // Use the current directory of the owner panel to form the fully qualified path.
   const GVfs& fs = panel->vfs.peek();
   GString path = fs.getCurrentDirectory(true);
   path += getFileName();
   return path;
}

GString LCmdFileItem::getFullPhysicalPath () const
{
   LCmdFileItem* self = const_cast<LCmdFileItem*>(this);
   const GVfs::File* f = self->getPhysicalFileRepresentation();
   if (f != null)
      return f->getPhysicalPath();
   else
      return getFullPath();
}
