/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgTextSearch.h"
#include "lcmd/LCmdOptions.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"

LCmdDlgTextSearch::LCmdDlgTextSearch ( const GString& iniSectName )
                  :GTextViewer::SearchParams(),
                   textViewer(null),
                   iniSectName(iniSectName),
                   fixedAutoParams(false)
{
}

LCmdDlgTextSearch::~LCmdDlgTextSearch ()
{
}

bool LCmdDlgTextSearch::userEditSearchParams ( GTextViewer& owner )
{
   if (fixedAutoParams)
      return true;
   textViewer = &owner;
   GProgram& prg = GProgram::GetProgram();
   GString darg = prg.executeDialog(&owner, "DlgTextSearch", this);
   textViewer = null;
   return darg == "DLG_OK";
}

void LCmdDlgTextSearch::onSearchStringNotFound ( GTextViewer& owner )
{
   // Do nothing but return if we run in "non-interactive batch mode".
   if (fixedAutoParams)
      return;
   // "Search"
   GStringl title("%Txt_DlgTextSearch_Title"); 
   // "Search string '%s' not found."
   GStringl msg("%Txt_DlgTextSearch_MsgTxtNotFound", GVArgs(searchString));
   owner.showMessageBox(msg, GMessageBox::TYPE_INFO, GString::Empty, title);
}

void LCmdDlgTextSearch::setFixedAutoParams ( const GTextViewer::SearchParams& params )
{
   this->fixedAutoParams = true;
   this->searchString = params.searchString;
   this->searchCaseSen = params.searchCaseSen;
   this->searchWord = params.searchWord;
   this->searchForward = params.searchForward;
}

bool LCmdDlgTextSearch::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: 
      {
         GString str;
         if (textViewer->isSelectedTextSingleLine())
         {
            // Use the selected text as the default text to search for.
            textViewer->getSelectedText(str);
            str.stripTrailingEol();
            int len = str.length();
            if (len <= 0 || len > 256)
            {
               // Use previous search string by default if the linefeed-
               // stripped selection is empty or more than 256 characters.
               str = searchString;
            }
         }
         else
         {
            // Use previous search string by default.
            str = searchString; 
         }

         /* TODO: Convert raw binary characters in str into human "\n\r\t" etc.
         str = GTextResource::MakeTextLiteral(str);
         */
         dlg.setComponentValue("SearchPattern", str);
         return true; 
      }

      case GM_QUERYPROFILE:
      {
         if (iniSectName != "")
         {
            GSectionBag& ini = GProgram::GetProgram().getIniProperties();
            searchCaseSen = ini.getBool(iniSectName, "CaseSen", searchCaseSen);
            searchWord = ini.getBool(iniSectName, "WholeWord", searchWord);
            searchForward = ini.getBool(iniSectName, "Foreward", searchForward);
         }
         dlg.setComponentValue("CaseSen", searchCaseSen);
         dlg.setComponentValue("WholeWord", searchWord);
         dlg.setComponentValue("Foreward", searchForward);
         dlg.setComponentValue("Backward", !searchForward);
         return true;
      }

      case GM_WRITEPROFILE:
      {
         if (iniSectName != "")
         {
            LCmdOptions& opt = LCmdOptions::GetOptions();
            GSectionBag& ini = GProgram::GetProgram().getIniProperties();
            ini.putBool(iniSectName, "CaseSen", searchCaseSen, opt.saveOnExit.otherOptions);
            ini.putBool(iniSectName, "WholeWord", searchWord, opt.saveOnExit.otherOptions);
            ini.putBool(iniSectName, "Foreward", searchForward, opt.saveOnExit.otherOptions);
         }
         return true;
      }

      case GM_CTRLCHANGED:
      {
         GString compID = msg.getParam1String();
         if (compID == "SearchPattern")
         {
            bool empty = dlg.isComponentEmpty("SearchPattern");
            dlg.setComponentEnabled("DLG_OK", !empty);
         }
         return true;
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            searchString = dlg.getComponentValue("SearchPattern");
            /*
            // Convert human "\n\r\t" etc. in str into raw binary characters.
            searchString = GTextResource::ParseTextLiteral(searchString, false);
            */
            searchCaseSen = dlg.getComponentBoolValue("CaseSen");
            searchWord = dlg.getComponentBoolValue("WholeWord");
            searchForward = dlg.getComponentBoolValue("Foreward");
            dlg.dismiss(cmdID);
         }
         else
         if (cmdID == "DLG_CANCEL")
         {
            dlg.dismiss(cmdID);
         }
         return true; 
      }

      default:
           return false;
   }
}
