/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgFileAttributes.h"
#include "lcmd/LCmdFileItem.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"

LCmdDlgFileAttributes::LCmdDlgFileAttributes ( LCmdFilePanel& panel )
                      :LCmdDirProgressBase(panel.vfs.peek(), &panel),
                       archive(2),
                       hidden(2),
                       system(2),
                       readOnly(2)
{
}

LCmdDlgFileAttributes::~LCmdDlgFileAttributes ()
{
}

bool LCmdDlgFileAttributes::updateAttr ( LCmdFileItem& fitem, int set, int attr )
{
   if (set == 0)
   {
      if ((fitem.attr & attr) != 0)
      {
         fitem.attr &= ~attr;
         return true;
      }
      return false;
   }
   else
   if (set == 1)
   {
      if ((fitem.attr & attr) == 0)
      {
         fitem.attr |= attr;
         return true;
      }
      return false;
   }
   else
   if (set == 2)
   {
      // Don't change the flag!
      return false;
   }
   else
   {
      return false;
   }
}

bool LCmdDlgFileAttributes::handleFileItem ( LCmdFileItem* fitem,
                                             int /*fileIndex*/,
                                             bool* skipped,
                                             bool* itemUpdated )
{
   bool anyFlagChanged = false;

   anyFlagChanged |= updateAttr(*fitem, readOnly, GVfs::FAttrReadOnly);
   anyFlagChanged |= updateAttr(*fitem, archive, GVfs::FAttrArchive);
   anyFlagChanged |= updateAttr(*fitem, hidden, GVfs::FAttrHidden);
   anyFlagChanged |= updateAttr(*fitem, system, GVfs::FAttrSystem);

   if (anyFlagChanged)
      statusOK = setPathInfo(*fitem, skipped, itemUpdated);

   return statusOK;
}

void LCmdDlgFileAttributes::execute ()
{
   int curSel = panel->getCurrentSelectedIndex();
   if (panel->markedFilesCount > 0)
   {
      readOnly = 2;
      archive = 2;
      hidden = 2;
      system = 2;
   }
   else
   if (curSel >= 0)
   {
      LCmdFileItem& fitem = panel->items.get(curSel);
      readOnly = fitem.isReadOnly() ? 1 : 0;
      archive = fitem.isArchive() ? 1 : 0;
      hidden = fitem.isHidden() ? 1 : 0;
      system = fitem.isSystem() ? 1 : 0;
   }
   else
      return;

   // Execute the secondary thread as soon as the progress bar has been created.
   startTheDirectoryProcessor("DlgFileAttributes", 
                              "%Txt_DlgFileAttributes_TitleProgBar");
}

bool LCmdDlgFileAttributes::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         LCmdDirProgressBase::handleDialogMessage(msg);
         dlg.setComponentValue("104", readOnly);
         dlg.setComponentValue("105", archive);
         dlg.setComponentValue("106", hidden);
         dlg.setComponentValue("107", system);
         dlg.setComponentFocus("DLG_OK");
         return true;
      }

      case GM_COMMAND:
      {
         LCmdDirProgressBase::handleDialogMessage(msg);
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            readOnly = dlg.getComponentIntValue("104");
            archive = dlg.getComponentIntValue("105");
            hidden = dlg.getComponentIntValue("106");
            system = dlg.getComponentIntValue("107");
            if (readOnly == 2 &&
                archive == 2 &&
                hidden == 2 &&
                system == 2)
            {
               // You have choosen not to change any file attribute. Is this what you want to do?
               GStringl msg("%Txt_DlgFileAttributes_ConfirmNoChange");
               GMessageBox::Answer answ = dlg.showMessageBox(msg, GMessageBox::TYPE_QUESTION, "Ync");
               if (answ != GMessageBox::IDYES)
                  return true;
               bCanceled = true;
            }
            dlg.dismiss(cmdID);
         }
         return true;
      }

      default:
         return LCmdDirProgressBase::handleDialogMessage(msg);
   }
}
