/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgEAEditor.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GTextEntry.h"
#include "glib/gui/GListBox.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/vfs/GExtendedAttributes.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GMath.h"

bool LCmdDlgEAEditor::DlgAdd::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         dlg.setComponentValue("NAME", name, false);
         dlg.setComponentValue("VALUE", value, false);
         return true;
      }

      case GM_COMMAND:
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            name = dlg.getComponentValue("NAME");
            value = dlg.getComponentValue("VALUE");
            if (name == "")
               return true; // EA-name must be != ""
         }
         dlg.dismiss(cmdID);
         return true;
      }

      default:
         return false;
   }
}

GString LCmdDlgEAEditor::DlgAdd::execute ( GWindow* ownerWin )
{
   GProgram& prg = GProgram::GetProgram();
   return prg.executeDialog(ownerWin, "DlgEAEditorEdAdd", this);
}

bool LCmdDlgEAEditor::DlgEdit::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         dlg.setComponentValue("NAME", name, false);
         dlg.setComponentValue("VALUE", value, false);
         GTextEntry& te = dynamic_cast<GTextEntry&>(dlg.getComponentByID("NAME"));
         te.setReadOnly(true);
         return true;
      }

      case GM_COMMAND:
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
            value = dlg.getComponentValue("VALUE");
         dlg.dismiss(cmdID);
         return true;
      }

      default:
         return false;
   }
}

GString LCmdDlgEAEditor::DlgEdit::execute ( GWindow* ownerWin, const GString& name, const GString& value )
{
   this->name = name;
   this->value = value;
   GProgram& prg = GProgram::GetProgram();
   return prg.executeDialog(ownerWin, "DlgEAEditorEdAdd", this);
}

LCmdDlgEAEditor::LCmdDlgEAEditor ( const GString& fname, GExtendedAttributes& eas, bool readOnly )
                :fname(fname),
                 eas(eas),
                 readOnly(readOnly)
{
}

GString LCmdDlgEAEditor::execute ( GWindow* ownerWin )
{
   GProgram& prg = GProgram::GetProgram();
   return prg.executeDialog(ownerWin, "DlgEAEditor", this);
}

bool LCmdDlgEAEditor::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         if (readOnly)
         {
            dlg.setComponentVisible("DLG_OK", false);
            dlg.setComponentVisible("ADD", false);
            dlg.setComponentVisible("EDIT", false);
            dlg.setComponentVisible("DELETE", false);
         }
         else
            dlg.setComponentEnabled("DLG_OK", false);
         dlg.setComponentValue("FNAME", fname, false);
         dlg.setComponentValue("VALUE", "", false);
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("LIST"));
         const int num = eas.getEACount();
         for (int i=0; i<num; i++)
         {
            const GString& name = eas.getIndexedEAName(i);
            list.addItem(name);
         }
         list.setSelectedIndex(0);
         return true;
      }

      case GM_CTRLCHANGED:
      {
         GString id = msg.getParam1String();
         if (id == "LIST")
         {
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("LIST"));
            int idx = list.getSelectedIndex();
            if (idx <= -1)
            {
               dlg.setComponentValue("VALUE", "", false);
               dlg.setComponentEnabled("EDIT", false);
               dlg.setComponentEnabled("DELETE", false);
            }
            else
            {
               GExtendedAttributes::HOLDFEA* ea = eas.getIndexedEA(idx);
               GExtendedAttributes::EAType type = eas.getIndexedEAType(idx);
               if (type == GExtendedAttributes::TYPE_STRING)
               {
                  GString name = list.getItemText(idx);
                  GString value = eas.getEAString(name);
                  dlg.setComponentValue("VALUE", value, false);
                  dlg.setComponentEnabled("EDIT", true);
                  dlg.setComponentEnabled("DELETE", true);
               }
               else
               {
                  GString value(4*4096);
                  char* ptr = ea->aValue;
                  USHORT type = *((USHORT*) ptr);
                  ptr += sizeof(USHORT);
                  // "Type=%04X, Size=%d bytes.\n-----------------"
                  value += GStringl("%Txt_DlgEAEditor_BinEAsHeader", GVArgs(type).add(ea->cbValue));

                  GString hexStr(24);
                  GString byteStr(8);

                  const int len = ea->cbValue - sizeof(USHORT);
                  for (int i=0; i<len; i++, ptr++)
                  {
                     if ((i > 0 && i % 8 == 0) || i == len - 1)
                     {
                        int pos = GMath::Max(0, i-8);
                        value += GString("\n%04X:  %-24s %-8s", GVArgs(pos).add(hexStr).add(byteStr));
                        hexStr.clear();
                        byteStr.clear();
                     }

                     int byteValue = (unsigned char) *ptr;
                     hexStr += GInteger::ToString(byteValue, 16, 2, '0');
                     hexStr += ' ';

                     if (byteValue < 32 || byteValue == 255)
                        byteStr += '.';
                     else
                        byteStr += *ptr;
                  }

                  value += '\n';
                  dlg.setComponentValue("VALUE", value, false);
                  dlg.setComponentEnabled("EDIT", false);
                  dlg.setComponentEnabled("DELETE", true);
               }
            }
         }
         return true;
      }

      case GM_COMMAND:
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            if (readOnly)
               return true;
            dlg.dismiss(cmdID);
         }
         else
         if (cmdID == "DLG_CANCEL")
         {
            dlg.dismiss(cmdID);
         }
         else
         if (cmdID == "ADD")
         {
            if (readOnly)
               return true;

            DlgAdd add;
            while (add.execute(&dlg) == "DLG_OK")
            {
               if (eas.isEADefined(add.name))
               {
                  // "Extended Attribute \"%s\" is already defined. Please use another name."
                  GStringl msg("%Txt_DlgEAEditor_MsgEAAlrDef", GVArgs(add.name));
                  dlg.showMessageBox(msg, GMessageBox::TYPE_WARNING);
                  continue;
               }

               GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("LIST"));
               eas.setEAString(add.name, add.value);
               list.addItem(add.name);
               list.setSelectedIndex(list.getItemCount() - 1);
               dlg.setComponentEnabled("DLG_OK", true);
               break;
            }
         }
         else
         if (cmdID == "EDIT")
         {
            if (readOnly)
               return true;

            DlgEdit edit;
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("LIST"));
            int idx = list.getSelectedIndex();
            GString name = list.getItemText(idx);
            GString value = eas.getEAString(name);
            if (edit.execute(&dlg, name, value) == "DLG_OK")
            {
               if (edit.value != value)
               {
                  eas.setEAString(name, edit.value);
                  dlg.setComponentValue("VALUE", edit.value);
                  dlg.setComponentEnabled("DLG_OK", true);
               }
            }
         }
         else
         if (cmdID == "DELETE")
         {
            if (readOnly)
               return true;
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("LIST"));
            int idx = list.getSelectedIndex();
            if (idx >= 0) // Will always be true, but in case.
            {
               dlg.setComponentValue("VALUE", "", false);
               GString name = eas.getIndexedEAName(idx);
               eas.removeEA(name);
               list.removeItem(idx);
               int count = list.getItemCount();
               list.setSelectedIndex(idx >= count ? count-1 : idx);
               dlg.setComponentEnabled("DLG_OK", true);
            }
         }
         return true;
      }

      default:
         return false;
   }
}

void LCmdDlgEAEditor::Doit ( GWindow* ownerWin, bool readOnly )
{
   LCmdFilePanel& cp = LCmdFilePanel::GetCurrentPanel();
   int curSel = cp.getCurrentSelectedIndex();
   if (curSel <= -1)
      return;

   LCmdFileItem& file = cp.items.get(curSel);
   if (file.isThisDir() || file.isUpDir())
      return;

   if (ownerWin == null)
      ownerWin = &GProgram::GetProgram().getMainWindow();

   GString fname = file.getFileName();
   GExtendedAttributes eas(fname);
   if (eas.getEACount() <= 0 && readOnly)
   {
      GStringl msg("%Txt_DlgEAEditor_MsgNoEAs", GVArgs(fname));
      ownerWin->showMessageBox(msg);
      return;
   }

   LCmdDlgEAEditor eaEditor(fname, eas, readOnly);
   if (eaEditor.execute(ownerWin) == "DLG_OK")
   {
      APIRET rc = eas.writeEAs(fname);
      if (rc != NO_ERROR)
      {
         GStringl msg("%Txt_DlgEAEditor_MsgErrWritten", GVArgs(fname).add(GSystem::GetApiRetString(rc)));
         ownerWin->showMessageBox(msg, GMessageBox::TYPE_ERROR);
      }
   }
}

void LCmdDlgEAEditor::EditEAs ( GWindow* ownerWin )
{
   Doit(ownerWin, false);
}

void LCmdDlgEAEditor::ShowEAs ( GWindow* ownerWin )
{
   Doit(ownerWin, true);
}
