/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgChildProcessSelector.h"
#include "lcmd/LCmdCmdLineEntry.h"
#include "lcmd/LCmdProcess.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GDropList.h"
#include "glib/gui/GListBox.h"
#include "glib/gui/event/GDialogMessage.h"

LCmdDlgChildProcessSelector::LCmdDlgChildProcessSelector ( LCmdCmdLineEntry& entry,
                                                           GSystem::BreakType defaultBt,
                                                           bool breakTree )
                            :entry(entry),
                             defaultBt(defaultBt),
                             indexMap(32),
                             selectedProcess(null),
                             selectedBt(defaultBt),
                             breakTree(breakTree),
                             pureProcessSelectionMode(false)
{
}

LCmdDlgChildProcessSelector::~LCmdDlgChildProcessSelector () 
{
}

LCmdProcessLauncher* LCmdDlgChildProcessSelector::execute ( GWindow& parentWin )
{
   selectedProcess = null;
   GProgram& prg = GProgram::GetProgram();
   GString dismissArg = prg.executeDialog(&parentWin, "DlgChildProcessSelector", this);
   if (dismissArg == "DLG_OK")
      return selectedProcess;
   else
      return null;
}

void LCmdDlgChildProcessSelector::setPureProcessSelectionMode ( bool flag )
{
   pureProcessSelectionMode = flag;
}

GSystem::BreakType LCmdDlgChildProcessSelector::getSelectedBreakType () const
{
   return selectedBt;
}

bool LCmdDlgChildProcessSelector::isBreakTree () const
{
   return breakTree;
}

bool LCmdDlgChildProcessSelector::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: 
      {
         GDropList& met = dynamic_cast<GDropList&>(dlg.getComponentByID("METHOD"));
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         entry.childProcessSelectorDlg = &dlg;

         // Adjust the dialog in order to respect the "pure process selection"
         // mode, if requested.
         if (pureProcessSelectionMode)
         {
            dlg.setComponentVisible("Leader1", false);
            dlg.setComponentVisible("METHOD", false);
            dlg.setComponentVisible("TREE", false);
            GComponentSize size = dlg.getComponentSize("101");
            size.height += 1;
            dlg.setComponentSize("101", size);
         }

         // Update the content of the list box
         indexMap.removeAll();
         int countAdded = 0;
         int num = entry.runningChildren.getCount();
         for (int i=0; i<num; i++)
         {
            LCmdProcessLauncher& pl = entry.runningChildren.get(i);
            if (pl.isRunningChild())
            {
               GTime stime = pl.getStartTime();
               GString text = stime.toString() + "  :  " + pl.getOriginalCommand();
               list.addItem(text, "IDP_FILEEXE");
               indexMap.add(new GInteger(i));
               countAdded++;
            }
         }

         // Select the last executed command.
         if (countAdded >= 1)
            list.setSelectedIndex(countAdded - 1);

         // Initiate the "Process Tree" switch.
         dlg.setComponentValue("TREE", breakTree);

         // Activate the initial/default break type.
         switch (defaultBt)
         {
            case GSystem::BT_CtrlC:
               met.setSelectedIndex(0);
               break;
            case GSystem::BT_CtrlBreak:
               met.setSelectedIndex(1);
               break;
            case GSystem::BT_Exit:
               met.setSelectedIndex(2);
               break;
            case GSystem::BT_Kill:
               met.setSelectedIndex(3);
               break;
         }

         return true; 
      }

      case GM_USER: 
      {
         GString msgID = msg.getParam1String();
         if (msgID == "RemoveItem")
         {
            // The Command Line window send this message when a child process
            // has exited or terminated by any means.
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
            LCmdProcessLauncher* pl = (LCmdProcessLauncher*) msg.getParam2();
            int num = entry.runningChildren.getCount();
            for (int i=0; i<num; i++)
            {
               if (pl == &entry.runningChildren.get(i))
               {
                  int numItems = indexMap.getCount();
                  for (int ii=0; ii<numItems; ii++)
                  {
                     if (indexMap.get(ii).intValue() == i)
                     {
                        // ii is the index of which list box item to remove
                        indexMap.remove(ii);
                        list.removeItem(ii);
                        break;
                     }
                  }
                  break;
               }
            }
         }
         return true; 
      }

      case GM_LISTBOX_DBLCLICK: 
      {
         GString ctrlID = msg.getParam1String();
         if (ctrlID == "101") // User has double clicked on an item
            dlg.sendDialogMessage(GM_COMMAND, "DLG_OK");
         return true; 
      }

      case GM_CTRLCHANGED:
      {
         GString compID = msg.getParam1String();
         if (compID == "METHOD")
         {
            // User has selected another "method".
            // The "Process Tree" toggle is not relevant for 
            // Ctrl+C and Ctrl+Break, so disable it it that case.
            bool enable;
            int idx = msg.getParam2Int(-1);
            switch (idx)
            {
               case 0: // GSystem::BT_CtrlC
               case 1: // GSystem::BT_CtrlBreak
                  enable = false;
                  break;
               case 2: // GSystem::BT_Exit
               case 3: // GSystem::BT_Kill
               default:
                  enable = true;
                  break;
            }
            dlg.setComponentEnabled("TREE", enable);
         }
         return true;
      }

      case GM_COMMAND: 
      {
         GDropList& met = dynamic_cast<GDropList&>(dlg.getComponentByID("METHOD"));
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            // Get the break type.
            int btIdx = met.getSelectedIndex();
            switch (btIdx)
            {
               case 0: selectedBt = GSystem::BT_CtrlC; break;
               case 1: selectedBt = GSystem::BT_CtrlBreak; break;
               case 2: selectedBt = GSystem::BT_Exit; break;
               case 3: selectedBt = GSystem::BT_Kill; break;
               default: return false; // Should never happen, but in case.
            }
            // Get the state of the "Process Tree" toggle.
            breakTree = dlg.getComponentBoolValue("TREE");
            // Get the selected process of which to break.
            LCmdProcessLauncher* pl = null;
            int curIndex = list.getSelectedIndex();
            if (curIndex >= 0)
            {
               int mappedIndex = indexMap.get(curIndex).intValue();
               pl = &entry.runningChildren.get(mappedIndex);
               if (pl != null)
                  if (!pl->isRunningChild())
                     pl = null;
            }
            selectedProcess = pl;
            dlg.dismiss(cmdID);
         }
         else
         if (cmdID == "DLG_CANCEL") // Dismiss without updating the history list.
         {
            dlg.dismiss(cmdID);
         }
         return true; 
      }

      default:
         return false;
   }
}
