/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgAbstractCmdHistory.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/gui/GListBox.h"

LCmdDlgAbstractCmdHistory::LCmdDlgAbstractCmdHistory ( int maxNum, 
                                                       const GString& titleStr )
                          :strings(maxNum),
                           cursor(0),
                           titleStr(titleStr)
{
}

LCmdDlgAbstractCmdHistory::~LCmdDlgAbstractCmdHistory () 
{
}

int LCmdDlgAbstractCmdHistory::getCursor () const 
{ 
   return cursor; 
}

const GString& LCmdDlgAbstractCmdHistory::getIndexedString ( int index ) const 
{ 
   return strings.get(index); 
}

int LCmdDlgAbstractCmdHistory::getStringCount () const 
{ 
   return strings.getCount(); 
}

const GString& LCmdDlgAbstractCmdHistory::getTitleString () const 
{ 
   return titleStr; 
}

void LCmdDlgAbstractCmdHistory::setCursor ( int cursor ) 
{ 
   this->cursor = cursor; 
}

int LCmdDlgAbstractCmdHistory::add ( const GString& str )
{
   cursor = 0;

   // Remove the string from the history if it is already in there.
   for (int i=strings.getCount()-1; i>=0; i--)
      if (str.equalsIgnoreCase(strings[i]))
         strings.remove(i);

   // If the Command History List is already full we must remove the
   // oldest one and pull the remaining items one step up to make room for
   // the new one at the bottom of the list.
   if (strings.getCount() >= strings.getAllocatedSize())
      strings.remove(0);

   strings.add(new GString(str));
   return strings.getCount() - 1;
}

void LCmdDlgAbstractCmdHistory::removeAllStrings () 
{ 
   strings.removeAll(); 
}

void LCmdDlgAbstractCmdHistory::scrollBackward ()
{
   int prevCursor = cursor++;

   if (cursor > strings.getCount())
      cursor = strings.getCount();

   if (prevCursor != cursor)
   {
      int index = strings.getCount() - cursor;
      retrieve(index);
   }
}

void LCmdDlgAbstractCmdHistory::scrollForeward ()
{
   int prevCursor = cursor--;

   if (cursor < 0)
      cursor = 0;

   if (prevCursor != cursor)
   {
      if (cursor <= 0)
      {
         retrieve(-1);
      }
      else
      {
         int index = strings.getCount() - cursor;
         retrieve(index);
      }
   }
}

bool LCmdDlgAbstractCmdHistory::pickFromDialog ()
{
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   GString str = prg.executeDialog(&mwin, "DlgAbstractCmdHistory", this);
   int index = GInteger::ParseInt(str);
   if (index <= -1)
      return false; // CANCEL was selected, so return false.
   else
      return retrieve(index);
}

void LCmdDlgAbstractCmdHistory::updateHistListFromDlg ( GDialogPanel& dlg )
{
   removeAllStrings();
   GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
   const int num = list.getItemCount();
   for (int i=0; i<num; i++)
   {
      GString text = list.getItemText(i);
      add(text);
   }
}

bool LCmdDlgAbstractCmdHistory::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: 
      {
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         dlg.getOwnerFrame().setText(getTitleString());

         int num = getStringCount();
         for (int i=0; i<num; i++)
         {
            const GString& str = getIndexedString(i);
            list.addItem(str);
         }

         list.setSelectedIndex(num-1);
         return true; 
      }

      case GM_CTRLCHANGED: 
      {
         GString ctrlID = msg.getParam1String();
         if (ctrlID == "101") // User has selected an item
         {
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
            int index = list.getSelectedIndex();
            int num = list.getItemCount();
            dlg.setComponentEnabled("104", num > 0 && index >= 0);
            dlg.setComponentEnabled("105", num > 0 && index > 0);
            dlg.setComponentEnabled("106", num > 0 && index >= 0 && index < num-1);
         }
         return true; 
      }

      case GM_LISTBOX_DBLCLICK: 
      {
         GString ctrlID = msg.getParam1String();
         if (ctrlID == "101") // User has double clicked on an item
            dlg.sendDialogMessage(GM_COMMAND, "DLG_OK");
         return true; 
      }

      case GM_COMMAND: 
      {
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            updateHistListFromDlg(dlg);
            int curIndex = list.getSelectedIndex();
            dlg.dismiss(GInteger::ToString(curIndex));
         }
         else
         if (cmdID == "DLG_CANCEL") // Dismiss without updating the history list.
         {
            dlg.dismiss(GInteger::ToString(-1));
         }
         else
         if (cmdID == "104") // Remove
         {
            int index = list.getSelectedIndex();
            if (index >= 0)
            {
               list.removeItem(index);
               int numRemaining = list.getItemCount();
               dlg.setComponentEnabled("104", numRemaining > 0);
               list.setSelectedIndex(index < numRemaining ? index : numRemaining-1);
               list.grabFocus();
            }
         }
         else
         if (cmdID == "105") // Up
         {
            int index = list.getSelectedIndex();
            if (index > 0)
            {
               GString txt = list.getItemText(index);
               list.removeItem(index);
               list.insertItem(index-1, txt);
               list.setSelectedIndex(index-1);
               list.grabFocus();
            }
         }
         else
         if (cmdID == "106") // Down
         {
            int numItems = list.getItemCount();
            int index = list.getSelectedIndex();
            if (index >= 0 && index < numItems - 1)
            {
               GString txt = list.getItemText(index);
               list.removeItem(index);
               list.insertItem(index+1, txt);
               list.setSelectedIndex(index+1);
               list.grabFocus();
            }
         }
         else
         if (cmdID == "102") // Clear
         {
            list.removeAllItems();
         }
         else
         if (cmdID == "103") // Close
         {
            updateHistListFromDlg(dlg);
            dlg.dismiss(GInteger::ToString(-1));
         }
         return true; 
      }

      default:
         return false;
   }
}
