/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDirProgressBase.h"
#include "lcmd/LCmdOptions.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelInfoBar.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/exceptions/GThreadStartException.h"
#include "glib/sys/GSystem.h"

LCmdDirProgressBase::LCmdDirProgressBase ( class GVfs& vfs,
                                           LCmdFilePanel *panel,
                                           bool markedOnly,
                                           bool selectProc,
                                           const char* titleStatus,
                                           const char* startDir,
                                           bool inclFiles,
                                           bool inclDirs,
                                           bool inclHidden,
                                           bool inclSys )
                    :GWorkerThread("DlgDirProgressBase", 0),
                     panel(panel),
                     vfs(vfs),
                     bDisableInclSubdirs(false),
                     bInclSubdirs(false),
                     useFilter(false),
                     bCanceled(false),
                     rereadPanelWhenFinish(false),
                     recalcPanelWhenFinish(false), // Subclass must set this to true if we shall recalc the panel (item widths, column widths, etc.) when progress has finished
                     skipAllByDefault(false),
                     selectProcessingItem(selectProc), // True if we shall select item in panel which is about being processed by the background thread
                     markedItemsOnly(markedOnly), // True if we shall operate on marked items only (except for eventually files in sub-directories)
                     titleStatus(titleStatus),
                     pFilter(&LCmdOptions::GetOptions().fileFilter), // Use the common filter options of the panel by default.
                     startDir(startDir),
                     statusOK(true),
                     inclFiles(inclFiles),
                     inclDirs(inclDirs),
                     inclHidden(inclHidden),
                     inclSys(inclSys)
{
}

LCmdDirProgressBase::~LCmdDirProgressBase ()
{
}

bool LCmdDirProgressBase::setup ()
// Setup the dir-progress object.
// Return true on success (continue) or else false on any error (cancel operation).
{
   if (startDir != null)
      return true;

   if (panel == null)
      return true;

   int curSel = panel->getCurrentSelectedIndex();
   if (panel->markedFilesCount > 0)
   {
      bInclSubdirs = false;

      // Check if any of the marked items are subdirs.
      // If so, then we should enable the "include subdirs" check button, or
      // else we must disable it.
      bDisableInclSubdirs = true;
      for (int i=0; i<panel->items.getCount(); i++)
      {
         LCmdFileItem& fitem = panel->items.get(i);
         if (!fitem.isMarked())
            continue;

         if (fitem.isDirectory())
         {
            bDisableInclSubdirs = false;
            break;
         }
      }
      cDescription = GString("%d file(s)", GVArgs(panel->markedFilesCount));
   }
   else
   if (curSel >= 0)
   {
      LCmdFileItem& fitem = panel->items.get(curSel); // Make this a fast one
      if (fitem.isUpDir() && markedItemsOnly)
         return false; // Can't handle the up-dir ("..")

      cDescription = fitem.getFileName();

      if (!fitem.isDirectory())
      {
         bInclSubdirs = false;
         bDisableInclSubdirs = true;
      }
   }
   else
   {
      return false;
   }

   return true;
}

void LCmdDirProgressBase::runTheWorkerThread ( GWorkerThread& worker )
{
   if (startDir == null)
      processFilePanel();
   else
      recurseSubDir();
}

void LCmdDirProgressBase::onWorkerThreadInitDialog ( GWorkerThread& worker,
                                                     GDialogPanel& monitor )
{
   statusOK = true;
   monitor.setTitleText(progressTitle);
   monitor.setComponentValue("200", titleStatus);
}

void LCmdDirProgressBase::onWorkerThreadUserEvent ( GWorkerThread& worker,
                                                    GDialogPanel& monitor,
                                                    const GString& msgID,
                                                    GObject* /*userParam*/ )
{
   if (msgID == "updtMonitor")
      monitor.setComponentValue("102", curDir);
}

void LCmdDirProgressBase::onWorkerThreadCommand ( GWorkerThread& worker,
                                                  GDialogPanel& monitor,
                                                  const GString& compID )
{
   if (compID == "DLG_CANCEL")
   {
      monitor.setComponentEnabled(compID, false);
      requestStop(false);
      statusOK = false;
   }
}

bool LCmdDirProgressBase::startTheDirectoryProcessor ( const GString& dlgID, 
                                                       const GString& title, 
                                                       int drive )
{
   if (!setup())
      return false;

   if (startDir == null || drive == 0)
      curDir = vfs.getCurrentDirectory(true);
   else
      curDir = GString("%c:\\%s%s", GVArgs(drive + 'A' - 1).add(startDir).add(startDir[0] ? "\\" : ""));

   progressTitle = title;

   GProgram& prg = GProgram::GetProgram();
   GWindow& mainWin = prg.getMainWindow();

   if (dlgID != "")
   {
      // Execute the dialog of where user can specify progress properties.
      GString darg = prg.executeDialog(&mainWin, dlgID, this);
      if (bCanceled)
         return false;
   }

   // Start the background worker thread and show the progress bar dialog.
   try {
      workModal(mainWin);
   } catch (GThreadStartException& /*e*/) {
      GStringl msg("Failed on create secondary worker thread!");
      mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   if (panel != null)
   {
      if (rereadPanelWhenFinish)
         panel->reRead();
      else
      if (recalcPanelWhenFinish)
         panel->getCurrentView().layout();

      // Make sure that all parts of the file panel and the infobar 
      // window will be completely repainted.
      panel->invalidateAll(true);
      panel->infoBar.updateAllFileItemInfoCells();
      panel->onItemSelectionHasChanged(); // Update info-panel if needed.
   }

   return statusOK;
}

bool LCmdDirProgressBase::processFilePanel ()
{
   bool bSkipped;

   if (panel == null)
      return statusOK;

   int curSel = panel->getCurrentSelectedIndex();
   if (panel->markedFilesCount > 0 || !markedItemsOnly)
   {
      const int num = panel->items.getCount();
      for (int i=0; i<num && statusOK; i++)
      {
         LCmdFileItem& fitem = panel->items[i];
         if (markedItemsOnly && !fitem.isMarked())
            continue;

         if (fitem.isUpDir())
            continue; // Ignore the up-dir ("..")

         // Select the item
         if (selectProcessingItem)
            panel->selectItem(i);

         // Update the file attributes as requested by user
         bSkipped = false; // Not skipped until the opposite has been proven
         statusOK = wrapHandleFileItem(&fitem, i, &bSkipped);
         if (!statusOK || bSkipped)
            ;
         else
         if (selectProcessingItem)
         {
            // Unmark the item and update infobar window
            fitem.setMarked(false);
            panel->markedFilesCount -= 1;
            panel->sizeOfMarkedFiles -= fitem.fileSize;
            panel->infoBar.updateAllFileItemInfoCells();
         }
      }
   }
   else
   if (curSel >= 0)
   {
      LCmdFileItem& fitem = panel->items.get(curSel); // Make this a fast one.
      if (!fitem.isUpDir())
      {
         bSkipped = false; // Not skipped until the opposite has been proven.
         int curSel = panel->getCurrentSelectedIndex();
         statusOK = wrapHandleFileItem(&fitem, curSel, &bSkipped);
         if (statusOK)
         {
            // Update the infobar, to show the new properties of that file.
            panel->infoBar.updateAllFileItemInfoCells();
         }
      }
   }

   return statusOK;
}

bool LCmdDirProgressBase::wrapHandleFileItem ( LCmdFileItem* fitem, INT fileIndex, bool *bSkipped )
{
   GString fullFileName = fitem->getFileName();
   curDir += fullFileName;

   // Make a temporary copy of *fitem of which to let the sub-class
   // play with. In case something goes wrong then nothing is touched on the
   // original item at all.
   LCmdFileItem playFile(*fitem);

   bool doOperate = true; // Until the opposite has been proven
   if (useFilter)
      doOperate = fitem->doesItemMatchFilter(*pFilter);
   bool itemUpdated = true;
   if (doOperate)
   {
      sendUserMessageToMonitor("updtMonitor");
      statusOK = handleFileItem(&playFile, fileIndex, bSkipped, &itemUpdated);
   }
   else
   {
      itemUpdated = false;
   }

   if (statusOK && !*bSkipped)
   {
      if (itemUpdated)
      {
         // Copy all properties of the possibly manipulated file item.
         *fitem = playFile;
      }

      if (!isStopRequested() && fitem->isDirectory() && bInclSubdirs)
      {
         vfs.slash(curDir);
         statusOK = recurseSubDir();
         curDir.removeLastChar(); // Remove the terminating "\\"
      }
   }

   // Remove the trailing directory/filename
   for (int index = curDir.length() - 1;
        index > 0 && !GFile::IsSlash(curDir.charAt(index));
        index--, curDir.removeLastChar());

   return statusOK;
}

bool LCmdDirProgressBase::setPathInfo ( const LCmdFileItem& fitem, 
                                        bool* skipped, 
                                        bool* itemUpdated )
{
   for (;;)
   {
      GError rc = vfs.writeAttrAndTimes(null, fitem, curDir);
      if (rc != GError::Ok)
      {
         GMessageBox::Answer answ;
         if (skipAllByDefault)
         {
            answ = GMessageBox::IDSKIP;
         }
         else
         {
            // "Failed on update: '%s'\nMessage from the system: %s"
            GStringl msg("%Txt_FAttr_Error_UpdateAttr", GVArgs(curDir).add(rc.getErrorMessage()));
            answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Ss!rc");
         }
         switch (answ)
         {
            case GMessageBox::IDCANCEL:
                 statusOK = false;
                 return statusOK;

            case GMessageBox::IDSKIPALL:
                 skipAllByDefault = true; // Don't ask again from now!
            case GMessageBox::IDSKIP:
                 *itemUpdated = false;
                 if (!fitem.isDirectory())
                    *skipped = true;
                 break;

            default:
                 continue; // Try again...
         }
      }
      break;
   }

   return true;
}

bool LCmdDirProgressBase::recurseSubDir ()
{
   // Let DosFindFirst() work on a temporary directory name buffer so that
   // the progress bar thread will not show the "\\*.*" at end of directory
   // when showing content of 'curDir' during progress.
   GFile path(curDir, "*.*");
   GString fullPath = path.getFullPath();
   GVfs::List list;
   vfs.fillList(list, fullPath, inclFiles, inclDirs, inclHidden, inclSys);
   for (int i=0, num=list.size();
        i<num && statusOK && !isStopRequested();
        i++)
   {
      const GVfs::List::Item& item = list[i];
      const GString& fname = item.getFName();
      path.setFileName(fname);
      LCmdFileItem file(vfs, path);
      bool bSkipped = false; // Not skipped until the opposite has been proven.
      statusOK = wrapHandleFileItem(&file, -1, &bSkipped);
   }

   return statusOK;
}

bool LCmdDirProgressBase::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         dlg.setComponentValue("103", cDescription);
         dlg.setComponentValue("101", bInclSubdirs);
         dlg.setComponentEnabled("101", !bDisableInclSubdirs);
         dlg.setComponentValue("120", useFilter);
         return true;
      }

      case GM_COMMAND:
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "121") // Filter...
         {
            LCmdOptions& opt = LCmdOptions::GetOptions();
            if (opt.fileFilter.userEditFilterOptions(dlg))
               dlg.setComponentValue("120", true);
         }
         else
         if (cmdID == "DLG_OK")
         {
            bInclSubdirs = dlg.getComponentBoolValue("101");
            useFilter = dlg.getComponentBoolValue("120");
         }
         else
         if (cmdID == "DLG_CANCEL")
         {
            bCanceled = true;
            dlg.dismiss(cmdID);
         }
         return true;
      }

      default:
         return false;
   }
}
