/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmdDlgDirCache.h"
#include "lcmd/LCmd.h"

#include "glib/GProgram.h"
#include "glib/io/GFileInputStream.h"
#include "glib/io/GFileOutputStream.h"

LCmdDirCache::LCmdDirCache ()
             :LCmdDirProgressBase(LCmdDirCache::GetLocalVfs(),
                                  null,
                                  false,
                                  false,
                                  "%TxtDirCache_ScannerNowReading",
                                  GString::Empty,
                                  false, // inclFiles
                                  true,  // inclDirs
                                  true,  // inclHidden
                                  true), // inclSys
              Filter(),
              dirs(512, -3, true)
{
   Filter.inclFiles = false;
   Filter.inclDirs = true;
   Filter.useInclFilterStr = true;
   Filter.inclFilterStr = "*";

   this->bInclSubdirs = true;
   this->useFilter = true;
   this->pFilter = &Filter;

   // Load the initial state of the Dynamic Directory Cache.
   GProgram& prg = GProgram::GetProgram();
   const GString& applName = prg.getApplicationName();
   GString userHomeDir = GSystem::GetUserApplicationDataDirectory(applName);
   this->filePath = userHomeDir + "lcmddircache.ini";

   // Initially read the Directory Cache from the Profile, in case it has
   // been written there by a recent session.
   readFromProfile();
}

LCmdDirCache::~LCmdDirCache ()
{
}

GVfsLocal& LCmdDirCache::GetLocalVfs ()
{
   static GVfsLocal LocalVfs;
   return LocalVfs;
}

void LCmdDirCache::readFromProfile ()
{
   // Load the directories from its file.
   try {
      GString buff(256);
      GFileInputStream file(vfs, filePath, true);
      // Clear the previous cache.
      removeAllCachedDirectories();
      for (;;)
      {
         if (file.readString(buff).length() <= 0)
            break;
         buff.stripTrailingEol();
         Item* item = new Item();
         dirs.put(buff, item);
      }
   } catch (GIOException& /*e*/) {
      // Failed on open the source file.
   }
}

void LCmdDirCache::writeToProfile ( bool /*force*/ )
{
   // Write the directories to its file.
   if (GLog::Filter(GLog::TEST))
      GLog::Log(this, "Write directory cache to file: '%s' (%s:%d)", GVArgs(filePath).add(__FUNCTION__).add(__LINE__));
   try {
      GFileOutputStream file(vfs, filePath, true, true, true);
      const int num = dirs.getCount();
      for (int i=0; i<num; i++)
         file.print(dirs.getKey(i) + "\n");
   } catch (GIOException& e) {
      // Error on open or write file.
      GString stackTrace = e.getStackTrace(e.toString());
      GLog::Log(this, stackTrace);
   }
}

bool LCmdDirCache::handleFileItem ( LCmdFileItem* /*pFile*/, int /*fileIndex*/, bool* /*bSkipped*/, bool* itemUpdated )
{
   *itemUpdated = false;
   addDirectoryToCache(curDir, false);
   return statusOK;
}

GString LCmdDirCache::pickFromList ( const GString& match )
{
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   LCmdDlgDirCache dlg(*this, match);
   GString dir = dlg.execute(&mwin);
   if (dir == "?") 
   {
      // No matching directories in the current cache.
      // "No directories like '#1' in the Dynamic Directory Cache of Larsen Commander at this time."
      GStringl msg("%TxtDirCache_NoSuchDirectory", GVArgs(match));
      mwin.showMessageBox(msg, GMessageBox::TYPE_INFO, "O");
      return GString::Empty;
   }
   else
   {
      return dir;
   }
}

const GString& LCmdDirCache::get ( int index )
{
   return dirs.getKey(index);
}

void LCmdDirCache::addDirectoryToCache ( const GString& dir, bool notifyListeners )
{
   Item* item = new Item();
   dirs.put(dir, item);
   if (notifyListeners)
      lcmd->mainWin.updateStatusbarDirCacheCount();
}

void LCmdDirCache::removeAllCachedDirectories ()
{
   dirs.removeAll();
}

int LCmdDirCache::removeDirectoryFromCache ( const GString& dir, bool inclSubdirs )
{
   int closestIndex = -1;
   int idx = dirs.findEntry(dir, &closestIndex);
   if (idx <= -1)
   {
      if (!inclSubdirs)
         return 0;
      idx = closestIndex;
   }
   GString slashedDir = dir;
   GFile::Slash(slashedDir);
   int count = 0;
   const bool ignoreCase = dirs.isIgnoreCase();
   while (idx >= 0 && idx < dirs.getCount())
   {
      const GString& key = dirs.getKey(idx);
      if (key.equalsString(dir, ignoreCase) ||
          key.beginsWith(slashedDir, ignoreCase))
      {
         dirs.removeIndexedItem(idx);
         count++;
         if (!inclSubdirs)
            break;
      }
      else
      {
         break;
      }
   }
   return count;
}

int LCmdDirCache::renameDirectoryInCache ( const GString& dir, const GString& newDir )
{
   int closestIndex = -1;
   int idx = dirs.findEntry(dir, &closestIndex);
   if (idx <= -1)
      idx = closestIndex;
   GString slashedDir = dir;
   GFile::Slash(slashedDir);
   const bool ignoreCase = dirs.isIgnoreCase();
   GKeyBag<Item> renamedDirs(256, -3, ignoreCase);
   while (idx >= 0 && idx < dirs.getCount())
   {
      const GString& key = dirs.getKey(idx);
      if (key.equalsString(dir, ignoreCase))
      {
         renamedDirs.put(newDir, &dirs.getIndexedItem(idx), false);
         dirs.removeIndexedItem(idx, false);
      }
      else
      if (key.beginsWith(slashedDir, ignoreCase))
      {
         GString newKey = newDir;
         GFile::Slash(newKey); 
         newKey += key.substring(slashedDir.length());
         renamedDirs.put(newKey, &dirs.getIndexedItem(idx), false);
         dirs.removeIndexedItem(idx, false);
      }
      else
      {
         break;
      }
   }
   for (int i=renamedDirs.getCount()-1; i>=0; i--)
   {
      const GString& key = renamedDirs.getKey(i);
      Item* item = &renamedDirs.getIndexedItem(i);
      dirs.put(key, item, true);
   }
   return renamedDirs.getCount();
}
