/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DELETE
#define __LCMD_DELETE

#include "glib/gui/GWorkerThread.h"
#include "glib/vfs/GVfs.h"

/**
 * This class implements the file deletion command, which operation will
 * perform by a background thread so that the foreground GUI is not
 * hogged while the deletion of possibly lots of thousands of files
 * performs.
 *
 * @author  Leif Erik Larsen
 * @since   2000.09.19
 */
class LCmdDelete : public GWorkerThread
{
   private:

      /** The file panel of where to delete from. */
      class LCmdFilePanel& panel;

      /** The Virtual File System of where to delete from. This is probably always the same VFS as of the File Panel. */
      class GVfs& vfs;

      /** The handle used for all file- and directory deletion operations. */
      GVfs::DeletionHandle hdel;

      /** Index of which item was selected upon cmdDelete() entry. */
      int prevSelected;

      /** True as long as there is no error. */
      volatile bool statusOK;

      /** Full path of current file that is about to be deleted. */
      GString curFile;

      /** Answer of last confirm question shown with {@link #confirmDelete}. */
      GMessageBox::Answer lastConfirmAnswer;

      /** True if delete non-empty dir without confirm. */
      bool answDelTree;

      /** True if the next top-level item is to be confirmed by user. */
      bool confirmNextItem;

      /** True if we shall delete to the Recycle Bin or Trash Can. */
      bool trashCan;

      /** Number of files/directories to delete totally, counted recursively. */
      int countItemsToDeleteTotally;

      /** Count how many files/directories has been deleted. */
      int countDeletedItems;

      /** The previous "estimated remaining number of seconds". Used by Eta. */
      double prevTimeSecRemaining;

      /** The GMessageBox-tags used for all deletion confirmation message boxes. */
      const GString ConfirmDelTags;

   public:

      LCmdDelete ( class LCmdFilePanel& panel );
      virtual ~LCmdDelete ();

   public:

      /**
       * Handle the menu command "cmdDeleteFile", which will delete the current
       * selected/marked files and directories in the specified file panel.
       *
       * @return  True if the deletion thread has been executed, or else false.
       */
      static bool CmdDelete ( class LCmdFilePanel& panel );

   private:

      /**
       * Ask the user to confirm the operation described in the 
       * specified question string. The user will be presented a dialog
       * box where he can answer either 1) "Yes", 2) "No", 3) "Cancel",
       * or 4) "Yes to all".
       *
       * The ID of the answer actually given by the user will be stored 
       * in the instance variable {@link #lastConfirmAnswer}, but only 
       * if this method returns true. Anyway, the actual answer will 
       * be returned in the <i>answ</i> parameter (if it is != null).
       *
       * This method is to be called by the GUI thread only.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.03
       * @return  True if user answer either 1) "Yes" or 2) "Yes to all".
       *          Else we will return false.
       */
      bool confirmDelete ( const GString& question, 
                           GMessageBox::Answer* answ = null );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.11.05
       * @param   attr The system dependent attributes of the current file.
       *               Needed in order to know if we shall confirm the delete
       *               operation with respect to the confirmation options 
       *               that is currently specified by the user.
       * @param   file The top level file item for which we are called, or 
       *               null if we are not called from the top level 
       *               method ({@link #deleteTopLevelItem}). We need this
       *               in order for the "Delete to Trash Can/Recycle Bin"
       *               operation to work as expected.
       * @param   calledByTopLevel True if we are called directly by 
       *               {@link #deleteTopLevelItem}, or else false.
       */
      bool deleteCurFile ( int attr, class LCmdFileItem* file, bool calledByTopLevel = false );
      
      /**
       * @author  Leif Erik Larsen
       * @since   2004.11.05
       * @param   attr             See {@link #deleteCurFile}.
       * @param   file             See {@link #deleteCurFile}.
       * @param   calledByTopLevel See {@link #deleteCurFile}.
       */
      bool deleteCurDir ( int attr, class LCmdFileItem* file, bool calledByTopLevel = false );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.08.06
       */
      bool deleteTopLevelItem ( class LCmdFileItem& file );

      /**
       * This is the main entry point of the background thread that will
       * do the actual task of deleting files.
       *
       * If some filenames are marked then delete all those filenames that are
       * marked in the actual file panel. Else delete just that filename that
       * is currently selected.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.19
       */
      virtual void runTheWorkerThread ( class GWorkerThread& worker );

      virtual void onWorkerThreadCommand ( class GWorkerThread& worker, 
                                           class GDialogPanel& monitor, 
                                           const GString& cmdID );

      virtual void onWorkerThreadInitDialog ( GWorkerThread& worker, 
                                              class GDialogPanel& monitor );

      virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker, 
                                             class GDialogPanel& monitor, 
                                             const GString& msgID, 
                                             GObject* userParam );
};

#endif // #ifndef __LCMD_DELETE
