/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdCopy.h"
#include "lcmd/LCmdDlgCalcDirSize.h"
#include "lcmd/LCmdOptions.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelInfoBar.h"
#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmd.h"

#include "glib/GProgram.h"
#include "glib/vfs/GExtendedAttributes.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/GProgressBar.h"
#include "glib/gui/GLinePlotter.h"
#include "glib/io/GFileNotFoundException.h"
#include "glib/primitives/GLong.h"
#include "glib/exceptions/GThreadStartException.h"
#include "glib/util/GStringUtil.h"
#include "glib/util/GMath.h"
#include "glib/sys/GSystem.h"

#define COPY_FILE_DEBUG GLog::TEST

LCmdCopy::SpecifyDestDir::SpecifyDestDir ( bool move,
                                           LCmdFilePanel& srcPanel,
                                           const GString& destPath,
                                           GArray<LCmdCopyFileItem>& items )
                         :move(move),
                          sourcePanel(srcPanel),
                          destPath(destPath),
                          items(items),
                          iError(LCmdCopy::ERR_NONE)
{
}

const char* LCmdCopy::ILLEGAL_FS_CHARS = "\\\"\n\r\t:*?/";
const char* LCmdCopy::REMOVE_LONGNAME_EAS[] = { EA_NAME_LONGNAME, "LCMD_SHORTNAME", null };

LCmdCopy::LCmdCopy ( GVfs& vfsSrc, GVfs& vfsDst, const GString& monDlgID )
         :GWorkerThread(monDlgID),
          vfsSrc(vfsSrc),
          vfsDst(vfsDst),
          hdelSrc(0),
          copyBuffer(null)
{
   clear();
}

LCmdCopy::~LCmdCopy ()
{
   clear();
}

void LCmdCopy::clear ()
{
   delete [] copyBuffer;
   if (hdelSrc != 0)
   {
      vfsSrc.closeDeletion(hdelSrc);
      hdelSrc = 0;
   }
   bMove = false;
   moveAtomicOnly = false;
   removeEAName = false;
   renameOnly = false;
   fpanel = null;
   fpanelIsSrc = false;
   dontCopyWriteProtect = false;
   items = null;
   totalBytesToCopy = 0;
   fileCount = 0;
   noConfirmations = false;
   showStatistics = false;
   waitOnFinish = false;
   waitingForFinishCommand = false;
   bReplaceAll = false;
   bReplaceAllReadOnly = false;
   bMergeAllDirs = false;
   bRemoveAllReadOnly = false;
   bIgnoreAllEAs = false;
   confirmAutoFilenameShorten = true;
   bHasFinished = false;
   timerCounter = 0;
   iCurItemIdx = 0;
   iPrevItemIdx = 0;
   maximumBuffSize = 0;
   activeBufferSize = 0;
   allocatedBufferSize = 0;
   copyBuffer = null;
   bytesCopied = 0;
   prevBytesCopied = 0;
   countFilesCopied = 0;
   countFilesSkipped = 0;
   prevTimeSinceStartMillis = 0;
   bpsPeak = 0;
   iError = ERR_NONE;
}

bool LCmdCopy::reInit ( GArray<LCmdCopyFileItem>* items,
                        LCmdFilePanel& fpanel,
                        bool fpanelIsSrc,
                        bool move,
                        bool renameOnly,
                        bool removeEAName )
{
   LCmdOptions& opt = LCmdOptions::GetOptions();
   clear();
   this->bMove = move;
   this->moveAtomicOnly = false;
   this->removeEAName = removeEAName;
   this->renameOnly = renameOnly;
   this->fpanel = &fpanel;
   this->fpanelIsSrc = fpanelIsSrc;
   this->items = items;
   this->showStatistics = opt.fileCopy.showStatistics;
   this->waitOnFinish = opt.fileCopy.waitOnFinish;
   this->maximumBuffSize = GMath::Max(opt.fileCopy.maxBuffSize, 1) * 1024;
   this->activeBufferSize = 1024;
   this->allocatedBufferSize = 4096;
   this->copyBuffer = new char[this->allocatedBufferSize];
   if (move)
      this->hdelSrc = vfsSrc.openDeletion();

   // Query drive information about the source drive.
   // This is for two reasons. First, we need to know the File System of
   // the drive in case it is a CD-ROM drive. In that case we will not copy
   // the Read-Only attribute of each file. Second, in order to support the
   // automatic save/restore of long filenames when copy/move to/from
   // FAT/HPFS drives.
   APIRET rc;
   LCmdCopyFileItem& firstItem = items->get(0);
   const GString& firstItemSrcPath = firstItem.getSourcePath();
   int firstItemSrcDrive = 0; // "Current drive" by default.
   if (GFile::ContainsDrive(firstItemSrcPath))
       firstItemSrcDrive = GCharacter::ToUpperCase(firstItemSrcPath[0]) - 'A' + 1;
   if ((rc = srcDriveInfo.update(firstItemSrcDrive)) != NO_ERROR)
   {
      GStringl msg("%Txt_Copy_Error_QueryDriveInfo", GVArgs(firstItemSrcDrive + 'A' - 1).add(GSystem::GetApiRetString(rc)));
      GWindow& mainWin = GProgram::GetProgram().getMainWindow();
      mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   if (srcDriveInfo.fsName == "CDFS")
      dontCopyWriteProtect = true;

   // Get the information of the destination drive as well.
   const GString& firstItemDstPath = firstItem.getDestPath();
   int firstItemDstDrive = 0; // "Current drive" by default.
   if (GFile::ContainsDrive(firstItemDstPath))
      firstItemDstDrive = GCharacter::ToUpperCase(firstItemDstPath[0]) - 'A' + 1;
   if ((rc = dstDriveInfo.update(firstItemDstDrive)) != NO_ERROR)
   {
      GWindow& mainWin = GProgram::GetProgram().getMainWindow();
      GStringl msg("%Txt_Copy_Error_QueryDriveInfo", GVArgs(firstItemDstDrive + 'A' - 1).add(GSystem::GetApiRetString(rc)));
      mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   // ---
   if (renameOnly && bMove) // Don't calc dirsize when operation is "rename only".
   {
      fileCount = 1; // One single file or directory only.
      moveAtomicOnly = true;
   }
   else
   if (bMove && (firstItemSrcDrive == firstItemDstDrive))
   {
      // No need to calculate total file count and size if we are about 
      // to move file(s) from a drive that is the same drive as the 
      // destination. In that case the move is more like a rename anyway, 
      // from the operating system point of view.
      fileCount = items->getCount();
      moveAtomicOnly = true;
   }
   else
   {
      LCmdFilePanel* fp = fpanelIsSrc ? &fpanel : null;
      if (!LCmdDlgCalcDirSize::CalcSizeOfAllItems(fp, vfsSrc, *items, &totalBytesToCopy, &fileCount))
         return false;
      moveAtomicOnly = false;
      if (opt.fileCopy.warnIfNotEnoughTargetSpace)
      {
         longlong driveFree = vfsDst.getFreeDriveSpace();
         if (totalBytesToCopy > driveFree)
         {
            GWindow& mainWin = GProgram::GetProgram().getMainWindow();
            GString driveFreeStr = GStringUtil::ToByteCountString(driveFree, false, false, false);
            GString totalBytesToCopyStr = GStringUtil::ToByteCountString(totalBytesToCopy, false, false, false);
            GStringl msg("%Txt_Copy_Confirm_NotEnoughtargetSpace", GVArgs(driveFreeStr).add(totalBytesToCopyStr));
            if (mainWin.showMessageBox(msg, GMessageBox::TYPE_WARNING, "Ync") != GMessageBox::IDYES)
               return false;
         }
      }
   }

   // Give total size one additional byte per file. This is useful in case we
   // are about being copying a bounch of zero-size files and/or directories.
   // This makes the progress bar useful even if all files are zero-sized.
   totalBytesToCopy += fileCount;
   return true;
}

GError LCmdCopy::copyFile_ ( GVfs::FileHandle srcFile,
                             GVfs::FileHandle dstFile,
                             const GString& originalFName,
                             const GString& shortenedFName,
                             const GString& dstFilePath,
                             bool yesCopyEAsIfAny )
{
   LCmdOptions& opt = LCmdOptions::GetOptions();

   // ---
   ulonglong startTime = 0;
   if (GLog::Filter(COPY_FILE_DEBUG))
      startTime = GSystem::CurrentTimeMillis();

   // Load the standard attributes of the source file
   if (GLog::Filter(COPY_FILE_DEBUG))
      GLog::Log(this, "Loading attributes of the source file");
   GFileItem fitem;
   GError rc = vfsSrc.loadFileInfo(srcFile, fitem);
   if (rc != GError::Ok)
   {
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Error loading attributes of the source file. Message is: %s", GVArgs(rc.getErrorMessage()));
      return rc;
   }

   // Switch off the Write Protection flag (if any) in case we are requested to
   if (dontCopyWriteProtect)
   {
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Switch off the Write Protection flag");
      fitem.setReadOnlyFlag(false);
   }

   // Copy any extended attributes as well.
   // If the file name was automatically shortened (in case of copy/move
   // a HPFS filename to a FAT drive) then store the original filename
   // as the .LONGNAME EA of the destination file.
   if (yesCopyEAsIfAny)
   {
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Loading EA's to be copied");
      GExtendedAttributes eas = GExtendedAttributes(GSysFileHandle(srcFile));
      if (shortenedFName != "")
      {
         eas.setEAString("LCMD_SHORTNAME", shortenedFName);
         eas.setEAString(EA_NAME_LONGNAME, originalFName);
      }
      if (eas.getEACount() > 0) // Attempt to write EA's only if there are any at all.
      {
         if (GLog::Filter(COPY_FILE_DEBUG))
            GLog::Log(this, "Writing EA's");
         rc = eas.writeEAs(GSysFileHandle(dstFile));
         if (rc != GError::Ok)
            return rc;
      }
   }

   // Allocate as many bytes to the destination file as we know we are
   // going to need. This is the same as the size of the source file.
   // Doing this helps the underlying File System to prevent fragmentation.
   // Another positive side effect is that the file copy/move progress
   // usually will run dramatically faster. The negative side effect can
   // be some system UI "hoggs". At least on some systems, and espessially
   // when copying or moving very large files.
   GString dstDrive = GFile::GetDrive(dstFilePath);
   if (dstDrive == "")
      dstDrive = GFile::GetDrive(vfsDst.getCurrentDirectory(true));
   if (opt.fileCopy.isPreallocSpaceForDrive(dstDrive))
   {
      if (fitem.fileSize <= ulonglong(opt.fileCopy.maxPreallocateSize) &&
          fitem.fileSize <= ulonglong(GLong::MAX_VALUE))
      {
         if (GLog::Filter(COPY_FILE_DEBUG))
            GLog::Log(this, "Preallocating filespace on drive \"%s\"; fileSize=%s", GVArgs(dstDrive).add(fitem.fileSize));
         GError err = vfsDst.setFileSize(dstFile, fitem.fileSize);
         if (err != GError::Ok)
         {
            // This error is probably not critical by any means, so just 
            // ignore it and continue copy/move without preallocating.
            // Print a log statement, however.
            if (GLog::Filter(GLog::PROD))
               GLog::Log(this, "Preallocating filespace failed: %s", GVArgs(err.getErrorMessage()));
         }
      }
      else
      {
         if (GLog::Filter(COPY_FILE_DEBUG))
            GLog::Log(this, "Skip preallocating filespace, because fileSize is too large; fileSize=%s", GVArgs(fitem.fileSize));
      }
   }

   // Loop to perform the copy of the actual file bytes.
   if (GLog::Filter(COPY_FILE_DEBUG))
      GLog::Log(this, "Begin copy. fileSize=%s, yesCopyEAsIfAny=%s, maximumBuffSize=%d, originalFName=\"%s\", shortenedFName=\"%s\"", GVArgs(fitem.fileSize).add(yesCopyEAsIfAny).add(maximumBuffSize).add(originalFName).add(shortenedFName));
   ulonglong prevTime = GSystem::CurrentTimeMillis();
   ulonglong prevTimeUpdtMon = prevTime;
   for (int i=0;;i++)
   {
      if (iError != ERR_NONE) // In case user has pressed "cancel"
         return GError::Ok;

      int read = 0;
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Block ##%d, read; activeBufferSize=%d", GVArgs(i).add(activeBufferSize));
      rc = vfsSrc.readFromFile(srcFile, copyBuffer, activeBufferSize, &read);
      if (rc != GError::Ok)
         return rc;

      if (read == 0)
         break;

      if (iError != ERR_NONE) // In case user has pressed "cancel"
         return GError::Ok;

      int written;
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Block ##%d, write; read=%d", GVArgs(i).add(read));
      rc = vfsDst.writeToFile(dstFile, copyBuffer, read, &written);
      if (rc != GError::Ok)
         return rc;
      else
      if (written != read)
         return ERROR_DISK_FULL;

      bytesCopied += written;

      // Update the dynamic buffering size if needed, attempting to
      // prevent each read/write block operation from taking more than
      // a specific number of milliseconds. This is in order to get a
      // smooth update frequency in the progress window components and
      // progress bars.
      const int updateFreqMillis = 4000; // Number of milliseconds between each update of progress dialog.
      ulonglong curTime = GSystem::CurrentTimeMillis();
      int blockTimeMillis = int(curTime - prevTime);
      int idealBuffSize = int((written * updateFreqMillis) / GMath::Max(1, blockTimeMillis));
      activeBufferSize = GMath::Clip(idealBuffSize, 16, maximumBuffSize);
      if (activeBufferSize > 4096)
         activeBufferSize -= activeBufferSize % 4096;
      if (activeBufferSize > allocatedBufferSize)
      {
         delete [] copyBuffer;
         allocatedBufferSize = activeBufferSize;
         copyBuffer = new char[allocatedBufferSize];
      }

      // Update the progress dialog, but no more than four times per second.
      if (curTime - prevTimeUpdtMon >= 250)
      {
         sendUserMessageToMonitor("UpdtMon");
         prevTimeUpdtMon = curTime;
      }

      // GProgram::GetProgram().printF("Block ##%d, blockTimeMillis=%d, idealBuffSize=%d, activeBufferSize=%d", GVArgs(i).add(blockTimeMillis).add(idealBuffSize).add(activeBufferSize));

      // Prepare for another read/write block.
      prevTime = curTime;
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Block ##%d, blockTimeMillis=%d, idealBuffSize=%d, activeBufferSize=%d", GVArgs(i).add(blockTimeMillis).add(idealBuffSize).add(activeBufferSize));
   }

   // Make the destination file have the same attributes
   // and time stamps as of the source file.
   if (GLog::Filter(COPY_FILE_DEBUG))
      GLog::Log(this, "Set file attributes and time stamp.");
   rc = vfsDst.writeAttrAndTimes(dstFile, fitem, dstFilePath);
   if (rc != GError::Ok)
      return rc;

   // ---
   if (GLog::Filter(COPY_FILE_DEBUG))
   {
      ulonglong endTime = GSystem::CurrentTimeMillis();
      ulonglong usedTime = endTime - startTime;
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "Finished copying file \"%s\", without any error. It took %d ms.", GVArgs(originalFName).add(usedTime));
   }

   return NO_ERROR;
}

LCmdCopy::ErrType LCmdCopy::copyFile ( longlong srcSize,
                                       const GFile& sourcePath,
                                       GFile& destPath,
                                       bool* pbSkipped,
                                       bool isLevel1 )
{
   LCmdOptions& opt = LCmdOptions::GetOptions();

   GError err;
   GVfs::FileHandle srcFile;
   GVfs::FileHandle dstFile;

   *pbSkipped = true; // Until the opposite has been proven

   curSourcePath = sourcePath.getFullPath();
   curDestPath = destPath.getFullPath();
   sendUserMessageToMonitor("UpdtMon");

   // Open the source file for sequential reading only.
   // Loop to give the user a chance to retry in case of any error.
   for (;;)
   {
      srcFile = vfsSrc.openFile(curSourcePath, &err,
                                GVfs::Mode_ReadOnly,
                                GVfs::Create_Never,
                                GVfs::Share_DenyWrite,
                                GVfs::OF_FLAG_SEQUENTIAL_ACCESS);
      if (err != GError::Ok)
      {
         GStringl msg("%Txt_Copy_Error_OpenSource", GVArgs(curSourcePath).add(err.getErrorMessage()));
         if (GLog::Filter(COPY_FILE_DEBUG))
            GLog::Log(this, "LCmdCopy::copyFile(): error: %s", GVArgs(msg));
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
         switch (answ)
         {
            case GMessageBox::IDSKIP:
                 countFilesCopied += 1;
                 countFilesSkipped += 1;
                 bytesCopied += srcSize + 1;
                 return ERR_NONE;
            case GMessageBox::IDABORT:
                 return ERR_IOERROR;
            case GMessageBox::IDRETRY:
            default:
                 continue;
         }
      }
      else
         break;
   }

   // Keep a copy of the original full filename, in case we need to put it
   // into the EA's of the file, in case destination file system has support
   // for old style (short) FAT filenames only.
   GString originalFileName = sourcePath.getFileName();
   GString shortenedFName = "";

   // Get a copy of the EA set of the source file.
   // This is to test the .LONGNAME attribute to restore the long
   // filename for instance when copy/move a file back from a FAT drive
   // to a HPFS drive. However, we will automatically restore the .LONGNAME
   // only if the current filename has not changed since it was written
   // by Larsen Commander. That is if LCMD_SHORTNAME is the same as the
   // current filename.
   bool tryingToRestoreLongFName = false;
   GExtendedAttributes eas(sourcePath);
   GString longName = eas.getEAString(EA_NAME_LONGNAME);
   GString shortName = eas.getEAString("LCMD_SHORTNAME");
   if (longName != "" &&
       shortName.equalsIgnoreCase(originalFileName) &&
       srcDriveInfo.drive != dstDriveInfo.drive &&
       sourcePath.getFileName().equalsIgnoreCase(destPath.getFileName()))
   {
      // Replace all special characters from the .LONGNAME EA and replace
      // them with an exclamation (just as the WPS does).
      longName.replaceAll(ILLEGAL_FS_CHARS, '!');
      destPath.setFileName(longName);
      tryingToRestoreLongFName = true;
   }

   // Open the destination file for sequential writing only, but test if the
   // destination file already exist or not, first. Also check if the
   // destination file is read-only, and if so, ask the user to confirm.
   // Loop to give the user a chance to retry in case of any error.
   int countAttempts = 1;
   for (;;)
   {
      // Test if the destination file already exist and if it does then
      // make sure that the user does agree to overwrite it.
      if (!bReplaceAll &&
          !noConfirmations &&
          opt.confirm.overwriteFileOrDir &&
          vfsDst.exist(destPath))
      {
         GStringl msg("%Txt_Copy_Confirm_DestAlreadyExist", GVArgs(destPath));
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, "Yy!nc");
         switch (answ)
         {
            case GMessageBox::IDYES:
                 break;

            case GMessageBox::IDYESTOALL:
                 bReplaceAll = true;
                 break;

            case GMessageBox::IDCANCEL:
                 vfsSrc.closeFile(srcFile);
                 return ERR_CANCELED;

            case GMessageBox::IDNO:
            default:
                 countFilesCopied += 1;
                 bytesCopied += srcSize + 1;
                 vfsSrc.closeFile(srcFile);
                 return ERR_NONE;
         }
      }

      // Test if destination file is write protected or is marked as a system,
      // hidden or read only file. In that case confirm with the user if it
      // still is ok to overwrite it.
      int dstAttrs = vfsDst.getFileAttributes(destPath, &err);
      if (err == GError::Ok && ((dstAttrs & GVfs::FAttrHidden) != 0 ||
                                (dstAttrs & GVfs::FAttrSystem) != 0 ||
                                (dstAttrs & GVfs::FAttrReadOnly) != 0))
      {
         if (!bReplaceAllReadOnly && !noConfirmations)
         {
            GStringl msg("%Txt_Copy_Confirm_OverwriteSysHidRdoFile", GVArgs(destPath));
            GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, "Yy!sc");
            switch (answ)
            {
               case GMessageBox::IDYES:
                    break;

               case GMessageBox::IDYESTOALL:
                    bReplaceAllReadOnly = true;
                    break;

               case GMessageBox::IDCANCEL:
                    vfsSrc.closeFile(srcFile);
                    return ERR_CANCELED;

               case GMessageBox::IDSKIP:
               default:
                    countFilesCopied += 1;
                    countFilesSkipped += 1;
                    bytesCopied += srcSize + 1;
                    vfsSrc.closeFile(srcFile);
                    return ERR_NONE;
            }
         }

         // Remove the read-only attribute, so that we will be able to replace
         // the write protected destination file. This operation will probably
         // never fail but if it does then it isn't a critical problem because
         // then we will fail when we try to create the destination file and the
         // user will see a corresponding error message anyway.
         dstAttrs = GVfs::FAttrNormal;
         vfsDst.setFileAttributes(destPath, dstAttrs);
      }

      // Open the destination file.
      dstFile = vfsDst.openFile(destPath, &err,
                                GVfs::Mode_ReadWrite,
                                GVfs::Create_Always,
                                GVfs::Share_DenyWrite,
                                GVfs::OF_FLAG_SEQUENTIAL_ACCESS);

      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "LCmdCopy::copyFile(): renameOnly=%d", GVArgs(renameOnly));

      if (countAttempts == 1 && !renameOnly &&
         (srcDriveInfo.drive != dstDriveInfo.drive) &&
         (err == GError::FilenameExceedRange || err == GError::InvalidName))
      {
         if (tryingToRestoreLongFName)
         {
            destPath.setFileName(originalFileName);
            tryingToRestoreLongFName = false;
            countAttempts = 0; // Adjust for the next += 1
         }
         else
         {
            if (confirmAutoFilenameShorten && !noConfirmations)
            {
               GStringl msg("%Txt_Copy_Confirm_AutoShortenFName", GVArgs(destPath.getFileName()));
               GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_WARNING, "Yy!sa");
               switch (answ)
               {
                  case GMessageBox::IDSKIP:
                     vfsSrc.closeFile(srcFile);
                     countFilesCopied += 1;
                     countFilesSkipped += 1;
                     bytesCopied += srcSize + 1;
                     return ERR_NONE;

                  case GMessageBox::IDABORT:
                     vfsSrc.closeFile(srcFile);
                     return ERR_IOERROR;

                  case GMessageBox::IDYESTOALL:
                     confirmAutoFilenameShorten = false;
                     break;

                  case GMessageBox::IDYES:
                  default:
                     break;
               }
            }

            GString fullPath = destPath.getFullPath();
            GString shortenedPath = vfsDst.getShortenedDirOrPath(fullPath);
            destPath.setFullPath(shortenedPath);
            shortenedFName = destPath.getFileName();
         }
         countAttempts += 1;
         continue;
      }

      if (err == GError::Ok)
      {
         break;
      }
      else
      {
         GStringl msg("%Txt_Copy_Error_CreateDest", GVArgs(destPath).add(err.getErrorMessage()));
         if (GLog::Filter(COPY_FILE_DEBUG))
            GLog::Log(this, "LCmdCopy::copyFile(): error: %s", GVArgs(msg));
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
         switch (answ)
         {
            case GMessageBox::IDSKIP:
                 vfsSrc.closeFile(srcFile);
                 countFilesCopied += 1;
                 countFilesSkipped += 1;
                 bytesCopied += srcSize + 1;
                 return ERR_NONE;
            case GMessageBox::IDABORT:
                 vfsSrc.closeFile(srcFile);
                 return ERR_IOERROR;
            case GMessageBox::IDRETRY:
            default:
                 continue;
         }
      }
   }

   // Do the actual work of copying all the bytes.
   // Loop to give the user a chance to retry in case of any error.
   bool yesCopyEAsIfAny = true;
   longlong oldCopiedBytes = bytesCopied;
   for (;;)
   {
      err = copyFile_(srcFile, dstFile,
                      originalFileName, shortenedFName,
                      destPath, yesCopyEAsIfAny);
      // ---
      if (err == GError::ExtendedAttributesNotSupported && yesCopyEAsIfAny)
      {
         GMessageBox::Answer answ = GMessageBox::IDYES;
         if (!bIgnoreAllEAs)
         {
            GStringl msg("%Txt_Copy_Confirm_EAsNotSupported");
            answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Yy!nc");
         }
         switch (answ)
         {
            case GMessageBox::IDYESTOALL:
                 bIgnoreAllEAs = true;
                 // Yes, continue without a break statement!
            case GMessageBox::IDYES:
                 yesCopyEAsIfAny = false;
                 bytesCopied = oldCopiedBytes;
                 // Reset file pointers
                 vfsSrc.setFileSeekPosFromStart(srcFile, 0);
                 vfsDst.setFileSeekPosFromStart(dstFile, 0);
                 continue; // Retry, but ignore the EA's

            case GMessageBox::IDCANCEL:
                 err = GError::Ok;
                 iError = ERR_IOERROR;
                 break;

            case GMessageBox::IDNO:
            default:
                 break;
         }
      }

      // ---
      if (err != GError::Ok)
      {
         GStringl msg("%Txt_Copy_Error_UnknownCopyErr", GVArgs(sourcePath).add(destPath).add(err.getErrorMessage()));
         if (GLog::Filter(COPY_FILE_DEBUG))
            GLog::Log(this, "LCmdCopy::copyFile(): error: %s", GVArgs(msg));
         iError = ERR_IOERROR;
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
         switch (answ)
         {
            case GMessageBox::IDSKIP:
                 vfsSrc.closeFile(srcFile);
                 vfsDst.closeFile(dstFile);
                 countFilesCopied += 1;
                 countFilesSkipped += 1;
                 bytesCopied = oldCopiedBytes + srcSize + 1;
                 vfsDst.removeFile(0, destPath, false);
                 return ERR_NONE;
            case GMessageBox::IDABORT:
                 break;
            case GMessageBox::IDRETRY:
            default: {
                 bytesCopied = oldCopiedBytes;
                 // Pretend everything is OK as part of the retry operation.
                 iError = ERR_NONE;
                 // Reset file pointers.
                 vfsSrc.setFileSeekPosFromStart(srcFile, 0);
                 vfsDst.setFileSeekPosFromStart(dstFile, 0);
                 } continue;
         }
      }
      break;
   }

   vfsSrc.closeFile(srcFile);

   // Don't close the destination file yet. This will deny any attempt to
   // delete the destination file because it is still open in
   // OPEN_SHARE_DENYWRITE mode. This is just to make 100% sure
   // that it is not the destination file that we will delete
   // with DosForceDelete() later on.
   //
   // This extra care has to do with some very special circumstances that
   // can - in theory - occur when moving files. For instance if the user
   // tryes to move a file to it self (?). I don't think we actually need
   // to care about this, because it will probably never happen.
   // But just to be 100% safe... Don't close the destination file yet!
   // GFile::closeFile(dstFile);

   if (iError == ERR_NONE)
   {
      if (bMove)
      {
         // Test if the source file is write protected. If so then confirm
         // with the user if it's still OK to remove it.
         int srcAttr = vfsSrc.getFileAttributes(sourcePath, &err);
         if (err == GError::Ok && (srcAttr & GVfs::FAttrReadOnly))
         {
            if (!bRemoveAllReadOnly && !noConfirmations)
            {
               GStringl msg("%Txt_Copy_Confirm_MoveReadOnlySourceFile", GVArgs(sourcePath));
               GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, "Yy!sc");
               switch (answ)
               {
                  case GMessageBox::IDYES:
                       break;

                  case GMessageBox::IDYESTOALL:
                       bRemoveAllReadOnly = true;
                       break;

                  case GMessageBox::IDCANCEL:
                       vfsDst.closeFile(dstFile);
                       vfsDst.removeFile(0, destPath, false); // Source file has not been deleted yet ;-)
                       return ERR_CANCELED;

                  case GMessageBox::IDSKIP:
                  default:
                       vfsDst.closeFile(dstFile);
                       vfsDst.removeFile(0, destPath, false); // Source file has not been delete yet ;-)
                       countFilesCopied += 1;
                       countFilesSkipped += 1;
                       bytesCopied += srcSize + 1;
                       return ERR_NONE;
               }
            }

            // Remove the read-only attribute, so that we will be able to
            // successfully remove the write protected source file.
            // This operation will probably never fail but if it does then it
            // isn't a critical problem because then we will fail when we try
            // to remove the source file and the user will see an error message.
            srcAttr = GVfs::FAttrNormal;
            vfsSrc.setFileAttributes(sourcePath, srcAttr);
         }

         for (;;)
         {
            err = vfsSrc.removeFile(hdelSrc, sourcePath, false);
            if (err == GError::Ok)
            {
               // Update the newly renamed or moved file or directrory, in the 
               // Dynamic Directory Cache (in case it is contained in there).
               if (lcmd->options.dirCache.autoStripDirsAfterDirDeletions)
               {
                  GString fullSrcPath = vfsSrc.getFullVirtualPathTo(sourcePath.getFullPath());
                  GString fullDstPath = vfsSrc.getFullVirtualPathTo(destPath.getFullPath());
                  lcmd->dirCache->renameDirectoryInCache(fullSrcPath, fullDstPath);
               }
               break;
            }
            else
            {
               GStringl msg("%Txt_Copy_Error_DeleteSourceFile", GVArgs(sourcePath).add(err.getErrorMessage()));
               if (GLog::Filter(COPY_FILE_DEBUG))
                  GLog::Log(this, "LCmdCopy::copyFile(): error: %s", GVArgs(msg));
               GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
               switch (answ)
               {
                  case GMessageBox::IDSKIP:
                       vfsDst.closeFile(dstFile);
                       vfsDst.removeFile(0, destPath, false); // Source file has not been delete yet ;-)
                       countFilesCopied += 1;
                       countFilesSkipped += 1;
                       bytesCopied += srcSize + 1;
                       return ERR_NONE;

                  case GMessageBox::IDABORT:
                       vfsDst.closeFile(dstFile);
                       vfsDst.removeFile(0, destPath, false); // Source file has not been delete yet ;-)
                       return ERR_IOERROR;

                  case GMessageBox::IDRETRY:
                  default:
                       continue;
               }
            }
         }
      }

      vfsDst.closeFile(dstFile); // Everything has performed OK. So it should be safe to close the destination file.
      *pbSkipped = false;
      countFilesCopied += 1;
      bytesCopied += 1;
   }
   else
   {
      vfsDst.closeFile(dstFile);
      vfsDst.removeFile(0, destPath, false);
   }

   if (iError == ERR_NONE)
   {
      // If a single file or directory is being moved or copied to a new
      // name then the ".LONGNAME" EA should always be removed from
      // the destination.
      if (isLevel1 && removeEAName)
      {
         for (int i=0; REMOVE_LONGNAME_EAS[i] != null; i++)
            destPath.removeEA(REMOVE_LONGNAME_EAS[i]);
      }
   }

   return iError;
}

LCmdCopy::ErrType LCmdCopy::copyDirectory ( const GFile& sourcePath,
                                            GFile& destPath,
                                            bool* pbSkipped,
                                            bool isLevel1 )
{
   GFile::Slash(curSourcePath = sourcePath.getFullPath());
   GFile::Slash(curDestPath = destPath.getFullPath());
   sendUserMessageToMonitor("UpdtMon");

   // ---
   *pbSkipped = true; // Until the opposite has been proven

   // Keep a copy of the original full filename, in case we need to put it
   // into the EA's of the file, in case destination file system has support
   // for old style (short) FAT filenames only.
   GString originalFName = sourcePath.getFileName();
   GString shortenedFName;

   // Read the EAs of the source directory.
   // This is to test the .LONGNAME attribute to restore the long
   // filename for instance when copy/move a file back from a FAT drive
   // to a HPFS drive.
   bool tryingToRestoreLongFName = false;
   GExtendedAttributes eas(sourcePath);
   GString longName = eas.getEAString(EA_NAME_LONGNAME);
   GString shortName = eas.getEAString("LCMD_SHORTNAME");
   if (longName != "" &&
       shortName.equalsIgnoreCase(originalFName) &&
       srcDriveInfo.drive != dstDriveInfo.drive &&
       sourcePath.getFileName().equalsIgnoreCase(destPath.getFileName()))
   {
      // Replace all special characters from the .LONGNAME EA and replace
      // them with an exclamation (just as the WPS does).
      longName.replaceAll(ILLEGAL_FS_CHARS, '!');
      destPath.setFileName(longName);
      tryingToRestoreLongFName = true;
   }

   // First, make sure that the specified destination directory actually
   // exist. If it doesn't then try to create it.
   int countAttempts = 1;
   while (!vfsDst.exist(destPath))
   {
      GError err = vfsDst.createDirectory(destPath);

      if (countAttempts == 1 && !renameOnly &&
         (srcDriveInfo.drive != dstDriveInfo.drive) &&
         (err == GError::FilenameExceedRange || err == GError::InvalidName ||
                  (err == GError::PathNotFound &&
                   tryingToRestoreLongFName))) // Can happen when directory name was
                                               // fetched from .LONGNAME EA because
                                               // of some special (linefeed?)
                                               // characters.
      {
         if (tryingToRestoreLongFName)
         {
            destPath.setFileName(originalFName);
            tryingToRestoreLongFName = false;
            countAttempts = 0; // Adjust for the next += 1
         }
         else
         {
            if (confirmAutoFilenameShorten && !noConfirmations)
            {
               GStringl msg("%Txt_Copy_Confirm_AutoShortenFName", GVArgs(destPath.getFileName()));
               GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_WARNING, "Yy!sa");
               switch (answ)
               {
                  case GMessageBox::IDSKIP:
                       countFilesCopied += 1;
                       countFilesSkipped += 1;
                       bytesCopied += 1;
                       return ERR_NONE;
                  case GMessageBox::IDABORT:
                       return ERR_IOERROR;
                  case GMessageBox::IDYESTOALL:
                       confirmAutoFilenameShorten = false;
                       break;
                  case GMessageBox::IDYES:
                  default:
                       break;
               }
            }

            GString fullPath = destPath.getFullPath();
            GString shortenedDir = vfsDst.getShortenedDirOrPath(fullPath);
            destPath.setFullPath(shortenedDir);
            shortenedFName += destPath.getFileName(); // Get the last directory name
         }
         countAttempts += 1;
         continue;
      }

      if (err == GError::Ok)
         break;

      GStringl msg("%Txt_Copy_Error_CreateDir", GVArgs(destPath).add(err.getErrorMessage()));
      GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
      switch (answ)
      {
         case GMessageBox::IDSKIP:
              countFilesCopied += 1;
              countFilesSkipped += 1;
              bytesCopied += 1;
              return ERR_NONE;
         case GMessageBox::IDABORT:
              return ERR_IOERROR;
         case GMessageBox::IDRETRY:
         default:
              continue;
      }
   }

   // If the directory name was automatically shortened (in case of copy/move
   // a HPFS filename to a FAT drive) then store the original filename
   // as the .LONGNAME EA of the destination file.
   if (shortenedFName != "")
   {
      eas.setEAString("LCMD_SHORTNAME", shortenedFName);
      eas.setEAString(EA_NAME_LONGNAME, originalFName);
   }

   // Copy the extended attributes of the directory, just as for files.
   GError err = eas.writeEAs(destPath);
   if (err != GError::Ok)
      return ERR_IOERROR;

   // If a single file or directory is being moved or copied to a new
   // name then the ".LONGNAME" EA should always be removed from
   // the destination.
   if (isLevel1 && removeEAName)
   {
      for (int i=0; REMOVE_LONGNAME_EAS[i] != null; i++)
         destPath.removeEA(REMOVE_LONGNAME_EAS[i]);
   }

   // Copy the standard attributes of the directory, just as for files.
   int dirAttrs = vfsSrc.getFileAttributes(sourcePath, &err);
   int dirAttrsOriginal = dirAttrs;
   if (err != GError::Ok)
      return ERR_IOERROR;

   // Switch off the Write Protection flag (if any) in case we are requested to
   if (dontCopyWriteProtect)
      dirAttrs &= ~GVfs::FAttrReadOnly;

   err = vfsDst.setFileAttributes(destPath, dirAttrs);
   if (err != GError::Ok)
      return ERR_IOERROR;

   countFilesCopied += 1; // Count # files/directories copied
   bytesCopied += 1;

   // Copy all files and sub-directories that are contained in the
   // current source directory.
   GString srcDir = curSourcePath; // Current source directory, incl. terminating slash.
   GString dstDir = curDestPath; // Current dest. directory, incl. terminating slash.
   GVfs::List list;
   ErrType ret = ERR_NONE;
   GString temp = srcDir + "*.*";
   vfsSrc.fillList(list, temp, true, true, true, true);
   for (int i=0, num = list.size();
        i<num && ret == ERR_NONE;
        i++)
   {
      const GVfs::List::Item& item = list[i];
      const GString& fname = item.getFName();
      longlong fsize = item.getSizeIdeal();
      GFile src(srcDir + fname);
      GFile dst(dstDir + fname);
      const int flags = item.getFlags();
      if (flags & GVfs::FAttrDirectory)
         ret = copyDirectory(src, dst, pbSkipped, false); // Recurse into this directory to copy all items in the directory including its subdirectories.
      else
         ret = copyFile(fsize, src, dst, pbSkipped, false);
   }

   // ---
   if (ret == ERR_NONE && bMove)
   {
      // If the source directory is write protected then confirm
      // with the user if it's still OK to remove it.
      if (dirAttrsOriginal & GVfs::FAttrReadOnly)
      {
         if (!bRemoveAllReadOnly && !noConfirmations)
         {
            GStringl msg("%Txt_Copy_Confirm_MoveReadOnlySourceDir", GVArgs(sourcePath));
            GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, "Yy!sc");
            switch (answ)
            {
               case GMessageBox::IDYES:
                    break;

               case GMessageBox::IDYESTOALL:
                    bRemoveAllReadOnly = true;
                    break;

               case GMessageBox::IDCANCEL:
                    return ERR_CANCELED;

               case GMessageBox::IDSKIP:
               default:
                    return ERR_NONE;
            }
         }

         // Remove the read-only attribute, so that we will be able to
         // successfully remove the write protected source directory.
         err = vfsSrc.setFileAttributes(sourcePath, GVfs::FAttrNormal);
      }

      for (;;)
      {
         err = vfsSrc.removeDirectory(hdelSrc, sourcePath, false);
         if (err == GError::Ok)
         {
            // Update the newly renamed or moved file or directrory, in the 
            // Dynamic Directory Cache (in case it is contained in there).
            if (lcmd->options.dirCache.autoStripDirsAfterDirDeletions)
            {
               GString fullSrcPath = vfsSrc.getFullVirtualPathTo(sourcePath.getFullPath());
               GString fullDstPath = vfsSrc.getFullVirtualPathTo(destPath.getFullPath());
               lcmd->dirCache->renameDirectoryInCache(fullSrcPath, fullDstPath);
            }
            break;
         }
         else
         {
            GStringl msg("%Txt_Copy_Error_DeleteSourceDir", GVArgs(sourcePath).add(err.getErrorMessage()));
            GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
            switch (answ)
            {
               case GMessageBox::IDSKIP:
                    return ERR_NONE;

               case GMessageBox::IDABORT:
                    return ERR_IOERROR;

               case GMessageBox::IDRETRY:
               default:
                    continue;
            }
         }
      }
   }

   *pbSkipped = false;

   return ret;
}

LCmdCopy::ErrType LCmdCopy::doCopyOrMoveItem ( bool isDir,
                                               longlong srcFileSize,
                                               const GString& srcPath,
                                               const GString& dstPath,
                                               bool *pSkipped,
                                               bool renameOnly )
{
   bool bDone = false; // Until the opposite has been proven

   *pSkipped = false; // Ditto

   while (bMove)
   {
      // First, try an atomic move (don't copy-then-remove).
      GError rc = vfsSrc.moveOrRenameFile(srcPath, dstPath, false);
      if (GLog::Filter(COPY_FILE_DEBUG))
         GLog::Log(this, "\tReturned %d (%s)", GVArgs(rc.sysErrorCode).add(rc.getErrorMessage()));
      if (rc == GError::Ok)
      {
         countFilesCopied += 1;
         if (renameOnly)
            bytesCopied += 1; // "Renaming single file" means "one byte" only.
         else
            bytesCopied += srcFileSize + 1;
         bDone = true;
         sendUserMessageToMonitor("UpdtMon");

         // Update the newly renamed or moved file or directrory, in the 
         // Dynamic Directory Cache (in case it is contained in there).
         if (lcmd->options.dirCache.autoStripDirsAfterDirDeletions)
         {
            GString fullSrcPath = vfsSrc.getFullVirtualPathTo(srcPath);
            GString fullDstPath = vfsSrc.getFullVirtualPathTo(dstPath);
            lcmd->dirCache->renameDirectoryInCache(fullSrcPath, fullDstPath);
         }

         // If we are about to _rename_ one single file or directory,
         // then remove the ".LONGNAME" EA, if any. In case of any error
         // when doing this, just ignore it. This is not a critical
         // operation anyway.
         if (removeEAName)
         {
            for (int i=0; REMOVE_LONGNAME_EAS[i] != null; i++)
               GExtendedAttributes::DiscardEA(dstPath, REMOVE_LONGNAME_EAS[i]);
         }
         break;
      }
      else
      {
         // vfsSrc.moveOrRenameFile() will always fail in some situations 
         // that we can actually work around by rather doing 
         // a copy-then-delete operation.
         bool recoverableError;
         switch (rc.sysErrorCode)
         {
            case ERROR_NOT_SAME_DEVICE:
               // Source- and destination drive/device is not the same.
            case ERROR_FILENAME_EXCED_RANGE:
               // Attempt to move a longname file to a drive/device 
               // that does not support long filenames.
               recoverableError = true;
               break;
            case ERROR_ACCESS_DENIED:
               // Source is open in some other application.
            default:
               recoverableError = false;
               break;
         }

         // If we are about to perform a "pure" rename operation on one single
         // file only, or if the system reported error is of a type that let
         // us foresee that there is no need to attempt to move manually (by
         // first-copy-then-delete) then show an error message to the user.
         if (renameOnly || !recoverableError || moveAtomicOnly)
         {
            const char* str = (renameOnly ? "%Txt_Copy_Error_RenameFileOrDir" : "%Txt_Copy_Error_UnknownMoveErr");
            GStringl msg(str, GVArgs(srcPath).add(dstPath).add(rc.getErrorMessage()));
            if (GLog::Filter(COPY_FILE_DEBUG))
               GLog::Log(this, "LCmdCopy::doCopyOrMoveItem(), error: %s", GVArgs(msg));
            GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
            switch (answ)
            {
               case GMessageBox::IDSKIP:
                    iError = ERR_NONE;
                    *pSkipped = true;
                    countFilesCopied += 1;
                    countFilesSkipped += 1;
                    if (renameOnly)
                       bytesCopied += 1;
                    else
                       bytesCopied += srcFileSize + 1;
                    return iError;

               case GMessageBox::IDABORT:
                    iError = ERR_IOERROR;
                    return iError;

               case GMessageBox::IDRETRY:
               default:
                    continue;
            }
         }
         else
         if (recoverableError)
         {
            // vfsSrc.moveOrRenameFile() didn't do it, but we can attempt 
            // to do it our self by first copy to the target and then
            // delete from the source.
            break;
         }
      }
   }

   if (!bDone)
   {
      GFile src(srcPath);
      GFile dst(dstPath);
      if (isDir)
      {
         if (vfsDst.exist(dst))
         {
            if (!bMergeAllDirs && !noConfirmations)
            {
               GStringl msg("%Txt_Copy_Confirm_MergeDir", GVArgs(dstPath));
               GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, "Yy!cs");
               switch (answ)
               {
                  case GMessageBox::IDYES:
                       break;

                  case GMessageBox::IDYESTOALL:
                       bMergeAllDirs = true;
                       break;

                  case GMessageBox::IDSKIP:
                       *pSkipped = true;
                       countFilesCopied += 1;
                       countFilesSkipped += 1;
                       if (renameOnly)
                          bytesCopied += 1; // "Renaming single file" means "one byte" only.
                       else
                          bytesCopied += srcFileSize + 1;
                       return ERR_NONE;

                  case GMessageBox::IDCANCEL:
                  default:
                       iError = ERR_IOERROR;
                       return iError;
               }
            }
         }
         iError = copyDirectory(src, dst, pSkipped, true);
      }
      else
      {
         iError = copyFile(srcFileSize, src, dst, pSkipped, true);
      }
   }

   return iError;
}

void LCmdCopy::runTheWorkerThread ( GWorkerThread& worker )
{
   bool skipped = false;

   // Now, do the work of copy the file(s) including the sub-dirs (if any).

   const int num = items->getCount();
   for (int i=0; i<num; i++)
   {
      LCmdCopyFileItem& item = items->get(i);
      GString srcPath = item.getSourcePath();
      GString dstPath = item.getDestPath();
      try {
         LCmdFileItem file(vfsSrc, srcPath);
         GString fileName = file.getFileName();
         if (fpanelIsSrc)
            iCurItemIdx = fpanel->findItem(fileName);
         else
            iCurItemIdx = -1; // Just a dummy statement

         bool srcAndDestIsTheSame = srcPath.equalsIgnoreCase(dstPath);

         bool doit = true;
         if (file.isDirectory() && !srcAndDestIsTheSame)
         {
            GError rc = GVfs::TestDirectoryCircularity(vfsSrc, srcPath, vfsDst, dstPath);
            if (rc != GError::Ok)
            {
               GStringl msg(bMove ? "%Txt_Copy_Error_UnknownMoveErr" : "%Txt_Copy_Error_UnknownCopyErr", GVArgs(srcPath).add(dstPath).add(rc.getErrorMessage()));
               GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "As");
               switch (answ)
               {
                  case GMessageBox::IDSKIP:
                       doit = false;
                       skipped = true;
                       countFilesSkipped += 1;
                       break;

                  case GMessageBox::IDABORT:
                  default:
                       doit = false;
                       iError = ERR_IOERROR;
                       break;
               }
            }
         }

         if (doit)
         {
            // Do the inner work
            bool dir = file.isDirectory();
            iError = doCopyOrMoveItem(dir, file.fileSize,
                                      srcPath, dstPath, &skipped,
                                      srcAndDestIsTheSame);
         }

         if (iError != ERR_NONE)
            break;

         if (!skipped && fpanelIsSrc)
         {
            // Update the item in the file panel (and dialog box showing status)
            int fitemIndex = fpanel->findItem(fileName);
            LCmdFileItem& fitem = fpanel->items[fitemIndex];
            int flags = fitem.getInternalFlags();
            fitem.setInternalFlags(flags & ~LCmdFileItem::FINFO_SHOWSIZE);
            if (fpanel->markedFilesCount >= 1)
            {
               fitem.setMarked(false);
               fpanel->sizeOfMarkedFiles -= fitem.fileSize;
               fpanel->markedFilesCount -= 1;

               // Update the content of the infobar, which is used to show the 
               // remaining number of files and bytes to copy/move.
               sendUserMessageToMonitor("UpdtInfobar");
            }
         }
      } catch (GFileNotFoundException& /*e*/) {
         // Source file no longer exist.
         // This can happen e.g. if user tries to move/copy a file that is 
         // contained in the file panel even though the file actually has 
         // been removed by some other application in the background.
         GStringl msg("%Txt_Copy_Error_SrcNotFound", GVArgs(srcPath)); // "Source file not found:\n%s"
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Sa");
         bool doAbort = false;
         switch (answ)
         {
            case GMessageBox::IDSKIP:
                  countFilesSkipped += 1;
                  break;

            case GMessageBox::IDABORT:
            default:
                  doAbort = true;
                  iError = ERR_IOERROR;
                  break;
         }
         if (doAbort)
            break;
         else
            continue; 
      }
   }

   // Perform all source deletions, in case source VFS has not performed 
   // all deletions directly.
   if (iError == ERR_NONE && bMove)
   {
      if (hdelSrc != 0)
      {
         vfsSrc.performDeletion(hdelSrc);
         vfsSrc.closeDeletion(hdelSrc);
         hdelSrc = 0;
      }
   }

   // Now we are finished with the copy or move progress it self.
   if (iError == ERR_NONE)
   {
      sendUserMessageToMonitor("FINISHED!");
      if (waitOnFinish)
      {
         waitingForFinishCommand = true;
         waitingForFinishedSem.wait();
      }
   }
}

void LCmdCopy::updateMonitor ( GDialogPanel& dlg, bool forceUpdateAll )
{
   ulonglong timeSinceStartMillis = getTimeMillisSinceStart(true);
   double timeSinceStartSeconds = timeSinceStartMillis / 1000.0;

   // Update the progress bars.
   if (curDestPath != "")
   {
      dlg.setComponentValue("102", curSourcePath);
      dlg.setComponentValue("104", curDestPath);
   }

   // Update the panel cursor showing which item is about being
   // copied or moved.
   if (fpanelIsSrc && iCurItemIdx != iPrevItemIdx)
   {
      // Make sure all items that have been processed since the
      // previous update are all repainted on screen. This is to
      // remove the tag marks, so that the visualizing of the
      // copy progress is rock solid. However, repaint only those
      // items that actually are located within the visible area.
      LCmdFilePanelModeAbstract& curView = fpanel->getCurrentView();
      int firstVisible = curView.getFirstVisibleIndex();
      int first = GMath::Max(firstVisible, iPrevItemIdx);
      for (int i=first; i<iCurItemIdx; i++)
         fpanel->drawItem(i);
      iPrevItemIdx = iCurItemIdx;
      fpanel->selectItem(iCurItemIdx);
   }

   // Update the progress bar.
   int percentFinished = 0;
   if (totalBytesToCopy > 0)
   {
      int perThousandFinished = int((bytesCopied * 1000) / totalBytesToCopy);
      percentFinished = perThousandFinished / 10;
      GProgressBar& pbar = dynamic_cast<GProgressBar&>(dlg.getComponentByID("106"));
      pbar.setCurrentValue(perThousandFinished);
   }

   // Update the percent-finished component.
   GString txt("%d%%", GVArgs(percentFinished));
   dlg.setComponentValue("119", txt);

   if (!showStatistics)
      return;

   // ---
   if (curDestPath != "")
   {
      dlg.setComponentValue("112", fileCount - countFilesCopied);
      dlg.setComponentValue("135", countFilesCopied);
   }

   // ---
   // Update the time-since-start component.
   unsigned long tss = (unsigned long) timeSinceStartSeconds;
   int hour = tss / 3600;
   tss -= hour * 3600;
   int min = tss / 60;
   tss -= min * 60;
   int sec = tss;
   const GLocaleData& ld = GProgram::GetProgram().getLocaleData();
   GString str("%02d%s%02d%s%02d", GVArgs(hour).add(ld.timeSeparator).add(min).add(ld.timeSeparator).add(sec));
   dlg.setComponentValue("114", str);

   // ---
   double bpsAverage = 0;
   if (timeSinceStartSeconds != 0.0) // Prevent "dive by zero".
      bpsAverage = bytesCopied / timeSinceStartSeconds;

   // ---
   // Update the byte-count related components once each seconds at most.
   ulonglong timeDiffMillis = timeSinceStartMillis - prevTimeSinceStartMillis;
   ulonglong bytesLeft = totalBytesToCopy - bytesCopied;
   if (bytesCopied != prevBytesCopied && // Only if any more bytes are copied at all.
      (forceUpdateAll || timeDiffMillis >= 1000))
   {
      dlg.setComponentValue("107", GStringUtil::ToByteCountString(bytesCopied, false, false, false));
      dlg.setComponentValue("131", GStringUtil::ToByteCountString(bytesLeft, false, false, false));

      // ---
      // Update the bytes-per-second (bps) component.
      if (!(bMove && renameOnly))
      {
         dlg.setComponentValue("117", GStringUtil::ToByteCountString(longlong(bpsAverage), false, false, false));

         // Update the BPS Monitor
         double timeDiffSeconds = timeDiffMillis / 1000.0;
         longlong bpsLast = 0;
         if (timeDiffSeconds != 0.0) // Prevent "divide by zero".
            bpsLast = longlong((bytesCopied - prevBytesCopied) / timeDiffSeconds);
         if (bpsLast > bpsPeak)
         {
            bpsPeak = bpsLast;
            dlg.setComponentValue("125", GStringUtil::ToByteCountString(bpsPeak, false, false, false));
         }

         // Update the BPS monitor, but not if this is the last (final) update.
         if (!forceUpdateAll)
         {
            dlg.setComponentValue("137", GStringUtil::ToByteCountString(bpsLast, false, false, false));

            // Add another BPS sample to the monitor. One sample for
            // each second that have passed since the last sample was added.
            GLinePlotter& lplot = dynamic_cast<GLinePlotter&>(dlg.getComponentByID("122"));
            const int num = int(timeDiffSeconds);
            for (int i=0; i<num; i++)
               lplot.addSample(double(bpsLast));
         }
      }

      // ---
      // Update the component showing the estimated time.
      if (bpsAverage != 0.0) // Prevent "divide by zero".
      {
         double etaSeconds = bytesLeft / bpsAverage;
         int ietaSec = int(etaSeconds + 0.5);
         if (ietaSec <= 0)
         {
            dlg.setComponentValue("110", "");
         }
         else
         {
            if (ietaSec > 86400) // If more than 24 hours.
            {
               GString str;
               if (timeSinceStartSeconds > 8) // Wait some seconds before showing this.
                  str = GStringl("%Txt_DlgFileCopyProgressBar_MoreThan24HoursLeft");
               dlg.setComponentValue("110", str);
            }
            else
            {
               const GLocaleData& ld = GProgram::GetProgram().getLocaleData();
               GString str = ld.getTimeStringFromMillis(ietaSec * 1000, false);
               dlg.setComponentValue("110", str);
            }
         }
      }

      // ---
      prevBytesCopied = bytesCopied;
      prevTimeSinceStartMillis = timeSinceStartMillis;
   }
}

void LCmdCopy::onWorkerThreadInitDialog ( GWorkerThread& worker,
                                          GDialogPanel& dlg )
{
   if (bMove)
   {
      dlg.setTitleText("%Txt_DlgFileCopyProgressBar_TitleMv");
      dlg.setComponentValue("200", "%Txt_DlgFileCopyProgressBar_CopyGroupMv");
   }
   else
   {
      dlg.setTitleText("%Txt_DlgFileCopyProgressBar_Title");
      dlg.setComponentValue("200", "%Txt_DlgFileCopyProgressBar_CopyGroup");
   }

   // Set the initial (cleared) value on most components.
   dlg.setComponentValue("109", GStringUtil::ToByteCountString(totalBytesToCopy, false, false, false));
   dlg.setComponentValue("112", fileCount);
   dlg.setComponentValue("133", fileCount);
   dlg.setComponentValue("114", GString::Empty);
   dlg.setComponentValue("110", GString::Empty);
   dlg.setComponentValue("107", GString::Empty);
   dlg.setComponentValue("131", GString::Empty);
   dlg.setComponentValue("125", "0");
   dlg.setComponentValue("117", "0");
   dlg.setComponentValue("137", "0");
   dlg.setComponentValue("135", "0");
   dlg.setComponentValue("102", GString::Empty);
   dlg.setComponentValue("104", GString::Empty);

   // Clear the progress bar.
   dlg.setComponentValue("106", 0);

   // Clear the line plotter.
   GLinePlotter& lplot = dynamic_cast<GLinePlotter&>(dlg.getComponentByID("122"));
   lplot.clearAllSamples();

   // Set the Start Time.
   const GLocaleData& loc = GProgram::GetProgram().getLocaleData();
   GString timeStr = loc.getCurrentTimeString(true);
   dlg.setComponentValue("127", timeStr);

   // Set the initial state on the toggle buttons.
   dlg.setComponentValue("138", waitOnFinish);
   dlg.setComponentValue("120", noConfirmations);

   // Restore the default state of the cancel-button.
   dlg.setComponentEnabled("DLG_CANCEL", true);
   GPushButton& butt = dynamic_cast<GPushButton&>(dlg.getComponentByID("DLG_CANCEL"));
   butt.setText("%DLG_CANCEL");
   butt.setIconID("ICON_CANCEL");

   // Make sure that the Cancel-button has the initial keyboard focus
   dlg.setComponentFocus("DLG_CANCEL");
}

void LCmdCopy::onWorkerThreadRequestStop ( GWorkerThread& worker,
                                           GDialogPanel& dlg,
                                           bool /*success*/ )
{
   if (bHasFinished)
   {
      // User has clicked the "Finsihed!" button to close the dialog
      dlg.dismiss();
   }
   else
   {
      // User has clicked the "Cancel" button to cancel the progress
      iError = ERR_CANCELED;
   }
}

void LCmdCopy::onWorkerThreadCtrlChanged ( GWorkerThread& worker,
                                           GDialogPanel& dlg,
                                           const GString& compID,
                                           const GString& /*newValue*/ )
{
   if (compID == "120") // Toggle button: No Confirmations
      noConfirmations = dlg.getComponentBoolValue("120");
   else
   if (compID == "138") // Toggle button: Wait on finish
      waitOnFinish = dlg.getComponentBoolValue("138");
}

void LCmdCopy::onWorkerThreadUserEvent ( GWorkerThread& worker,
                                         GDialogPanel& monitor,
                                         const GString& msgID,
                                         GObject* /*userParam*/ )
{
   if (msgID == "FINISHED!")
   {
      updateMonitor(monitor, true); // Update with the final result.
      waitingForFinishCommand = true;
      GPushButton& butt = dynamic_cast<GPushButton&>(monitor.getComponentByID("DLG_CANCEL"));
      butt.setText("%Txt_DlgFileCopyProgressBar_Finished");
      butt.setIconID("ICON_OK");
   }
   else
   if (msgID == "UpdtMon")
   {
      updateMonitor(monitor, false);
   }
   else
   if (msgID == "UpdtInfobar")
   {
      fpanel->infoBar.updateAllFileItemInfoCells();
   }
}

void LCmdCopy::onWorkerThreadCommand ( GWorkerThread& worker,
                                       GDialogPanel& monitor,
                                       const GString& cmd )
{
   if (cmd == "DLG_CANCEL")
   {
      monitor.setComponentEnabled(cmd, false);
      if (waitingForFinishCommand)
         waitingForFinishedSem.notifyAll();
      else
         requestStop(false);
   }
}

bool LCmdCopy::CopyOrMoveFiles ( GVfs& vfsSrc,
                                 GVfs& vfsDst,
                                 GArray<LCmdCopyFileItem>& items,
                                 bool move,
                                 LCmdFilePanel& fpanel,
                                 bool fpanelIsSrc,
                                 bool renameOnly,
                                 bool removeEAName,
                                 const GString& autoSelect1,
                                 const GString& autoSelect2,
                                 int* countFilesSkipped_,
                                 bool refillPanel )
{
   if (countFilesSkipped_ != null)
      *countFilesSkipped_ = 0;

   LCmdOptions& opt = LCmdOptions::GetOptions();
   bool showStat = opt.fileCopy.showStatistics;
   GString dlgName = (showStat ? "DlgFileCopyProgressBar" : "DlgFileCopyProgressBarNoStat");
   LCmdCopy cp(vfsSrc, vfsDst, dlgName);

   LCmdFilePanel& leftPanel = LCmdFilePanel::GetLeftPanel();
   LCmdFilePanel& rightPanel = LCmdFilePanel::GetRightPanel();
   GString selectedItem1 = leftPanel.getCurItemName();
   GString selectedItem2 = rightPanel.getCurItemName();
   int selectedItemIndex1 = leftPanel.getCurrentSelectedIndex();
   int selectedItemIndex2 = rightPanel.getCurrentSelectedIndex();

   if (!cp.reInit(&items, fpanel, fpanelIsSrc, move, renameOnly, removeEAName))
      return false;

   // Start the background worker thread and show the progress bar dialog.
   try {
      GWindow& owin = fpanel.getTopLevelWindow();
      cp.workModal(owin);
   } catch (GThreadStartException& /*e*/) {
      cp.iError = ERR_CREATETHREAD;
      cp.bHasFinished = true;
   }

   // ---
   if (countFilesSkipped_ != null)
      *countFilesSkipped_ = cp.countFilesSkipped;

   // ---
   if (refillPanel && (cp.iError == ERR_CANCELED || (cp.countFilesCopied > 0 && 
                                                     !(renameOnly && 
                                                     cp.countFilesSkipped > 0))))
   {
      leftPanel.reRead(false, -2);
      if (autoSelect1 != "")
         selectedItem1 = autoSelect1;
      if (leftPanel.selectItem(selectedItem1) == -1)
         if (leftPanel.selectItem(selectedItemIndex1) == -1)
            leftPanel.selectItem(leftPanel.items.getCount() - 1);

      rightPanel.reRead(false, -2);
      if (autoSelect2 != "")
         selectedItem2 = autoSelect2;
      if (rightPanel.selectItem(selectedItem2) == -1)
         if (rightPanel.selectItem(selectedItemIndex2) == -1)
            rightPanel.selectItem(rightPanel.items.getCount() - 1);
   }

   // ---
   LCmdFilePanel& currentPanel = LCmdFilePanel::GetCurrentPanel();
   currentPanel.activatePanelDriveAndDir(true);
   return cp.iError == ERR_NONE;
}

#undef COPY_FILE_DEBUG
