/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdCmdLineEntry.h"
#include "lcmd/LCmdCmdContainer.h"
#include "lcmd/LCmdOptions.h"
#include "lcmd/LCmdProcess.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelModeInfo.h"
#include "lcmd/LCmdFilePanelModeBrief.h"
#include "lcmd/LCmdDlgChildProcessSelector.h"
#include "lcmd/LCmd.h"

#include "glib/GProgram.h"
#include "glib/sys/GSystem.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GMouseEvent.h"
#include "glib/gui/event/GUserMessage.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/util/GExpressionParser.h"
#include "glib/util/GMath.h"
#include "glib/exceptions/GSyntaxErrorException.h"
#include "glib/exceptions/GThreadStartException.h"

DEFINE_COMMAND_TABLE(LCmdCmdLineEntry);
   ADD_COMMAND("cmdClear", cmdClear);
   ADD_COMMAND("cmdCopy", cmdCopy);
   ADD_COMMAND("cmdCopyAllCommandLineText", cmdCopyAllCommandLineText);
   ADD_COMMAND("cmdCut", cmdCut);
   ADD_COMMAND("cmdMoveCaretRight", cmdMoveCaretRight);
   ADD_COMMAND("cmdMoveCaretLeft", cmdMoveCaretLeft);
   ADD_COMMAND("cmdMoveCaretHome", cmdMoveCaretHome);
   ADD_COMMAND("cmdMoveCaretEnd", cmdMoveCaretEnd);
   ADD_COMMAND("cmdMoveCaretWordRight", cmdMoveCaretWordRight);
   ADD_COMMAND("cmdMoveCaretWordLeft", cmdMoveCaretWordLeft);
   ADD_COMMAND("cmdPaste", cmdPaste);
   ADD_COMMAND("cmdScrollConMonHome", cmdScrollConMonHome);
   ADD_COMMAND("cmdScrollConMonTop", cmdScrollConMonTop);
   ADD_COMMAND("cmdScrollConMonEnd", cmdScrollConMonEnd);
   ADD_COMMAND("cmdScrollConMonBottom", cmdScrollConMonBottom);
   ADD_COMMAND("cmdScrollConMonUp", cmdScrollConMonUp);
   ADD_COMMAND("cmdScrollConMonDown", cmdScrollConMonDown);
   ADD_COMMAND("cmdScrollConMonLeft", cmdScrollConMonLeft);
   ADD_COMMAND("cmdScrollConMonRight", cmdScrollConMonRight);
   ADD_COMMAND("cmdScrollConMonPageUp", cmdScrollConMonPageUp);
   ADD_COMMAND("cmdScrollConMonPageDown", cmdScrollConMonPageDown);
   ADD_COMMAND("cmdSelectAll", cmdSelectAll);
   ADD_COMMAND("cmdToggleInsertMode", cmdToggleInsertMode);
END_COMMAND_TABLE;

LCmdCmdLineEntry::MyMouseListener::MyMouseListener ( class LCmdCmdLineEntry& owner )
                                  :owner(owner)
{
}

LCmdCmdLineEntry::MyMouseListener::~MyMouseListener ()
{
}

bool LCmdCmdLineEntry::MyMouseListener::mousePressed ( const GMouseEvent& ev )
{
   if (ev.buttonId != GMouseEvent::BUTTON2)
      return false;
   owner.grabFocus();
   owner.setVisiblePopupMenu(true, ev.xpos, ev.ypos);
   return true;
}

LCmdCmdLineEntry::LCmdCmdLineEntry ( LCmdCmdLine& parentWin,
                                     const GString& constraints )
                 :GMultiLineEditor("Entry", constraints,
                                   parentWin,
                                   WS_VISIBLE,
                                   WS2_IGNORE_COLORS_PROFILE,
                                   0,     // maxTextLength
                                   false, // wordWrap
                                   false, // sysBorder
                                   false, // readOnly
                                   true,  // ignoreTab
                                   true), // useAsSingleLine
                  mouseListener(*this),
                  cmdLine(parentWin),
                  echoIsOn(true),
                  colorOpt(),
                  childWaitingForInput(null),
                  childProcessSelectorDlg(null),
                  aliases(),
                  internalCommands(60, 10, true),
                  sepSessionApps(32, -3, true),
                  runningChildren(4, 4),
                  cmdHist(*this, MAXCMDHIST),
                  dirHist(*this, MAXDIRHIST),
                  dirStack(8, 4)
{
   GWindow& peer = getPeer();
   peer.addMouseListener(&mouseListener);

   // Make some space for the border area.
   GLineBorder* lb = new GLineBorder(GLineBorder::AlignOuter);
   lb->setBackgroundColor(GColor::DGRAY.getLighter(64));
   setInsets(new GInsets(2, 2, 2, 2), true);
   setBorder(lb, true);

   // ---
   setInsertMode(true); // Set insert-mode ON upon startup by default
   setMaxTextLength(MAXCOMMANDSIZE);
   setAccelTable("CommandLineAccel");

   // ---
   colorOpt.entryBck = getBackgroundColor();
   colorOpt.entryTxt = getForegroundColor();
   defaultFontNameSize = lcmd->options.DefaultFontCmdLineEntry;
   setFontNameSize(defaultFontNameSize);

   // Activate the popup menu of which to display when the user
   // right-click on the command line entry window.
   GProgram& prg = GProgram::GetProgram();
   setPopupMenu("CommandLineMenu", prg.isUseFancyPopupMenues());

   // Initialize the bag of internal commands.
   static const char* IntCommands[] =
   {
      "ALIAS", 
      "CALC", 
      "CD", 
      "CHDIR", 
      "CLS", 
      "CLEAR",
      "ECHO", 
      "EXIT", 
      "FINDDUP", 
      "H", 
      "HIST",
      "HELP", 
      "INFO", 
      "POPD", 
      "PUSHD",
      "R", 
      "REP", 
      "SET", 
      "TAG", 
      "UNTAG",
      "VER", 
      "WHICH", 
      null
   };

   for (int i1=0; IntCommands[i1]; i1++)
   {
      InternalCommand *cmd = new InternalCommand(IntCommands[i1], false);
      internalCommands.put(IntCommands[i1], cmd);
   }

   // Initialize the bag of internal commands with the shell commands as well.
   static const char* ShellCommands[] =
   {
      "CALL", 
      "CHCP", 
      "COPY", 
      "DATE", 
      "DEL", 
      "DETACH", 
      "DIR", 
      "ERASE", 
      "FOR", 
      "GOTO", 
      "IF", 
      "MD", 
      "MKDIR", 
      "MOVE", 
      "PATH", 
      "PAUSE", 
      "REM", 
      "REN", 
      "RENAME", 
      "RD", 
      "RMDIR", 
      "RUN", 
      "SETLOCAL", 
      "START", 
      "TIME", 
      "TYPE", 
      "VOL", 
      null
    };

   for (int i2=0; ShellCommands[i2]; i2++)
   {
      InternalCommand *cmd = new InternalCommandToShell(ShellCommands[i2]);
      internalCommands.put(ShellCommands[i2], cmd);
   }
}

LCmdCmdLineEntry::~LCmdCmdLineEntry ()
{
}

void LCmdCmdLineEntry::writeProfile ( const GString& sectName, bool force )
{
   GString sn = sectName;
   LCmdOptions& opt = LCmdOptions::GetOptions();
   GSectionBag& ini = GProgram::GetProgram().getIniProperties();

   ini.putColor(sn, "MainBckColor", colorOpt.entryBck, force || opt.saveOnExit.colors);
   ini.putColor(sn, "MainFrgColor", colorOpt.entryTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "RunBckColor", colorOpt.bckDisabled, force || opt.saveOnExit.colors);
   ini.putColor(sn, "RunFrgColor", colorOpt.txtDisabled, force || opt.saveOnExit.colors);
   ini.putColor(sn, "WaitBckColor", colorOpt.bckWaitForInput, force || opt.saveOnExit.colors);
   ini.putColor(sn, "WaitFrgColor", colorOpt.txtWaitForInput, force || opt.saveOnExit.colors);

   // Internal programs
   int num = internalCommands.getCount();
   for (int i=0; i<num; i++)
   {
      InternalCommand& cmd = internalCommands.getIndexedItem(i);
      cmd.writeProfile(ini, force || opt.saveOnExit.otherOptions);
   }

   // Aliases
   aliases.writeProfile(ini, force || opt.saveOnExit.otherOptions);

   // Non Console Programs.
   sn = "SSA";
   ini.deleteSection(sn);
   num = sepSessionApps.getCount();
   for (int i=0; i<num; i++)
   {
      const GString& keyName = sepSessionApps.getKey(i);
      LCmdSepSessionApps& app = sepSessionApps.getIndexedItem(i);
      GString param("%c,%c", GVArgs(app.viaMain ? '1' : '0').add(app.viaPipe ? '1' : '0'));
      ini.putString(sn, keyName, param, force || opt.saveOnExit.otherOptions);
   }

   // Command History
   sn = "{CommandHistory}";
   ini.deleteSection(sn); // Remove old content to get rid of any unwanted items at the end (if any).
   num = cmdHist.getStringCount();
   for (int i=0; i<num; i++)
   {
      GString name("%04d", GVArgs(i+1));
      const GString& value = cmdHist.getIndexedString(i);
      ini.putString(sn, name, value, force || opt.saveOnExit.cmdHist);
   }

   // Directory History
   sn = "{DirectoryHistory}";
   ini.deleteSection(sn); // Remove old content to get rid of any unwanted items at the end (if any).
   num = dirHist.getStringCount();
   for (int i=0; i<num; i++)
   {
      GString name("%04d", GVArgs(i+1));
      const GString& value = dirHist.getIndexedString(i);
      ini.putString(sn, name, value, force || opt.saveOnExit.dirHist);
   }

   GMultiLineEditor::writeProfile(sectName, force);
}

void LCmdCmdLineEntry::queryProfile ( const GString& sectName )
{
   GString sn = sectName;
   GSectionBag& ini = GProgram::GetProgram().getIniProperties();

   colorOpt.entryBck = ini.getColor(sn, "MainBckColor", colorOpt.entryBck);
   colorOpt.entryTxt = ini.getColor(sn, "MainFrgColor", colorOpt.entryTxt);
   colorOpt.bckDisabled = ini.getColor(sn, "RunBckColor", colorOpt.bckDisabled);
   colorOpt.txtDisabled = ini.getColor(sn, "RunFrgColor", colorOpt.txtDisabled);
   colorOpt.bckWaitForInput = ini.getColor(sn, "WaitBckColor", colorOpt.bckWaitForInput);
   colorOpt.txtWaitForInput = ini.getColor(sn, "WaitFrgColor", colorOpt.txtWaitForInput);

   setBackgroundColor(colorOpt.entryBck);
   setForegroundColor(colorOpt.entryTxt);

   // Internal commands
   int num = internalCommands.getCount();
   for (int i=0; i<num; i++)
   {
      InternalCommand& cmd = internalCommands.getIndexedItem(i);
      cmd.readProfile(ini);
   }

   // Aliases
   aliases.readProfile(ini);

   // None Console Programs.
   sn = "SSA";
   GKeyBag<GString>* ssaBag = ini.getSectionBag(sn);
   if (ssaBag != null)
   {
      for (int i=0; i<ssaBag->getCount(); i++)
      {
         const GString& keyName = ssaBag->getKey(i);
         const GString& paramBuff = ssaBag->getIndexedItem(i);
         bool viaMain = (paramBuff[0] == '\0' || paramBuff[0] != '0');
         bool viaPipe = (paramBuff[0] == '\0' || paramBuff[1] != ',' || paramBuff[2] != '0');
         LCmdSepSessionApps* app = new LCmdSepSessionApps(viaMain, viaPipe);
         sepSessionApps.put(keyName, app);
      }
   }
   else
   {
      // Add a few default programs that are known to be incompatible with
      // the console monitor of Larsen Commander.
      for (int i=0; knownSSA[i].name != null; i++)
      {
         LCmdSepSessionApps* val = new LCmdSepSessionApps(knownSSA[i].viaMain, knownSSA[i].viaPipe);
         const GString& name = knownSSA[i].name;
         sepSessionApps.put(name, val);
      }
   }

   // Command History
   sn = "{CommandHistory}";
   for (int i=1; i<=MAXCMDHIST; i++)
   {
      GString cmd = ini.getString(sn, GString("%04d", GVArgs(i)));
      if (cmd == "")
         break;
      cmdHist.add(cmd);
   }

   // Directory History
   sn = "{DirectoryHistory}";
   for (int i=1; i<=MAXDIRHIST; i++)
   {
      GString dir = ini.getString(sn, GString("%04d", GVArgs(i)));
      if (dir == "")
         break;
      dirHist.add(dir);
   }

   GMultiLineEditor::queryProfile(sectName);
}

int LCmdCmdLineEntry::getRunningChildrenCount () const 
{ 
   return runningCounter; 
}

int LCmdCmdLineEntry::getIndexOfNewestChild ()
{
   const int num = runningChildren.getCount();
   for (int i=num-1; i>=0; i--)
   {
      LCmdProcessLauncher& child = runningChildren[i];
      if (child.isRunningChild())
         return i;
   }

   return -1;
}

LCmdCmdLineEntry& LCmdCmdLineEntry::GetCmdLineEntry ()
{
   return lcmd->cmdLine.entry;
}

void LCmdCmdLineEntry::breakChildProg ( GSystem::BreakType defaultBt )
{
   if (runningCounter <= 0)
   {
      // No running child, so there are nothing to do!
      // Why did the user press CTRL+C, then? ;-)
   }
   else
   {
      // Ask the user which child process to break, even if there is only
      // one child process.
      GSystem::BreakType bt = defaultBt;
      bool breakTree = true;
      LCmdProcessLauncher* pl = selectChildProcess(&bt, &breakTree);
      if (pl != null && pl->isRunningChild())
      {
         if (!pl->breakChildProg(bt, breakTree))
         {
            GStringl txt("%ErrBreakChildProcess");
            cmdLine.conmon.appendTextFromGuiThread("%s\n", GVArgs(txt));
         }
      }
   }
}

void LCmdCmdLineEntry::garbageCollectChildren ()
{
   const int num = runningChildren.getCount();
   for (int i=num-1; i>=0; i--)
   {
      LCmdProcessLauncher& child = runningChildren[i];
      if (child.hasFinished() && !child.isRunning())
         runningChildren.remove(i);
   }
}

LCmdSepSessionApps* LCmdCmdLineEntry::getNoneConsoleProgram ( const GString& progPath )
{
   // First, test if the program is contained in the bag of incompatible
   // SSA (Separate Session Application) programs as is.
   LCmdSepSessionApps *app = sepSessionApps.get(progPath);
   if (app != null)
      return app;

   // Separate the program file name and extension.
   GFile file(progPath);

   // Test if the program without directory nor extension is contained in
   // the bag of incompatible SSA programs.
   GString prg = file.getName();
   app = sepSessionApps.get(prg);
   if (app != null)
      return app;

   // Test if the program without directory, but with extension, is
   // contained in the bag of incompatible SSA programs.
   prg += file.getExtension();
   app = sepSessionApps.get(prg);
   if (app != null)
      return app;

   // The specified program was not found to be incompatible with the
   // Larsen Commander console monitor.
   return null;
}

const char* LCmdCmdLineEntry::FindNext ( const char* str, const char* chr )
{
   bool quoted = false;

   for (; *str; str++)
   {
      if (*str == '"')
      {
         quoted = !quoted;
         continue;
      }

      if (quoted)
         continue;

      for (const char* ptr = chr; *ptr; ptr++)
         if (*str == *ptr)
            return str;
   }

   return null;
}

bool LCmdCmdLineEntry::isNoneConsoleCommand ( const GString& progPath, const GString& params, bool testPipedCommandsOnly )
{
   // First, test if the program it self is contained in the bag of
   // incompatible SSA programs as is.
   if (!testPipedCommandsOnly)
   {
      LCmdSepSessionApps* app = getNoneConsoleProgram(progPath);
      if (app != null)
         return app->viaMain;
   }

   // If the parameter string does not contain any pipe characters then we
   // can assume that the specified program does not need to be forced
   // to run in a new session.
   const char *nextPipePos = FindNext(params.cstring(), "|<>");
   if (nextPipePos == null)
      return false;

   // Test if the command is to be redirected (via a pipe character) to
   // a program that is contained in the bag of incompatible SSA programs.
   GProgram& oprg = GProgram::GetProgram();
   GEnvironment& env = oprg.getEnvironmentVars();
   GString prg(128);
   do
   {
      nextPipePos++; // Skip the pipe character it self
      while (isspace(*nextPipePos)) // Skip all white characters (spaces, tabs, etc.)
         nextPipePos++;
      prg.clear();
      bool isQuoted = (*nextPipePos == '"');
      if (isQuoted)
         nextPipePos++; // Skip the quote character it self
      while(*nextPipePos)
      {
         if (isQuoted && *nextPipePos == '"')
            break;
         else
         if (!isQuoted && isspace(*nextPipePos))
            break;
         else
            prg += *nextPipePos++;
      }

      if (!GFile::ContainsExtension(prg))
      {
         // Command to the right of the pipe character is specified without
         // an extension, so we should find its exact location by
         // using  the WHICH-algorithm. This is to find out the type of the
         // command. That is, wether it is a EXE-, COM-, CMD- or a BAT-file.
         GString fullPath = env.which(prg);
         if (fullPath != "")
            prg = GFile(fullPath).getFileName();
      }

      // Check if the command is one of the defined SSA-apps, with
      // the extension.
      LCmdSepSessionApps* app = getNoneConsoleProgram(prg);
      if (app != null)
         return app->viaPipe;

      // Check if the command is one of the defined SSA-apps, without
      // the extension.
      GString fileName = GFile(prg).getName();
      app = getNoneConsoleProgram(fileName);
      if (app != null)
         return app->viaPipe;

      // Continue by testing the command behinf the next pipe character (if any)
      nextPipePos = FindNext(nextPipePos, "|<>");
   }
   while (nextPipePos != null);

   // The redirected program was not found in the table of SSA-apps, so
   // we should accept to try to run the command within the console monitor
   // of Larsen Commander.
   return false;
}

void LCmdCmdLineEntry::showHelpForCommand ( const GString& cmdName )
{
   InternalCommand* ic = internalCommands.get(cmdName);
   if (ic != null)
   {
      const GString& hlp = ic->getLongHelp();
      cmdLine.conmon.appendTextFromGuiThread(hlp);
   }
}

void LCmdCmdLineEntry::autoCompleteFileName ( bool foreward )
{
   replaceSelection(GString::Empty); // Remove selection (if any).
   GString text = getText();
   int curpos = getCaretPosition();
   int selstart, selend;

   // If the caret is currently inside a pair of quotes then we should
   // replace all characters inside that pair of quotes, including 
   // the quotes them selves. But only if the first quote is immediately 
   // ahead of the prefix string that we used when finding next match.
   GString prefix;
   bool hadQuotes;
   int quote1 = text.substring(0, curpos).lastIndexOf('"');
   int quote2 = text.indexOf('"', curpos);
   if (quote1 >= 0 && quote2 >= 0)
   {
      hadQuotes = true;
      prefix = text.substring(quote1 + 1, curpos);
      selstart = quote1;
      selend = quote2 + 1;
   }
   else
   {
      hadQuotes = false;
      prefix = getPartOfWordBeforeCaret();
      selstart = curpos - prefix.length();
      selend = curpos;

      // Find the position of the end of the word at the current position.
      for (int len=text.length(); selend<len; selend++)
      {
         char c = text[selend];
         if (isspace(c) || GFile::IsSlash(c) || c == '"')
            break;
      }
   }   

   LCmdFilePanel& fpanel = *lcmd->curPanel;
   int idx = fpanel.getCurrentSelectedIndex();
   if (foreward)
      idx = fpanel.findNextMatchingItem(prefix, idx);
   else
      idx = fpanel.findPrevMatchingItem(prefix, idx);
   
   if (idx < 0)
      return; // No more selectable items matching the prefix string.

   if (!fpanel.selectItem(idx))
      return;

   // Select the word and replace it with the autoselected filename.
   setSelection(selstart, selend);
   bool quotesInserted = false;
   GString fname = fpanel.getCurItemName();
   if (fname.indexOf(' ') >= 0)
   {
      // Since the filename to insert contains one or more spaces
      // we are better enclosing it in quotes.
      fname.insert('"');
      fname.append('"', 1);
      quotesInserted = true;
   }
   replaceSelection(fname);

   // Move the caret to the original position as of where it was
   // when the "auto filename" command was executed. This makes it
   // possible for the user to execute the same command once more to
   // automatically get the next matching filename.
   if (quotesInserted && !hadQuotes)
      curpos += 1;
   else
   if (!quotesInserted && hadQuotes)
      curpos -= 1;
   int start = curpos;
   int end = curpos + (fname.length() - prefix.length());
   if (quotesInserted)
      end -= 2; // Don't include the trailing quote in selection.
   setSelection(start, end);
}

LCmdProcessLauncher* LCmdCmdLineEntry::selectChildProcess ( GSystem::BreakType* bt, bool* tree )
{
   bool pureSelectOnly = (bt == null || tree == null);
   bool treeLocal = (pureSelectOnly ? false : *tree);
   GSystem::BreakType btLocal = (pureSelectOnly ? GSystem::BT_CtrlC : *bt);
   LCmdDlgChildProcessSelector dlg(*this, btLocal, treeLocal);
   if (pureSelectOnly)
      dlg.setPureProcessSelectionMode(true);
   LCmdProcessLauncher* pl = dlg.execute(lcmd->mainWin);
   if (pl != null && !pureSelectOnly)
   {
      *bt = dlg.getSelectedBreakType();
      *tree = dlg.isBreakTree();
   }
   childProcessSelectorDlg = null; // The dialog is no longer active
   return pl;
}

void LCmdCmdLineEntry::updateCommandLineColours ()
{
   if (childWaitingForInput != null)
   {
      setForegroundColor(colorOpt.txtWaitForInput);
      setBackgroundColor(colorOpt.bckWaitForInput);
   }
   else
   if (getRunningChildrenCount() > 0)
   {
      setForegroundColor(colorOpt.txtDisabled);
      setBackgroundColor(colorOpt.bckDisabled);
   }
   else
   {
      setForegroundColor(colorOpt.entryTxt);
      setBackgroundColor(colorOpt.entryBck);
   }
}

bool LCmdCmdLineEntry::onUserMessage ( GUserMessage& msg )
{
   GString id = msg.getParam1String();
   if (id == "UM_PROCESSLAUNCHERTHREADHASSTARTED")
   {
      childWaitingForInput = null;
      setForegroundColor(colorOpt.txtDisabled);
      setBackgroundColor(colorOpt.bckDisabled);
      updateWindow();
      return true;
   }
   else
   if (id == "UM_PROCESSLAUNCHERTHREADHASFINISHED")
   {
      LCmdProcessLauncher* pl = (LCmdProcessLauncher*) msg.getParam2();
      // Remove the terminated child entry from the child process picker dialog.
      if (childProcessSelectorDlg != null)
         childProcessSelectorDlg->sendDialogMessage(GDialogMessageHandler::GM_USER, "RemoveItem", pl);
      // ---
      childWaitingForInput = null;
      updateCommandLineColours();
      // The icon is sometimes changed when running child programs
      // vis CMD.EXE. So set the icon in the system menu back to the
      // Larsen Commander icon.
      sendUserMessage("UM_RESTORE_LCMD_ICON", pl);
      return true;
   }
   else
   if (id == "UmRunAsShellObject")
   {
      GString prgName = msg.getParam2String();
      if (!GSystem::OpenShellObject(prgName))
      {
         GStringl errmsg("%Txt_Cmd_Err_LaunchObj", GVArgs(prgName));
         lcmd->conmon.appendTextFromGuiThread("%s\n", GVArgs(errmsg));
      }
      return true;
   }
   else
   if (id == "UmChildWaitsForStdInData")
   {
      childWaitingForInput = (LCmdProcessLauncher*) msg.getParam2();
      int pid = childWaitingForInput->getPidOfRunningChildProcess();
      GString cmd = childWaitingForInput->getOriginalCommand();
      GStringl statusTxt("%Txt_Cmd_ChildProcessWaitsForInput", GVArgs(pid).add(cmd));
      lcmd->mainWin.setStatusbarText(statusTxt);
      updateCommandLineColours();
      childWaitingForInput->guiUserMessageHandlerHasFinished();
      return true;
   }
   else
   if (id == "UM_CHILD_PROCESS_HAS_LAUNCHED")
   {
      // Update the statusbar cell that shows the number of child processes.
      lcmd->mainWin.updateStatusbarChildProcessCount();
      sendUserMessage("UM_RESTORE_LCMD_ICON");
      return true;
   }
   else
   if (id == "UM_CHILD_PROCESS_HAS_FINISHED")
   {
      // Update the statusbar cell that shows the number of child processes.
      lcmd->mainWin.updateStatusbarChildProcessCount();

      // A child process has finished, so reread both file panels.
      bool preventReread = msg.getParam2Bool();
      if (!preventReread)
      {
         lcmd->curPanel->getOppositePanel().reRead();
         lcmd->curPanel->reRead();
      }
      // The icon is sometimes changed when running child programs
      // vis CMD.EXE. So set the icon in the system menu back to the
      // Larsen Commander icon.
      sendUserMessage("UM_RESTORE_LCMD_ICON");
      lcmd->curPanel->activatePanelDriveAndDir(true);
      return true;
   }
   else
   if (id == "UM_RESTORE_LCMD_ICON")
   {
      const int ID_ICON1 = 1;
      HPOINTER hicon = WinLoadPointer(HWND_DESKTOP, null, ID_ICON1);
      if (hicon != null)
         lcmd->mainWin.getFrame().sendMessage(WM_SETICON, MPARAM(hicon));
      return true;
   }
   else
   {
      return GMultiLineEditor::onUserMessage(msg);
   }
}

bool LCmdCmdLineEntry::onKeyDown ( const GKeyMessage& key )
{
   switch (key.getCode())
   {
      case GKey::KEY_SHIFT_LEFT:
      case GKey::KEY_SHIFT_RIGHT:
      case GKey::KEY_SHIFT_HOME:
      case GKey::KEY_SHIFT_END:
      {
         if (!isEmpty())
         {
            GWindow& peer = getPeer();
            GWindowMessage omsg = key.getOriginalMessage();
            peer.callDefaultMsgProc(omsg);
            return true;
         }
         break;
      }

      case GKey::KEY_LEFT:
      case GKey::KEY_RIGHT:
      {
         LCmdFilePanel& fpanel = LCmdFilePanel::GetCurrentPanel();
         LCmdFilePanelModeAbstract& curView = fpanel.getCurrentView();
         if (lcmd->options.panelsAreVisible && curView.isHorizontallyScrollable())
            return callDefaultKeyHandler(key, false, false);
         break;
      }

      case GKey::KEY_END:
      case GKey::KEY_HOME:
      {
         if (lcmd->options.panelsAreVisible)
            return callDefaultKeyHandler(key, false, false);
         break;
      }

      case GKey::KEY_DELETE:
      {
         int len = getTextLength();
         if (len == 0 && lcmd->options.fileDel.useDelKey)
         {
            lcmd->mainWin.cmdDeleteFile();
            return true;
         }
         if (!isAnySelectedText())
         {
            // When there is no selected text we want the delete-key to be
            // handled directly by the text entry field. Else we pass
            // it to our super class, which checks it against accelerator
            // keys as normal. Larsen Commander defines an accelerator 
            // command "cmsClear" which is usually mapped to this keyboard
            // event. It will delete the current selected text.
            GWindow& peer = getPeer();
            GWindowMessage omsg = key.getOriginalMessage();
            peer.callDefaultMsgProc(omsg);
            return true;
         }
         break;
      }
   }

   return GMultiLineEditor::onKeyDown(key);
}

bool LCmdCmdLineEntry::onBackgroundColorChanged ( const GColor& color )
{
   if (childWaitingForInput != null)
      colorOpt.bckWaitForInput = color;
   else
   if (getRunningChildrenCount() > 0)
      colorOpt.bckDisabled = color;
   else
      colorOpt.entryBck = color;
   GMultiLineEditor::onBackgroundColorChanged(color);
   return true;
}

bool LCmdCmdLineEntry::onForegroundColorChanged ( const GColor& color )
{
   if (childWaitingForInput != null)
      colorOpt.txtWaitForInput = color;
   else
   if (getRunningChildrenCount() > 0)
      colorOpt.txtDisabled = color;
   else
      colorOpt.entryTxt = color;
   GMultiLineEditor::onForegroundColorChanged(color);
   return true;
}

bool LCmdCmdLineEntry::onInitMenu ()
{
   int textLen = getTextLength();
   int childCount = getRunningChildrenCount();
   bool anySelected = isAnySelectedText();
   bool anyTextOnClipbrd = GSystem::IsAnyTextOnClipboard();
   bool insmodeIsOn = getInsertMode();

   setCommandEnableState("cmdCopy", anySelected);
   setCommandEnableState("cmdCopyAllCommandLineText", textLen > 0);
   setCommandEnableState("cmdCut", anySelected);
   setCommandEnableState("cmdPaste", anyTextOnClipbrd);
   setCommandEnableState("cmdClear", anySelected);
   setCommandEnableState("cmdSelectAll", textLen > 0);
   setCommandEnableState("cmdNavigateEnter", textLen > 0);
   setCommandEnableState("cmdRunCommandInNewSession", textLen > 0);
   setCommandEnableState("cmdFeedChildProcessStdIn", textLen > 0 && childCount > 0);
   setCommandEnableState("cmdScrollCmdHistUp", cmdHist.getCursor() < cmdHist.getStringCount());
   setCommandEnableState("cmdScrollCmdHistDown", cmdHist.getCursor() > 0);
   setCommandEnableState("cmdCommandHistory", true);

   setCommandToggleState("cmdToggleInsertMode", insmodeIsOn);

   return true;
}

void LCmdCmdLineEntry::paste ()
{
   GString txt(GSystem::GetClipboardText());
   int pos1 = txt.indexOf('\r');
   int pos2 = txt.indexOf('\n');
   int nlpos = -1;

   if (pos1 >= 0 || pos2 >= 0)
      nlpos = GMath::Min(pos1, pos2);

   if (nlpos == 0)
   {
      GSystem::Beep();
   }
   else
   if (nlpos >= 1)
   {
      replaceSelection(txt.cutTailFrom(nlpos));
   }
   else
   {
      GMultiLineEditor::paste();

      // In order to make sure that the caret is within visible area we have
      // to do the below statements. It is probably a bug in the MLE Control
      // of OS/2 that causes this.
      moveCursorLeft();
      moveCursorRight();
   }
}

void LCmdCmdLineEntry::cmdToggleInsertMode ( GAbstractCommand* /*cmd*/ )
{
   setInsertMode(!getInsertMode());
}

void LCmdCmdLineEntry::cmdMoveCaretRight ( GAbstractCommand* /*cmd*/ )
{
   moveCursorRight();
}

void LCmdCmdLineEntry::cmdMoveCaretLeft ( GAbstractCommand* /*cmd*/ )
{
   moveCursorLeft();
}

void LCmdCmdLineEntry::cmdMoveCaretHome ( GAbstractCommand* /*cmd*/ )
{
   moveCursorHome();
}

void LCmdCmdLineEntry::cmdMoveCaretEnd ( GAbstractCommand* /*cmd*/ )
{
   moveCursorEnd();
}

void LCmdCmdLineEntry::cmdMoveCaretWordRight ( GAbstractCommand* /*cmd*/ )
{
   moveCursorWordRight();
}

void LCmdCmdLineEntry::cmdMoveCaretWordLeft ( GAbstractCommand* /*cmd*/ )
{
   moveCursorWordLeft();
}

void LCmdCmdLineEntry::cmdScrollConMonHome ( GAbstractCommand* /*cmd*/ )
{
   if (cmdLine.conmon.getIndexOfFirstVisibleColumn() > 0)
      cmdLine.conmon.onHScrollSliderTrack(0);
}

void LCmdCmdLineEntry::cmdScrollConMonTop ( GAbstractCommand* /*cmd*/ )
{
   cmdLine.conmon.onVScrollSliderTrack(0);
}

void LCmdCmdLineEntry::cmdScrollConMonEnd ( GAbstractCommand* /*cmd*/ )
{
   if (cmdLine.conmon.getIndexOfFirstVisibleColumn() <= cmdLine.conmon.getWidestLine() - cmdLine.conmon.getWindowVisibleColumnsCount())
      cmdLine.conmon.onHScrollSliderTrack(cmdLine.conmon.getWidestLine() - cmdLine.conmon.getWindowVisibleColumnsCount() + 1);
}

void LCmdCmdLineEntry::cmdScrollConMonBottom ( GAbstractCommand* /*cmd*/ )
{
   if (cmdLine.conmon.getLinesCount() > cmdLine.conmon.getWindowVisibleLinesCount())
      cmdLine.conmon.onVScrollSliderTrack(cmdLine.conmon.getLinesCount() - cmdLine.conmon.getWindowVisibleLinesCount() + 1);
}

void LCmdCmdLineEntry::cmdScrollConMonUp ( GAbstractCommand* /*cmd*/ )
{
   cmdLine.conmon.onVScrollLineUp();
}

void LCmdCmdLineEntry::cmdScrollConMonDown ( GAbstractCommand* /*cmd*/ )
{
   cmdLine.conmon.onVScrollLineDown();
}

void LCmdCmdLineEntry::cmdScrollConMonLeft ( GAbstractCommand* /*cmd*/ )
{
   if (cmdLine.conmon.getIndexOfFirstVisibleColumn() > 0)
      cmdLine.conmon.onHScrollSliderTrack(cmdLine.conmon.getIndexOfFirstVisibleColumn() - 5);
}

void LCmdCmdLineEntry::cmdScrollConMonRight ( GAbstractCommand* /*cmd*/ )
{
   if (cmdLine.conmon.getWidestLine() > cmdLine.conmon.getWindowVisibleColumnsCount())
      if (cmdLine.conmon.getIndexOfFirstVisibleColumn() < cmdLine.conmon.getWidestLine() - cmdLine.conmon.getWindowVisibleColumnsCount())
         cmdLine.conmon.onHScrollSliderTrack(cmdLine.conmon.getIndexOfFirstVisibleColumn() + 5);
}

void LCmdCmdLineEntry::cmdScrollConMonPageUp ( GAbstractCommand* /*cmd*/ )
{
   cmdLine.conmon.onVScrollPageUp();
}

void LCmdCmdLineEntry::cmdScrollConMonPageDown ( GAbstractCommand* /*cmd*/ )
{
   cmdLine.conmon.onVScrollPageDown();
}

void LCmdCmdLineEntry::cmdClear ( GAbstractCommand* /*cmd*/ )
{
   replaceSelection(GString::Empty);
}

void LCmdCmdLineEntry::cmdCopy ( GAbstractCommand* /*cmd*/ )
{
   copy();
}

void LCmdCmdLineEntry::cmdCopyAllCommandLineText ( GAbstractCommand* /*cmd*/ )
{
   GString txt(getText());
   GSystem::CopyTextToClipboard(txt);
}

void LCmdCmdLineEntry::cmdCut ( GAbstractCommand* /*cmd*/ )
{
   cut();
}

void LCmdCmdLineEntry::cmdPaste ( GAbstractCommand* /*cmd*/ )
{
   paste();
}

void LCmdCmdLineEntry::cmdSelectAll ( GAbstractCommand* /*cmd*/ )
{
   selectAll();
}

void LCmdCmdLineEntry::replaceSelection ( const GString& text )
{
   if (cmdLine.isVisible())
      GMultiLineEditor::replaceSelection(text);
}

bool LCmdCmdLineEntry::executeHistoricCommand ( int index, const GString& workDir, int flags )
{
   // We must get a copy of the command from the history, rather than
   // getting a reference. This is in case the command is the same as will
   // be automatically removed and deleted from the history list before this
   // command which is re-executed is appended to the end if the history list.
   GString cmd = cmdHist.getIndexedString(index);
   return executeCommand(cmd, workDir, flags);
}

bool LCmdCmdLineEntry::executeCommand ( const GString& originalCmd,
                                        const GString& workDir,
                                        const GString& prgName,
                                        const GString& paramStr,
                                        int flags )
{
   GCmdLineParameters params(paramStr);
   bool forceNewSession = ((flags & ECF_FORCE_NEW_SESSION) != 0);
   bool doEcho = ((flags & ECF_DO_ECHO) != 0);
   bool forceRunViaShell = ((flags & ECF_FORCE_RUN_VIA_SHELL) != 0);
   bool closeOnExit = ((flags & ECF_CLOSEON_EXIT) != 0);
   bool dontAddToCmdHist ((flags & ECF_DONT_ADD_TO_CMD_HIST) != 0);

   if (GLog::Filter(GLog::TEST))
      GLog::Log(this, "%s: originalCmd='%s'", GVArgs(__FUNCTION__).add(originalCmd));

   if (forceRunViaShell)
   {
      if (doEcho)
      {
         GString cd = lcmd->curPanel->getCurrentSysDirectory(false);
         GString txt("\n%s>%s\n", GVArgs(cd).add(originalCmd));
         cmdLine.conmon.appendTextFromGuiThread(txt);
      }

      // Add the command to the Command History List.
      if (!dontAddToCmdHist)
         cmdHist.add(originalCmd);
   }
   else
   {
      // First, walk through the number of available internal commands. These
      // are commands that will be parsed, recognized and executed internally
      // by Larsen Commander without invoking any secondary OS/2 process.

      bool disableIt = false;
      InternalCommand *ic = internalCommands.get(prgName);
      if (ic != null && !ic->isTransferToShell())
      {
         disableIt = ic->isDisabled();
         if (!disableIt)
         {
            if (params.isContainingHelpRequest())
            {
               // Add the command to the Command History List.
               if (!dontAddToCmdHist)
                  cmdHist.add(originalCmd);

               if (doEcho)
               {
                  // Format and send the text to a temporary background thread
                  // which only purpose is to send the text to the console
                  // reader thread.
                  GString cd = lcmd->curPanel->getCurrentSysDirectory(false);
                  GString txt("\n%s>%s\n", GVArgs(cd).add(originalCmd));
                  cmdLine.conmon.appendTextFromGuiThread(txt);
               }

               showHelpForCommand(prgName);
               return true;
            }
         }
      }

      // Get an uppercase copy of the program/command name just to simplify
      // the logic needed to parse the command name.
      GString upperCasePrgName = prgName;
      upperCasePrgName.toUpperCase();

      // ---
      // Is it a command to repeat a recent command? - R or REP
      if (!disableIt && (upperCasePrgName == "R" || upperCasePrgName == "REP"))
      {
         int ccount = cmdHist.getStringCount();
         if (ccount <= 0)
         {
            // "No commands in the history list!\n"
            GStringl txt("%Txt_Cmd_HIST_NoCmdInHist");
            cmdLine.conmon.appendTextFromGuiThread(txt);
            return true;
         }

         int histFlags = ECF_DO_ECHO;
         if (forceNewSession)
            histFlags |= ECF_FORCE_NEW_SESSION;
         if (closeOnExit)
            histFlags |= ECF_CLOSEON_EXIT;

         if (paramStr == "")
         {
            return executeHistoricCommand(ccount - 1, workDir, histFlags);
         }
         else
         {
            const int plen = paramStr.length();
            for (int i=ccount-1; i>=0; i--)
            {
               const GString& hstr = cmdHist.getIndexedString(i);
               if (paramStr.equalsIgnoreCaseNum(hstr, plen))
                  return executeHistoricCommand(i, workDir, histFlags);
            }

            // The specified string does not match any of the commands in the
            // history list, so see if the specified string is a number that
            // can possibly be an index into the history list.
            for (int i=0; i<paramStr.length(); i++)
            {
               if (!isdigit(paramStr[i]))
               {
                  // "Specified text does not match any command in the history list!\n"
                  GStringl txt("%Txt_Cmd_REP_NoMatch");
                  cmdLine.conmon.appendTextFromGuiThread(txt);
                  return true;
               }
            }

            int idx = GInteger::ParseInt(paramStr);
            if (idx <= 0 || idx > ccount)
            {
               // "Illegal index!\n"
               GStringl txt("%Txt_Cmd_REP_IllegalIndex");
               cmdLine.conmon.appendTextFromGuiThread(txt);
               return true;
            }

            return executeHistoricCommand(ccount - idx, workDir, histFlags);
         }
      }

      // ---
      // Is it a Clear Screen Command? - CLS or CLEAR
      if (!disableIt && (upperCasePrgName == "CLS" || upperCasePrgName == "CLEAR"))
      {
         if (!dontAddToCmdHist)
            cmdHist.add(originalCmd); // Add the command to the Command History List.
         cmdLine.conmon.removeAllText();
         lcmd->mainWin.setStatusbarText(GString::Empty);

         // Return as much as possible of the unused space on the heap to the OS
         _heapmin();

         return true;
      }

      if (doEcho)
      {
         // Format and send the text to the console reader thread.
         GString cd = lcmd->curPanel->getCurrentSysDirectory(false);
         GString txt("\n%s>%s\n", GVArgs(cd).add(originalCmd));
         cmdLine.conmon.appendTextFromGuiThread(txt);
      }

      // ---
      // Add the command to the Command History List, but only of the
      // command it self is not a "History" command (and of course not 
      // if caller has requested us not to add the command to the history).
      if (!dontAddToCmdHist && upperCasePrgName != "H" && upperCasePrgName != "HIST")
         cmdHist.add(originalCmd); // Add the command to the Command History List.

      // ---
      // Is it a Drive Change Command? - X:
      if (originalCmd.length() == 2 && isalpha(originalCmd[0]) && originalCmd[1] == ':')
      {
         char drv = originalCmd[0];
         lcmd->curPanel->activateDrive(drv);
         return true;
      }

      // ---
      // Is it an internal command that should be executed by the program
      // launcher thread?
      if (!disableIt &&
          (upperCasePrgName == "ALIAS" ||
           upperCasePrgName == "FINDDUP" ||
           upperCasePrgName == "HELP" || upperCasePrgName == "?" ||
           upperCasePrgName == "H" || upperCasePrgName == "HIST" ||
           upperCasePrgName == "INFO" ||
           upperCasePrgName == "WHICH"))
      {
         flags |= ECF_IS_INTERNAL;
      }

      // ---
      // Is it a Change Directory Command? - CD or CHDIR
      if (!disableIt && (upperCasePrgName == "CD" || upperCasePrgName == "CHDIR"))
      {
         GString dirto;
         const int count = params.getCount();
         if (count == 0)
         {
            GString cd = lcmd->curPanel->getCurrentSysDirectory(false);
            cmdLine.conmon.appendTextFromGuiThread("%s\n", GVArgs(cd));
         }
         else
         {
            if (count == 1)
               dirto = params.getIndexedString(0); // Don't include quotes, if any.
            else
               dirto = paramStr; // Probably a directory which name has space(s).

            if (dirto.length() == 2 && isalpha(dirto[0]) && dirto[1] == ':')
            {
               try {
                  // Display the current directory of the specified drive.
                  int drive = GCharacter::ToUpperCase(dirto[0]) - 'A' + 1;
                  GString cd = GFile::GetCurrentDir(drive);
                  cmdLine.conmon.appendTextFromGuiThread("%s\n", GVArgs(cd));
               } catch (APIRET& rc) {
                  GString msg = GSystem::GetApiRetString(rc);
                  cmdLine.conmon.appendTextFromGuiThread("%s\n", GVArgs(msg));
               }
            }
            else
            {
               // Activate the specified directory.
               lcmd->curPanel->walkDir(dirto);
            }
         }
         return true;
      }

      // ---
      // Is it a Push Directory Command? - PUSHD
      if (!disableIt && upperCasePrgName == "PUSHD")
      {
         GString curdir = lcmd->curPanel->getCurrentSysDirectory(false);

         const int count = params.getCount();
         if (count == 0)
         {
            // PUSHD with no arguments pushes the current directory
            dirStack.add(new GString(curdir));
            return true;
         }

         GString dirto;
         if (count == 1)
            dirto = params.getIndexedString(0); // Don't include quotes, if any.
         else
            dirto = paramStr; // Probably a directory which name has space(s).

         if (lcmd->curPanel->walkDir(dirto))
            dirStack.add(new GString(curdir));

         return true;
      }

      // ---
      // Is it a Pop Directory Command? - POPD
      if (!disableIt && upperCasePrgName == "POPD")
      {
         switch (params.getCount())
         {
            case 0: {
                 if (dirStack.getCount() <= 0)
                 {
                    // "No more items in the Directory Stack.\n"
                    GStringl txt("%Txt_Cmd_POPD_EmptyDirStack");
                    cmdLine.conmon.appendTextFromGuiThread(txt);
                    return true;
                 }
                 const int num = dirStack.getCount();
                 const GString& dir = dirStack.get(num - 1);
                 lcmd->curPanel->walkDir(dir);
                 dirStack.remove(num - 1);
                 return true; }

            case 1:
                 if (params.getIndexedString(0) == "*")
                 {
                    dirStack.removeAll();
                    return true;
                 }
                 else
                 if (params.getIndexedString(0) == "?")
                 {
                    // Print the current directory stack. This must be done
                    // by the background thread, to prevent deadlock due to
                    // limited PIPE buffer size.
                    const int num = dirStack.getCount();
                    if (num <= 0)
                    {
                       GString txt("No items in the Directory Stack.\n");
                       cmdLine.conmon.appendTextFromGuiThread(txt);
                       return true;
                    }
                    flags |= ECF_IS_INTERNAL;
                    break;
                 }

                 // Yes, continue with the next case ...

            default:
                 showHelpForCommand(prgName);
                 return false;
         }
      }

      // ---
      // Is it a command to Exit Larsen Commander? - EXIT
      if (!disableIt && upperCasePrgName == "EXIT")
      {
         lcmd->mainWin.cmdCloseAndExit();
         return true;
      }

      // ---
      // Is it a command to Tag Files and/or Directories? - TAG
      if (!disableIt && upperCasePrgName == "TAG")
      {
         switch (params.getCount())
         {
            case 0:
                 showHelpForCommand(prgName);
                 return false;

            default: {
                 const int num = params.getCount();
                 for (int i=0; i<num; i++)
                 {
                    LCmdFilePanelSelectOptions tag;
                    tag.filter = params.getIndexedString(i);
                    tag.inclFiles = lcmd->curPanel->selectOpt.inclFiles;
                    tag.inclDirs = lcmd->curPanel->selectOpt.inclDirs;
                    lcmd->curPanel->select(tag);
                 }
                 return true; }
         }
      }

      // ---
      // Is it a command to Untag Files and/or Directories? - UNTAG
      if (!disableIt && upperCasePrgName == "UNTAG")
      {
         switch (params.getCount())
         {
            case 0:
                 showHelpForCommand(prgName);
                 return false;

            default: {
                 const int num = params.getCount();
                 for (int i=0; i<num; i++)
                 {
                    LCmdFilePanelSelectOptions tag;
                    tag.filter = params.getIndexedString(i);
                    tag.inclFiles = lcmd->curPanel->selectOpt.inclFiles;
                    tag.inclDirs = lcmd->curPanel->selectOpt.inclDirs;
                    lcmd->curPanel->unselect(tag);
                 }
                 return true; }
         }
      }

      // ---
      // Is it a command to show version information? - VER
      if (!disableIt && upperCasePrgName == "VER")
      {
         // "\nLarsen Commander %s\n%s\n"
         GString verStr = lcmd->aboutBox.getVersionString();
         GString copyRight = lcmd->aboutBox.getCopyrightString();
         GStringl txt("%Txt_Cmd_VER_Ver", GVArgs(verStr).add(copyRight));
         cmdLine.conmon.appendTextFromGuiThread(txt);
      }

      // ---
      // Is it a Set Environment Variable Command? - SET
      if (!disableIt && upperCasePrgName == "SET")
      {
         if (paramStr.indexOf('=') >= 0)
         {
            lcmd->setEnvironmentVar(paramStr);
            return true;
         }
         else
         if (paramStr != "")
         {
            GString envname = paramStr;
            // TODO: envname.toUpperCase();
            GString envval = lcmd->getEnvironmentVar(envname);
            cmdLine.conmon.appendTextFromGuiThread("%s=%s\n", GVArgs(envname).add(envval));
            return true;
         }
      }

      // ---
      // Is it a Define Alias command? - ALIAS
      if (!disableIt && upperCasePrgName == "ALIAS")
      {
         if (paramStr.indexOf('=') >= 0)
         {
            aliases.defineAlias(paramStr);
            return true;
         }
         else
         if (paramStr != "")
         {
            GString aliasName = paramStr;
            aliasName.toUpperCase();
            const GString* aliasStr = aliases.getAliasStr(paramStr);
            if (aliasStr == null)
            {
               // "Undefined alias: %s\n"
               GStringl txt("%Txt_Cmd_ALIAS_UndefinedAlias", GVArgs(aliasName));
               cmdLine.conmon.appendTextFromGuiThread(txt);
            }
            else
            {
               cmdLine.conmon.appendTextFromGuiThread("%s=%s\n", GVArgs(aliasName).add(*aliasStr));
            }
            return true;
         }
      }

      // ---
      // Is it a Calculator command? - CALC
      if (!disableIt && upperCasePrgName == "CALC")
      {
         // Use the original command string, which includes quotes (if any).
         // The is in order to make it easy for the user to enter expressions
         // with string types. For instance, by doing this the user can type
         //    calc "test" + 1
         // instead of having to type
         //    calc "'test' + 1"
         GString expr = originalCmd;
         expr.trim();
         if(toupper(expr[0]) == upperCasePrgName[0]) // In case the CALC token was given in a pair of quotes.
         {
            expr.remove(0, upperCasePrgName.length()); // Remove the "CALC" token it self.
            expr.trim();
         }
         else
         {
            expr = paramStr;
         }
         if (expr == "")
         {
            // "No expression. Try CALC -? or CALC /?\n"
            GStringl txt("%Txt_Cmd_CALC_NoExpr");
            cmdLine.conmon.appendTextFromGuiThread(txt);
         }
         else
         try {
            aptr<GExpressionParser::RValue> res = GExpressionParser::GetParsedExpression(expr);
            GString resstr = res->getString();
            cmdLine.conmon.appendTextFromGuiThread("= %s\n", GVArgs(resstr));
         } catch (GSyntaxErrorException& e) {
            int pos = e.getColumnPos();
            GString tag(pos + 1);
            tag.append(' ', pos);
            tag += '^';
            GString txt = expr + "\n" + tag + "\n" + e.toString();
            cmdLine.conmon.appendTextFromGuiThread(txt);
         }
         return true;
      }

      // ---
      // Is it a command to toggle the ECHO mode? - ECHO
      if (!disableIt && upperCasePrgName == "ECHO")
      {
         GStringl txtOn("%Txt_Cmd_ECHO_On");
         GStringl txtOff("%Txt_Cmd_ECHO_Off");
         if (paramStr.equalsIgnoreCase("ON") || paramStr.equalsIgnoreCase(txtOn))
         {
            echoIsOn = true;
            return true;
         }
         else
         if (paramStr.equalsIgnoreCase("OFF") || paramStr.equalsIgnoreCase(txtOff))
         {
            echoIsOn = false;
            return true;
         }
         else
         if (paramStr == "")
         {
            // "ECHO is %s\n"
            GStringl txt("%Txt_Cmd_ECHO_EchoIsOnOrOff", GVArgs(echoIsOn ? txtOn : txtOff));
            cmdLine.conmon.appendTextFromGuiThread(txt);
            return true;
         }
      }
   }

   // Execute the command in a background process launcher thread.
   // This thread will post a UM_PROCESSLAUNCHERTHREADHASFINISHED message
   // to the command line entry as soon as the child process has finished.
   // It is up to the launcher thread to decide whether the new
   // task should run as a child process or not.
   bool isInternal = ((flags & ECF_IS_INTERNAL) != 0);
   bool runAsObject = ((flags & ECF_RUN_AS_SHELL_OBJECT) != 0);
   LCmdProcessLauncher* child = new LCmdProcessLauncher(*this, originalCmd, workDir, prgName, paramStr, forceNewSession, isInternal, runAsObject, closeOnExit, forceRunViaShell);
   runningChildren.add(child);
   try {
      child->start();
      // The child will delete it self when thread has finished.
      return true;
   } catch (GThreadStartException& e) {
      GString errmsg = e.toString();
      GString stackTrace = e.getStackTrace(errmsg);
      cmdLine.conmon.appendTextFromGuiThread(stackTrace);
      GLog::Log(this, stackTrace);
      delete child;
      return false;
   }
}

bool LCmdCmdLineEntry::executeCommand ( const GString& cmd,
                                        const GString& workDir,
                                        int flags )
{
   garbageCollectChildren();

   GEnvironment& env = lcmd->getEnvironmentVars();
   GString fullCommand = aliases.processString(*lcmd->curPanel, cmd, &env, workDir);
   bool forceRunViaShell = ((flags & ECF_FORCE_RUN_VIA_SHELL) != 0);

   // Skip all prefix blanks and @-characters.
   if (!echoIsOn)
      flags &= ~ECF_DO_ECHO;
   fullCommand.trim();
   while (!fullCommand.isEmpty() && fullCommand[0] == '@')
   {
      fullCommand.removeFirstChar();
      flags &= ~ECF_DO_ECHO;
   }

   // Fetch out program/command name of the specified command string.
   GString prgName(128);
   bool isQuoted = false;
   const char *pcCmd = fullCommand;
   char *ptr = (char *) pcCmd;
   for (; *ptr; ptr++)
   {
      if (*ptr == '"')
         isQuoted = !isQuoted;
      else
      if (*ptr == '|' || *ptr == '<' || *ptr == '>')
      {
         forceRunViaShell = true;
         break;
      }
      else
      if ((*ptr == ' ' ||
          (*ptr == '\\' && (ptr - pcCmd) == 2 && strnicmp("cd\\", pcCmd, 3) == 0) ||
          (*ptr == '.' && (ptr - pcCmd) == 2 && strnicmp("cd..", pcCmd, 4) == 0)) && !isQuoted)
      {
         break;
      }
      else
         prgName += *ptr;
   }

   while (isspace(*ptr))
      ptr++;

   // Fetch out the parameter(s) string of the specified command string.
   GString paramStr(256);
   paramStr = ptr;
   isQuoted = false;
   for (ptr = (char *) pcCmd; *ptr && !forceRunViaShell; ptr++)
   {
      if (*ptr == '"')
         isQuoted = !isQuoted;
      else
      if (!isQuoted)
      {
         if (*ptr == '|' || *ptr == '<' || *ptr == '>')
         {
            forceRunViaShell = true;
            break;
         }
      }
   }

   if (forceRunViaShell)
      flags |= ECF_FORCE_RUN_VIA_SHELL;

   prgName.trim();
   paramStr.trim();

   // Execute the parsed command line.
   return executeCommand(cmd, workDir, prgName, paramStr, flags);
}

bool LCmdCmdLineEntry::feedChildProcessStdIn ( LCmdProcessLauncher* child, const GString& txt )
{
   if (txt == "")
      return true;
   if (getRunningChildrenCount() <= 0)
      return false;
   if (child == null)
      child = selectChildProcess(null, null);
   if (child == null)
      return false;
   int lcount = cmdLine.conmon.getLinesCount();
   cmdLine.conmon.gotoPos(lcount, 0);
   child->addTextToChildStdInQueue(txt);
   childWaitingForInput = null;
   updateCommandLineColours();
   return true;
}
