/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_ALIASES
#define __LCMD_ALIASES

#include "glib/util/GSectionBag.h"

/**
 * This class is the implementation of the code that has to do with
 * the support for aliases in command strings.
 *
 * @author  Leif Erik Larsen
 * @since   1999.05.07
 */
class LCmdAliases
{
   public:

      /** The actual bag of where the aliases are stored. */
      GKeyBag<GString> aliases;

   public:

      /**
       * Create a new and initially empty bag of aliases.
       */
      LCmdAliases ();

   public:

      /**
       * Test if the specified string is the name of an alias in
       * the aliases bag.
       *
       * @return True if there is an alias with the specified name,
       *         or else false.
       */
      bool isAlias ( const GString& name );

      /**
       * Add or update the specified alias in the aliases bag.
       *
       * There are a few aliases that are reserved by the system.
       * This means that they are not possible to be redefined by the
       * user. These are all aliases that begins with an exclamation mark.
       *
       * @return True if the alias was actually added to or updated in the
       *         bag of aliases, or else false on any error (for instance
       *         if the specified alias name is illegal).
       */
      bool defineAlias ( const GString& name, const GString& value );

      /**
       * Set the specified alias name and value of Larsen Commander.
       * The specified string must be of the form
       * <pre>
       *     "X=Y"
       * </pre>
       * where X is the alias name (will be converted to all uppercase
       * characters automatically) and Y is the alias value.
       */
      bool defineAlias ( const GString& script );

      /**
       * Get the current number of aliases that are contained in the bag.
       */
      int getCount ();

      /**
       * Get a reference to the name of the indexed alias.
       */
      const GString& getAliasName ( int index );

      /**
       * Get a reference to the alias value of the indexed alias.
       */
      const GString& getAliasStr ( int index );

      /**
       * Get a reference to the alias value of the named alias.
       * @return A pointer to the alias value, or NULL if there is no
       *         alias with the specified name.
       */
      const GString* getAliasStr ( const GString& name );

      /**
       * Remove and delete the named alias from the bag.
       */
      void undefineAlias ( const GString& name );

      void readProfile ( GSectionBag& ini );

      void writeProfile ( GSectionBag& ini, bool doIt );

      /**
       * Walk through the specified source string an convert all
       * occurences of the reqognized aliases.
       *
       * @return  The destination string.
       */
      GString processString ( class LCmdFilePanel& panel, 
                              const GString& src, 
                              class GEnvironment *envList = null, 
                              const GString& homeDir = GString::Empty );
};

#endif // #ifndef __LCMD_ALIASES
