/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_H
#define __LCMD_H

#define __ENABLE_TREE_VIEW 0 // Disable tree-view until it is fully functional.

#include "lcmd/LCmdMainWindow.h"
#include "lcmd/LCmdDlgChooseDrive.h"
#include "lcmd/LCmdInternalViewer.h"
#include "lcmd/LCmdDlgAbout.h"
#include "lcmd/LCmdDlgCopyFile.h"

#include "glib/GProgram.h"
#include "glib/sys/GSystemInfo.h"

/**
 * The Larsen Commander application program class it self.
 *
 * Below is a brief "map" of the various layers of window panels in LCMD.
 *
 * <pre>
 * LCmdMainWindow
 *   |--GToolbar
 *   |--LCmdSubClient
 *   |    |--LCmdPanelsParent
 *   |    |    |--LCmdFilePanelFrame
 *   |    |    |    |--LCmdFilePanelHeader
 *   |    |    |    |    |--LCmdFilePanelHeaderDriveButt
 *   |    |    |    |    |--LCmdFilePanelHeaderUpDirButt
 *   |    |    |    |    |--LCmdFilePanelHeaderRootButt
 *   |    |    |    |    +--GMultiLineEditor
 *   |    |    |    |--LCmdFilePanel
 *   |    |    |    |    +--GTabbedPanel
 *   |    |    |    |         |--LCmdFilePanelModeBrief
 *   |    |    |    |         |--LCmdFilePanelModeWide
 *   |    |    |    |         |--LCmdFilePanelModeFull
 *   |    |    |    |         |    |--LCmdFilePanelColumnsBar
 *   |    |    |    |         |    +--LCmdFilePanelModeFull::List
 *   |    |    |    |         |--LCmdFilePanelModeTree
 *   |    |    |    |         +--LCmdFilePanelModeInfo
 *   |    |    |    +--LCmdPanelInfobar
 *   |    |    +-- >>> Same as for the left LCmdFilePanelFrame.
 *   |    +--LCmdCmdLineContainer
 *   |         |--GContainerWindow
 *   |         |    |--GTextViewer
 *   |         |    |--GScrollbar
 *   |         |    +--GScrollbar
 *   |         +--LCmdCmdLine
 *   |              |--LCmdCmdLinePrompt
 *   |              +--LCmdCmdLineEntry
 *   |--GKeybar
 *   +--GStatusbar
 * </pre>
 *
 * @author  Leif Erik Larsen
 * @since   1996.05.26
 */
class LCmd : public GProgram
{
   public:

      /** True when the startup progress has finished. */
      bool finishedStartup;

      /** Used to synchronization multithreaded calls to {@link #printF}. */
      GObject printFLock;

      /** 
        * File Panel #1, which is always the panel to the left. 
        * @see   LCmdFilePanel#getLeftPanel
        */
      class LCmdFilePanel* fp1;

      /** 
       * File Panel #2, which is always the panel to the left. 
        * @see   LCmdFilePanel#getRightPanel
       */
      class LCmdFilePanel* fp2;

      /** Pointer to the current file panel. */
      class LCmdFilePanel* curPanel;

      /** Process, Thread and Module info. */
      GSystemInfo sysInfo;

      /** Information about each available drive on the system. */
      LCmdDlgChooseDrive drives;

      /** 
       * User customizable program options. 
       * @see   LCmdOptions#lcmdOptions
       */
      LCmdOptions options;

      /** The dialog used to let the user edit the various options visually. */
      class LCmdDlgOptions* optionsDlg;

      /** The "find text" dialog used by the console monitor. */
      LCmdDlgTextSearch conMonTextSearch;

      /** The main window (frame window) of the program. */
      LCmdMainWindow mainWin;

      /** Array of active instances of {@link LCmdInternalViewer}'s. */
      GArray<LCmdInternalViewer> openViewers;

      /** The About-box of Larsen Commander. */
      LCmdDlgAbout aboutBox;

      /** The dialog of where to specify destination path for copy operations. */
      LCmdDlgCopyFile dlgCopy;

      /** The Dynamic Directory Cache. */
      class LCmdDirCache* dirCache;

      /** Text to show in the files-panel for the "up-dir" ("%Txt_FP_UpDir"). */
      const GStringl txtUpDir;

      /** Text to show in the files-panel for "dub-dirs" ("%Txt_FP_SubDir"). */
      const GStringl txtSubDir;

      /** A reference to the command container window (contains the conmon and the cmdLine). */
      class LCmdCmdContainer& cmdCont;

      /** A reference to the console monitor window. */
      class LCmdConsoleMonitor& conmon;

      /** A reference to the command line window. */
      class LCmdCmdLine& cmdLine;

      /** A reference to the command line entry window. */
      class LCmdCmdLineEntry& cmdEntry;

      /** Used to stop any child process(es) upon exit. */
      class LCmdExitWorker* exitWorker;

   public:

      LCmd ( const GKeyBag<GString>& ppdefinesForTheRcCompiler );
      virtual ~LCmd ();

   public:

      void parseExitParams ( const GCmdLineParameters& params );
      bool parseStartupParams ( const GCmdLineParameters& params );
      void setupInitialDirForFilePanel ( LCmdFilePanel& fp, const GString& dir );

      /**
       * Overrides {@link GProgram#getMainWindow}.
       */
      GWindow& getMainWindow ();

      /**
       * Overrides {@link GProgram#isAutoSaveColorOptions}.
       *
       * Return true if and only if the window color options are to be
       * automatically saved to the program profile upon saving options.
       */
      bool isAutoSaveColorOptions () const;

      /**
       * Overrides {@link GProgram#isAutoSaveFontOptions}.
       *
       * Return true if and only if the window font options are to be
       * automatically saved to the program profile upon saving options.
       */
      bool isAutoSaveFontOptions () const;

      /**
       * Overrides {@link GProgram#isAutoSaveVisibilityStateOptions}.
       */
      bool isAutoSaveVisibilityStateOptions () const;

      /**
       * Return true if and only if we shall paint icons in the
       * menubar items.
       */
      bool isUseFancyMenubars () const;

      /**
       * Return true if and only if we shall paint icons in the items of
       * the various popup menues.
       */
      bool isUseFancyPopupMenues () const;

      /**
       * This is the entry point of the application from the
       * application point of view.
       */
      virtual int mainStart ();

      /**
       * Print the formatted text to the console monitor, synchronously.
       * This method is synchronized and both the main thread and background 
       * threads of Larsen Commander can safely call this method.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.14
       */
      virtual void printF ( const GString& str, const GVArgs& args = GVArgs() );

      /**
       * Write the profile settings of the program to an INI-file.
       * If the specified path is null then we will write to the
       * default INI-file.
       *
       * @return True on success, or else false on any error.
       * @see    LCmdMainWindow#writeProfile
       */
      bool writeProfile ( bool forceSaveEveryThing, 
                          bool saveOnlyOptionsWhatToSaveOnExit, 
                          const GString* path = null );

      /**
       * Override this method and do nothing but return, because we have 
       * our own control of the user profile management, which is 
       * implemented by {@link #writeProfile}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       */
      virtual void writeUserProfile ();
};

/** A reference to the Larsen Commander program object it self. */
extern LCmd* lcmd;

#endif // #ifndef __LCMD_H
