/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/sys/GSystem.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/util/GStringUtil.h"

#include "lcmd/LCmd.h"
#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmdExitWorker.h"
#include "lcmd/LCmdDlgStartupTip.h"
#include "lcmd/LCmdSubClient.h"
#include <time.h>

LCmd* lcmd; // A pointer to the program object it self.

/**
 * This is the main entry point of the application
 * from Larsen Commander's point of view.
 */
int gmain ( const GCmdLineParameters& )
{
   GKeyBag<GString> ppdefines;
   ppdefines.put("__ENABLE_TREE_VIEW", new GString(GInteger::ToString(__ENABLE_TREE_VIEW))); // Temporary, until LCmd-treeview is ready.
   LCmd app(ppdefines);
   return app.runTheProgram();
}

LCmd::LCmd ( const GKeyBag<GString>& ppdefinesForTheRcCompiler )
     :GProgram("Larsen Commander", true, true, true, &ppdefinesForTheRcCompiler),
      finishedStartup(false),
      fp1(null),
      fp2(null),
      curPanel(null),
      drives(*this),
      options(*this),
      optionsDlg(null),
      conMonTextSearch("MainWin.Console.DlgTextSearch"),
      mainWin(*this),
      openViewers(16),
      txtUpDir("%Txt_FP_UpDir"),
      txtSubDir("%Txt_FP_SubDir"),
      cmdCont(mainWin.subClient->cmdcont),
      conmon(mainWin.subClient->cmdcont.conmon),
      cmdLine(mainWin.subClient->cmdcont.cmdline),
      cmdEntry(mainWin.subClient->cmdcont.cmdline.entry),
      dirCache(new LCmdDirCache()),
      exitWorker(null)
{
}

LCmd::~LCmd ()
{
   delete exitWorker;
   delete dirCache;
   delete optionsDlg;
}


/*
 * This is the entry point of the application from the
 * application point of view.
 */
int LCmd::mainStart ()
{
   // Use a C++ class to ensure that we always call LCmd.parseExitParams()
   // regardless how we return from this method, even if we "return" due
   // to some exception. This is needed since ANSI C++ doesn't have any
   // "finally" keyword like Java.
   class ExitParamsParser {
      LCmd* lc;
      const GCmdLineParameters* p;
   public:
      ExitParamsParser ( LCmd* lc, const GCmdLineParameters& p ) {
         this->lc = lc;
         this->p = &p;
      }
      ~ExitParamsParser () {
         lc->parseExitParams(*p);
      }
   };

   const GCmdLineParameters& params = getCmdLineParameters();
   ExitParamsParser epp(this, params);

   // Read setup from our INI-file.
   // Mark that this will not restore the frame window position and size.
   // It will not make it visible either. This is done later by calling
   // {@link GFrameWindow#restoreWindowPos} when all the initialization
   // has been performed.
   mainWin.queryProfile(mainWin.getName());

   // Parse the application specific command line parameters.
   if (!parseStartupParams(params))
      return 0;

   // Increase the priority of the main thread of ours.
   // This is the GUI-thread that is also used to dispatch GUI-messages
   // to the message handler methods of all our application windows.
   // We increase its priority in order for the GUI event handling 
   // code to keep good responsibility even when there are some other 
   // programs running at normal priority in the background.
   GThread& mainGuiThread = GThread::GetCurrentThread();
   mainGuiThread.setPriority(GThread::PRIORITY_NORMAL_PLUS);

   // Make the statusbar cells initially show up with correct layout.
   mainWin.onTimer("INFOUPDATE", null);
   mainWin.getStatusbar()->layout();

   // Print some interesting startup information to the log-file.
   // It is important that we do this AFTER the main window timer "INFOUPDATE" 
   // has been prosessed at least once, because we need the system-info
   // object of Larsen Commander to be initialized BEFORE the below code run.
   GTimeStamp timeNow;
   GString localeRcPath = getResourcePathLocale();
   GLog::PrintLn("00: -------------------------------------------------------------------");
   GLog::PrintLn("01: Startup Time At: %s", GVArgs(timeNow.toString()));
   GLog::PrintLn("02: Time used to start Larsen Commander: %s ms.", GVArgs(GSystem::CurrentTimeMillis() - GProgram::TimeMillisEarlyStartup));
   GLog::PrintLn("03: Main Resource Table: %s", GVArgs(getResourcePath()));
   GLog::PrintLn("04: Time used to load the Main Resource Table: %s ms.", GVArgs(getMillisSpentOnMainResourceTableLoad()));
   GLog::PrintLn("05: Locale Resource Table: %s", GVArgs(localeRcPath != "" ? localeRcPath.cstring() : "N/A"));
   GLog::PrintLn("06: Time used to load the Locale Resource Table: %s ms.", GVArgs(getMillisSpentOnLocaleResourceTableLoad()));
   GLog::PrintLn("07: INI-file: %s", GVArgs(getIniProperties().getPath()));
   GLog::PrintLn("09: Operating System Name: %s", GVArgs(GSystem::GetPlatformName()));
   GLog::PrintLn("10: System Desktop Screen Size: %s", GVArgs(GSystem::GetScreenSize().toString()));
   GLog::PrintLn("11: Physical Memory Size: %s", GVArgs(GStringUtil::ToByteCountString(sysInfo.getPhysicalRAM())));
   GLog::PrintLn("12: GLog Filter Level: %s", GVArgs(GLog::GetFilterLevelName()));
   GLog::PrintLn("13: Log-file: %s", GVArgs(GLog::GetLogFilePath() != "" ? GLog::GetLogFilePath() : GStringl("N/A")));
   GLog::PrintLn("14: -------------------------------------------------------------------");

   // Restore the main window position and minimize/maximize state.
   // This will also make the frame window visible.
   mainWin.restoreWindowPos(mainWin.getName(), "Pos");

   // Initially read filenames into left and right file panel.
   fp1->fillList(true, true);
   fp2->fillList(true, true);

   // Repaint headerbars, in case root dir is current and ".." and "\" button
   // must therefore be initially painted as disabled.
   fp1->headerWin.invalidateAll(true);
   fp2->headerWin.invalidateAll(true);

   // Force everything to repaint at startup.
   // This should not actually be required, but due to a bug in some
   // OS/2 display drivers this is the best thing to do in order to
   // guarantee that Larsen Commander does not start up with a corrupt
   // display. Remember how important the first impression is!
   mainWin.invalidateAll(true);

   // Make sure that the initially active file panel has a valid
   // and existing current drive and directory upon startup.
   curPanel->activatePanelDriveAndDir(true);

   // Show the startup tip, if not disabled.
   if (LCmdDlgStartupTip::ShouldShowUponStartup())
      mainWin.cmdHelpShowTip();

   // Ensure the initial focus.
   finishedStartup = true;
   mainWin.ensureFocusOK(); 

   // Enter into the modal message dispatcher loop.
   // This will not return until the program is about to exit normally.
   enterModalMessageLoop();

   // Close all open (if any) text file viewer windows.
   while (openViewers.getCount() > 0)
   {
      int lastIdx = openViewers.getCount() - 1;
      LCmdInternalViewer* v = &openViewers.get(lastIdx);
      mainWin.closeAndDestroyTextViewer(v);
   }

   // Update the INI-file if needed, and hide the main window.
   writeProfile(false, false);
   mainWin.setVisible(false);

   return 0;
}

bool LCmd::parseStartupParams ( const GCmdLineParameters& params )
{
   // Show help for the supported command line parameters if requested.
   if (params.isContainingHelpRequest())
   {
      GStringl helpText("%TxtSupportedCommandLineArguments");
      GStringl helpTextDetails("%TxtSupportedCommandLineArguments_Details");
      showMessageBox(helpText, GMessageBox::TYPE_INFO, "Do", GString::Empty, false, helpTextDetails);
      return false;
   }

   // Use the initial directories as are possibly requested by parameters
   // that are given to the program from the command line.
   const int num = params.getCount() - 1;
   for (int i=0; i<num; i++)
   {
      const GString& p = params.getIndexedString(i);
      if (p.equalsIgnoreCase("-leftDir"))
      {
         const GString& dir = params.getIndexedString(++i);
         setupInitialDirForFilePanel(*fp1, dir);
      }
      else
      if (p.equalsIgnoreCase("-rightDir"))
      {
         const GString& dir = params.getIndexedString(++i);
         setupInitialDirForFilePanel(*fp2, dir);
      }
   }

   // ---
   return true;
}

void LCmd::parseExitParams ( const GCmdLineParameters& params )
{
   if (params.isContainingParam("-delIni"))
   {
      // The -delIni parameter was specified at command line, so delete the
      // INI-file that we read upon startup. This is a feature that was
      // implemented to be used in conjunction with
      // the "Clone Program Instance" command.
      GVfsLocal vfs;
      GVfs::AutoDeletionHandleCloser delCloser(vfs, vfs.openDeletion(), false);
      GString path = getIniProperties().getPath();
      vfs.removeFile(delCloser.hdel, path, false);
      vfs.performDeletion(delCloser.hdel);
   }
}

void LCmd::setupInitialDirForFilePanel ( LCmdFilePanel& fp, const GString& dir )
{
   GString path(256);
   GVfsLocal& fs = fp.vfs.root();
   GError rc = fs.getWalkedPath(dir, path);
   fs.setCurrentDirectory(rc == GError::Ok ? path : dir);
}

void LCmd::printF ( const GString& str, const GVArgs& args )
{
   GString formattedStr(str, args);
   bool containsLF = formattedStr.endsWithEol() || formattedStr.indexOfAnyChar('\n', '\r') >= 0;
   if (!containsLF)
      GLog::PrintLn(formattedStr);
   if (finishedStartup)
   {
      static int counter = 0;
      GThread& thread = GThread::GetCurrentThread();
      GObject::Synchronizer synch(printFLock); // One thread at a time!
      GString text;
      if (containsLF)
         text = formattedStr;
      else
         text = GString("%05d: %s\n", GVArgs(++counter).add(formattedStr));
      if (thread.isMainThread()) // If the thread is the GUI-thread (main thread).
         conmon.appendTextFromGuiThread(text);
      else
         conmon.appendTextFromSecondaryThread(thread, text);
   }
}

// Dummy method just to override {@link GProgram#writeUserProfile}.
void LCmd::writeUserProfile ()
{
}

bool LCmd::writeProfile ( bool forceSaveEveryThing,
                          bool saveOnlyOptionsWhatToSaveOnExit,
                          const GString* path )
{
   GSectionBag& ini = getIniProperties();
   LCmdOptions& opt = options;

   // Save Settings On Exit.
   GString sn = mainWin.getName() + ".SaveSettingsOnExit";
   ini.putBool(sn, "All", opt.saveOnExit.everything);
   ini.putBool(sn, "CurDir", opt.saveOnExit.currentDirs);
   ini.putBool(sn, "StoredDirs", opt.saveOnExit.storedDirs);
   ini.putBool(sn, "DirHistory", opt.saveOnExit.dirHist);
   ini.putBool(sn, "CmdHistory", opt.saveOnExit.cmdHist);
   ini.putBool(sn, "DirCache", opt.saveOnExit.dirCache);
   ini.putBool(sn, "CurSelectedFileItem", opt.saveOnExit.curSelFiles);
   ini.putBool(sn, "VisibilityStates", opt.saveOnExit.visibleState);
   ini.putBool(sn, "PanelModesAndSortOptions", opt.saveOnExit.panelModeSort);
   ini.putBool(sn, "MainWinPos", opt.saveOnExit.frameWinPos);
   ini.putBool(sn, "Colors", opt.saveOnExit.colors);
   ini.putBool(sn, "Fonts", opt.saveOnExit.fonts);
   ini.putBool(sn, "AllOther", opt.saveOnExit.otherOptions);

   if (!saveOnlyOptionsWhatToSaveOnExit)
   {
      const GString& sectName = mainWin.getName();
      mainWin.writeProfile(sectName, forceSaveEveryThing);
   }

   // Finally, physically write the profile to the file on disk.
   try {
      if (path != null)
      {
         ini.write(*path);
      }
      else
      {
         if (ini.isDirty())
         {
            ini.write();
         }
         else
         {
            // "Setup not saved because it hasn't changed (\"%s\")".
            GStringl info("%StrStatusbar_Status_IniNotDirty", GVArgs(ini.getPath()));
            mainWin.setStatusbarText(info);
            return true;
         }
      }
      // "Setup successfully saved (\"%s\".
      GStringl info("%StrStatusbar_Status_IniSavedOk", GVArgs(ini.getPath()));
      mainWin.setStatusbarText(info);
      return true;
   } catch (GIOException& e) {
      // "Could not save setup. Message is: %s".
      GStringl msg("%StrStatusbar_Status_IniSaveErr", GVArgs(e.toString()));
      mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }
}

GWindow& LCmd::getMainWindow () 
{ 
   return mainWin; 
}

bool LCmd::isAutoSaveColorOptions () const 
{ 
   return options.saveOnExit.colors || options.saveOnExit.everything; 
}

bool LCmd::isAutoSaveFontOptions () const 
{ 
   return options.saveOnExit.fonts || options.saveOnExit.everything; 
}

bool LCmd::isAutoSaveVisibilityStateOptions () const 
{ 
   return options.saveOnExit.visibleState || options.saveOnExit.everything; 
}

bool LCmd::isUseFancyMenubars () const 
{ 
   return options.fancyMenuBars; 
}

bool LCmd::isUseFancyPopupMenues () const 
{ 
   return options.fancyPopupMenues; 
}
