/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GVfsZip.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/io/GFileOutputStream.h"
#include "glib/io/GRandomAccessFile.h"
#include "glib/util/GMath.h"
#include "zlib/zlib.h"

int GVfsZip::OpenFileCounter = 0;
GHashtable<GInteger, GVfsZip::OpenFile> GVfsZip::OpenFiles;

GVfsZip::GVfsZip ( GVfsLocal& localVfs, 
                   GVfs& parentVfs, 
                   GVfs::File* vfile,
                   const GString& toolPath,
                   const GString& paramsDel )
        :GVfsArchiveFile(localVfs, parentVfs, vfile,
                         "ZIP", // fileSystemName
                         '/', // slashChar
                         false, // supportsChangeFileNameCase
                         true, // isFileNameCasePreserved
                         true, // isFileNameCaseSensitive
                         toolPath, // toolPath
                         paramsDel) // paramsDel
{
}

GVfsZip::~GVfsZip ()
{
}

void GVfsZip::loadItems ( bool* cancelSem, int* preLoadCounter ) const
{
   items.removeAll();

   // Make sure only one thread can perform read/write operations 
   // on the physical archive file at once.
   GObject::Synchronizer synch(archiveFileAccessLock);
   GRandomAccessFile& zipFile = openArchiveFile();
   AutoArchiveFileCloser autoCloseArchiveFile(*this);

   // Find and read the End-Of-Central-Directory,
   // and seek to the first CentralDirFileHeader.
   EndOfCentralDir ecd;
   try {
      int attempts = 0;
      longlong pos = longlong(0) - longlong(sizeof(EndOfCentralDir));
      zipFile.seekFromEnd(pos);
      while (cancelSem == null || !(*cancelSem))
      {
         zipFile.readExact(&ecd, sizeof(ecd));
         if (ecd.signature == 0x06054b50)
         {
            // Seek to the first CentralDirFileHeader.
            zipFile.seekFromStart(ecd.offset);
            break;
         }
         if (++attempts >= 0xFFFF) // Max ZIP-file comment length is 0xFFFF.
            gthrow_(GIOException(""));
         zipFile.seekFromCur(0 - sizeof(ecd) - 1);
      }
   } catch (GIOException& /*e*/) {
      GString zipPath = archiveFileItem.getFullPath();
      gthrow_(GIOException("End-Of-Central-Directory not found in ZIP-file: " + zipPath));
   }

   // ---
   CentralDirFileHeader fhead;
   char buffer[4096]; // 4096 characters should be anough for very long filenames in the ZIP-file! :-)
   const int bufferSize = sizeof(buffer);
   GString bufferStr;
   int fcount = 0;

   // Read all CentralDirFileHeader's.
   while (cancelSem == null || !(*cancelSem))
   {
      // Read file head.
      if (zipFile.read(&fhead, sizeof(fhead)) != sizeof(fhead))
         break; // We have probably reached end-of-file.

      // Check signature.
      if (fhead.signature != 0x02014b50)
         break;

      // Check if it is encrypted. Is file password protected?
      if (fhead.gflag & 0x1)
         passwordProtected = true;

      // Read filename.
      if (fhead.fnameLength >= bufferSize)
         gthrow_(GIOException("Central-Directory filename too long in ZIP-file: " + archiveFileItem.getFullPath()));
      if (zipFile.read(buffer, fhead.fnameLength) != fhead.fnameLength)
         gthrow_(GIOException("Error reading filename element from ZIP-file: " + archiveFileItem.getFullPath()));
      buffer[fhead.fnameLength] = 0;

      // Convert the loaded filename into a string object, and make sure 
      // its directory elements are separated by the correct type of slash.
      bufferStr = buffer;
      translateSlashes(bufferStr);

      // Skip extra data.
      zipFile.seekFromCur(fhead.extraLength + fhead.comLength);

      // Extract timestamp data.
      int year = ((fhead.fileDate & 0x0FE00) / 0x0200) + 1980;
      int month = (fhead.fileDate & 0x1E0) / 0x20;
      int day = fhead.fileDate & 0x1F;
      int hour = (fhead.fileTime & 0xF800) / 0x800;
      int min = (fhead.fileTime & 0x7E0) / 0x20;
      int sec = 2 * (fhead.fileTime & 0x1f);

      // Add data to list.
      Item* item = new Item(fhead.headOffs, fhead.compSize);
      item->unsortedIndex = fcount++;
      item->path = bufferStr;
      item->timeWrite.setDate(year, month, day);
      item->timeWrite.setTime(hour, min, sec);
      item->timeCreate = item->timeWrite;
      item->timeAccess = item->timeWrite;
      item->fileSize = fhead.uncompSize;
      item->fileSizeAlloc = fhead.compSize;
      if (isSlash(buffer[fhead.fnameLength - 1]))
         item->attr |= GVfs::FAttrDirectory;
      else
         item->attr |= GVfs::FAttrArchive;
      items.put(item->path, item);

      // ---
      if (preLoadCounter != null)
         (*preLoadCounter)++;
   }
}

GVfsZip::OpenFile::OpenFile ( GVfsZip& ownerVfs,
                              GRandomAccessFile& zipFile,
                              GVfsArchiveFile::Item& item,
                              bool writable,
                              bool create, 
                              bool append )
                  :ownerVfs(ownerVfs),
                   zipFile(zipFile),
                   item(item),
                   handle(null),
                   currentStreamSeekPos(0),
                   writableMode(writable),
                   writeHandle(0),
                   writableFileObj(null),
                   chunk(64 * 1024), // 64 KBytes.
                   compressedDataPos(0),
                   compressedDataCount(0),
                   compressedData(chunk),
                   uncompressedDataPos(0),
                   uncompressedDataCount(0),
                   uncompressedData(chunk),
                   strm(null),
                   zlibInitErrCode(Z_OK),
                   remainingZippedBytes(0)
{
   if (writableMode)
   {
      // The zip-entry is to be opened in writable mode. Create a temporary 
      // physical file on the local file system, and open that temporary file
      // for usage by all file i/o operations.
      GString path = item.getFullPath();
      GVfs::WorkStatus wstat;
      GString prefix = GString::Empty;
      writableFileObj = ownerVfs.preparePhysicalFile(path, wstat, prefix);
   }
   else
   {
      initZipStream();
   }
}

GVfsZip::OpenFile::~OpenFile ()
{
   if (writableMode)
   {
      // TODO: Start ZIP-utility to update the zip-entry.
      delete writableFileObj;
   }
   else
   {
      clearZipStream();
   }
}

void GVfsZip::OpenFile::clearZipStream ()
{
   if (strm != null && zlibInitErrCode == Z_OK)
      ::inflateEnd(strm);
   delete strm;
   strm = null;
}

void GVfsZip::OpenFile::initZipStream ()
{
   // Delete the old zip-stream, if any.
   if (strm != null)
      clearZipStream();
   currentStreamSeekPos = 0;
   compressedDataPos = 0;
   compressedDataCount = 0;
   uncompressedDataPos = 0;
   uncompressedDataCount = 0;
   zlibInitErrCode = Z_OK;
   remainingZippedBytes = 0;

   // Do the initialization of the zip-stream.
   strm = new z_stream;
   memset(strm, 0, sizeof(*strm));
   zlibInitErrCode = ::inflateInit2(strm, -MAX_WBITS);
   if (zlibInitErrCode != Z_OK)
      return;

   zipFile.seekFromStart(item.headPos);
   zipFile.read(&head, sizeof(head));
   zipFile.seekFromCur(head.fnameLength);
   zipFile.seekFromCur(head.extraLength);

   // In some zip-files (or jar-files) the "head.compSize" is zero.
   // If this is the case, use the data size originating from the cetral 
   // directory data block of the zip-file.
   if (head.compSize == 0)
      remainingZippedBytes = item.fileDataSize; // Data from cetral directory.
   else
      remainingZippedBytes = head.compSize;

   synchronized (GVfsZip::OpenFiles) 
   {
      int fileHandle;
      for (;;)
      {
         GVfsZip::OpenFileCounter++;
         if (GVfsZip::OpenFileCounter < 1)
            GVfsZip::OpenFileCounter = 1;
         GInteger key(GVfsZip::OpenFileCounter);
         if (!GVfsZip::OpenFiles.containsKey(key))
         {
            fileHandle = GVfsZip::OpenFileCounter;
            break;
         }
      }

      handle = static_cast<GVfs::FileHandle>(fileHandle);
      GInteger* key = new GInteger(fileHandle);
      GVfsZip::OpenFiles.put(key, this, true, true);
   } synchronized_end;
}

int GVfsZip::OpenFile::readZipStream ( void* buff, int numBytesToRead )
{
   if (numBytesToRead <= 0)
      return 0;

   // Copy next chunk of already uncompressed data, if any.
   if (uncompressedDataPos < uncompressedDataCount)
   {
      int count = GMath::Min(numBytesToRead, uncompressedDataCount - uncompressedDataPos);
      memcpy(buff, uncompressedData.theBuffer + uncompressedDataPos, count);
      uncompressedDataPos += count;
      currentStreamSeekPos += count;
      return count;
   }

   // Read the next chunk of compressed data from the file, if needed.
   if (compressedDataPos >= compressedDataCount)
   {
      if (remainingZippedBytes <= 0)
         return 0; // End-of-file.
      int tryRead = int(GMath::Min(remainingZippedBytes, longlong(chunk)));
      compressedDataCount = zipFile.read(compressedData.theBuffer, tryRead);
      if (compressedDataCount <= 0)
         return 0; // End-of-file.
      remainingZippedBytes -= compressedDataCount;
      compressedDataPos = 0;
   }

   // If the source data is uncompressed then just copy and return.
   if (head.compMethod == 0)
   {
      // Source data is uncompressed. So just copy from "compressedData".
      int count = GMath::Min(numBytesToRead, compressedDataCount - compressedDataPos);
      memcpy(buff, compressedData.theBuffer + compressedDataPos, count);
      compressedDataPos += count;
      currentStreamSeekPos += count;
      return count;
   }

   // Source data is in fact compressed. So uncompress the next chunk of data.
   int inBufferLen = compressedDataCount - compressedDataPos;
   strm->avail_in = inBufferLen;
   strm->next_in = (unsigned char*) (compressedData.theBuffer + compressedDataPos);
   strm->avail_out = chunk;
   strm->next_out = (unsigned char*) uncompressedData.theBuffer;
   int rc = ::inflate(strm, Z_SYNC_FLUSH);
   if (rc == Z_STREAM_ERROR || rc == Z_NEED_DICT || rc == Z_DATA_ERROR || rc == Z_MEM_ERROR)
      gthrow_(GIOException(GString("ZLib::inflate() returned: %d", GVArgs(rc))));
   uncompressedDataPos = 0;
   uncompressedDataCount = chunk - strm->avail_out;
   compressedDataPos += inBufferLen - strm->avail_in;
   if (uncompressedDataCount <= 0)
      return 0; // Eof

   // Copy from newly filled buffer of uncompressed data.
   return readZipStream(buff, numBytesToRead);
}

GVfs::FileHandle GVfsZip::openFile ( const GString& path,
                                     GError* errorCode,
                                     OF_Mode modeOpt,
                                     OF_Create createOpt,
                                     OF_Share shareOpt,
                                     int flagsOpt,
                                     OF_ActionTaken* actionTaken )
{
   // Make sure we have a non-null errorCode-pointer.
   GError localRc;
   if (errorCode == null)
      errorCode = &localRc;
   *errorCode = GError::Ok;

   // ---
   bool append = (flagsOpt & GVfs::OF_FLAG_APPEND);
   bool writable = (modeOpt != GVfs::Mode_ReadOnly);
   Item* item = getItem(path);
   bool exist = (item != null);
   bool create = false;
   switch (createOpt)
   {
      case GVfs::Create_Never: create = false; break;
      case GVfs::Create_IfNew: create = !exist; break;
      case GVfs::Create_IfExist: create = exist; break;
      case GVfs::Create_Always: create = true; break;
      default: *errorCode = GError::InvalidArgument; return null;
   }
   if (exist)
   {
      if (flagsOpt & OF_FLAG_DONT_OPEN_IF_EXIST)
      {
         *errorCode = GError::FileExists;
         return null;
      }
   }
   else
   if (!create)
   {
      *errorCode = GError::FileNotFound;
      return null;
   }

   // TODO: ???
   if (item == null)
   {
      // The item doesn't exist in the zip-file.
      // This means we will have to create it, but this is not supported yet!
      *errorCode = GError::NotSupported;
      return null;
   }

   // Find the next free file handle and create a new corresponding object 
   // to represent the open file in our hashtable of open files.
   GVfsZip::OpenFile* ofile = null;
   try {
      GRandomAccessFile& archiveFile = openArchiveFile(); // Will increase reference counter also.
      GVfsZip::OpenFile* ofile = new GVfsZip::OpenFile(*this, archiveFile, *item, writable, create, append);
      if (ofile->zlibInitErrCode == Z_OK)
         return ofile->handle;
   } catch (GIOException& e) {
      *errorCode = e.getSystemErrorCode();
      delete ofile;
      return null;
   }

   *errorCode = ERROR_BAD_FORMAT;
   delete ofile;
   return null;
}

GError GVfsZip::closeFile ( FileHandle hfile )
{
   synchronized (OpenFiles) 
   {
      GVfsZip::OpenFile* ofile = getOpenFile(hfile);
      if (ofile == null)
         return GError::InvalidHandle;
      GInteger key(static_cast<int>(hfile));
      OpenFiles.remove(key);
      closeArchiveFile(); // Will close if reference counter == 0.
   } synchronized_end;
   return GError::Ok;
}

GVfsZip::OpenFile* GVfsZip::getOpenFile ( GVfs::FileHandle hfile )
{
   synchronized (OpenFiles) 
   {
      GInteger key(static_cast<int>(hfile));
      return OpenFiles.get(key);
   } synchronized_end;
}

GError GVfsZip::readFromFile ( GVfs::FileHandle hfile, 
                               void* buff, 
                               int numBytesToRead, 
                               int* numBytesActuallyRead )
{
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   if (ofile->writableMode)
      return localVfs.readFromFile(ofile->writeHandle, buff, numBytesToRead, numBytesActuallyRead);
   try {
      if (numBytesActuallyRead != null)
         *numBytesActuallyRead = 0;
      int read = ofile->readZipStream(buff, numBytesToRead);
      if (numBytesActuallyRead != null)
         *numBytesActuallyRead = read;
      return GError::Ok;
   } catch (GIOException& /*e*/) {
      return GError::AccessDenied; // TODO: ???
   }
}

GError GVfsZip::loadFileInfo ( GVfs::FileHandle hfile, GFileItem& info )
{
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   info = ofile->item;
   return GError::Ok;
}

longlong GVfsZip::getFileSize ( GVfs::FileHandle hfile, GError* err )
{
   GError localErr;
   if (err == null)
      err = &localErr;
   *err = GError::Ok; // Until the opposite has been proven.
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
   {
      *err = GError::InvalidHandle;
      return -1;
   }
   if (ofile->writableMode)
      return localVfs.getFileSize(ofile->writeHandle, err);
   return ofile->item.fileSize;
}

longlong GVfsZip::getFileSeekPos ( GVfs::FileHandle hfile, GError* err )
{
   GError localErr;
   if (err == null)
      err = &localErr;
   *err = GError::Ok; // Until the opposite has been proven.
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
   {
      *err = GError::InvalidHandle;
      return -1;
   }
   if (ofile->writableMode)
      return localVfs.getFileSeekPos(ofile->writeHandle, err);
   return ofile->currentStreamSeekPos;
}

GError GVfsZip::setFileSeekPosFromCurrent ( GVfs::FileHandle hfile, longlong distanceToMove )
{
   if (distanceToMove < 0)
      return GError::NotSupported; // Backward seek not supported on this VFS.
   // Perform dummy read foreward to change the seek position.
   const int buffSize = 1024;
   GBuffer<BYTE> buff(buffSize);
   int numBytesActuallyRead = 0;
   while (distanceToMove > 0)
   {
      GError err = readFromFile(hfile, buff.theBuffer, buffSize, &numBytesActuallyRead);
      if (err != GError::Ok)
         return err;
      if (numBytesActuallyRead <= 0)
         return GError::InvalidArgument; // Cannot seek beyound end-of-file.
      distanceToMove -= numBytesActuallyRead;
   }
   return GError::Ok;
}

GError GVfsZip::setFileSeekPosFromStart ( GVfs::FileHandle hfile, longlong distanceToMove )
{
   if (distanceToMove < 0)
      return GError::InvalidArgument; // Seeking ahead of start-of-file is not possible.
   if (distanceToMove == 0)
      return GError::Ok; // No-op.
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   if (ofile->writableMode)
      return localVfs.setFileSeekPosFromStart(ofile->writeHandle, distanceToMove);
   if (ofile->currentStreamSeekPos != 0)
      ofile->initZipStream(); // Re-initialize the zip-stream, to reset the seek pos.
   return setFileSeekPosFromCurrent(hfile, distanceToMove);
}

GError GVfsZip::setFileSeekPosFromEnd ( GVfs::FileHandle hfile, longlong distanceToMove )
{
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   if (distanceToMove == 0)
      return GError::Ok; // No-op.
   if (ofile->writableMode)
      return localVfs.setFileSeekPosFromEnd(ofile->writeHandle, distanceToMove);
   if (distanceToMove < 0)
      return GError::NotSupported; // Backward seek not supported on this VFS.
   return GError::NotSupported; // Extending file size is not supported on this VFS.
}

GError GVfsZip::writeToFile ( GVfs::FileHandle hfile,
                              const void* buff,
                              int numBytesToWrite,
                              int* numBytesActuallyWritten )
{
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   if (!ofile->writableMode)
      return GError::AccessDenied;
   return localVfs.writeToFile(ofile->writeHandle, buff, numBytesToWrite, numBytesActuallyWritten);
}

GError GVfsZip::setFileSize ( GVfs::FileHandle hfile, longlong size )
{
   GVfsZip::OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   if (!ofile->writableMode)
      return GError::AccessDenied;
   return localVfs.setFileSize(ofile->writeHandle, size);
}

GError GVfsZip::setFileAttributes ( const GString& path, int attr )
{
   return GError::NotSupported; // TODO: Not implemented yet!
}

GError GVfsZip::writeAttrAndTimes ( GVfs::FileHandle hfile, 
                                    const GFileItem& info,
                                    const GString& path )
{
   return GError::NotSupported; // TODO: Not implemented yet!
}
