/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FILEITEM
#define __GLIB_FILEITEM

#include "glib/vfs/GFile.h"
#include "glib/util/GTimeStamp.h"

/**
 * This class is used to represent the information about a single file or
 * directory item, for instance in an archive file or on the file system.
 *
 * @author  Leif Erik Larsen
 * @since   2002.07.28
 */
class GFileItem : public GObject
{
   public:

      /** Index of item before array of items was sorted. */
      int unsortedIndex;

      /** Normal file attributes. */
      int attr;

      /** File date and time (creation). */
      GTimeStamp timeCreate;

      /** File date and time (last write/update). */
      GTimeStamp timeWrite;

      /** File date and time (last access). */
      GTimeStamp timeAccess;

      /** Number of bytes in file (on disk). */
      ulonglong fileSize;

      /** Number of bytes allocated by file (on disk), or 0 if unknown. */
      ulonglong fileSizeAlloc;

      /** The path/url of the filename. Fully qualified or not. */
      GFile path;

   public:

      /**
       * Create a new File Item Object that is initially empty.
       */
      GFileItem ();

      /**
       * Create a new File Item Object by cloning the specified
       * File Item Object.
       */
      explicit GFileItem ( const GFileItem& src );

      /**
       * Create a new File Item and initialize it using the specified file
       * information block from OS/2.
       */
      GFileItem ( const FILEFINDBUF3& src );
      GFileItem ( const FILEFINDBUF3L& src );

      /**
       * Create a new File Item and initialize it using the
       * specified file path.
       *
       * @param  vfs  The Virtual File System of where the specified 
       *              file item (path) is contained.
       * @param  path The path of which file or directory to be 
       *              represented by this new file item.
       * @throws GFileNotFoundException if we fail to access the 
       *              specified file.
       */
      GFileItem ( class GVfs& vfs, const GString& path );

      virtual ~GFileItem ();

   public:

      const GFileItem& operator= ( const FILEFINDBUF3& src );
      const GFileItem& operator= ( const FILEFINDBUF3L& src );

      bool operator== ( const GFileItem& src ) const;
      bool operator!= ( const GFileItem& src ) const;

   public:

      void clear ();

      // Methods for checking the Standard File Attributes.

      bool isDirectory () const;
      bool isArchive () const;
      bool isSystem () const;
      bool isHidden () const;
      bool isReadOnly () const;

      void setDirectoryFlag ( bool on = true );
      void setArchiveFlag ( bool on = true );
      void setSystemFlag ( bool on = true );
      void setHiddenFlag ( bool on = true );
      void setReadOnlyFlag ( bool on = true );

   public:

      /**
       * Return true if and only if the path points to an
       * archive file (ZIP, ARJ, etc.).
       */
      bool isZipOrArchiveFile () const;
      static bool IsZipOrArchiveFile ( const GString& path );

      /**
       * Check if the item is possibly a program file by file extension.
       */
      bool isProgramFileName () const;

      /**
       * Check if the item is the "." item.
       */
      bool isThisDir () const;

      /**
       * Check if the item is the ".." item.
       */
      bool isUpDir () const;

      /**
       * Get a copy of the filename of the item, including extension.
       */
      GString getFileName () const;

      /**
       * Get a reference to the filename of the item, excluding extension.
       */
      const GString& getName () const;

      /**
       * Get a reference to the filename extension of the item.
       */
      const GString& getExtension () const;

      /**
       * Get a copy of the full path of the filename item.
       */
      virtual GString getFullPath () const;

      /**
       * Get a copy of the directory (including drive and terminating
       * slash) of the filename item.
       */
      virtual GString getDirectory () const;

      void setName ( const GString& fname );
      void setExtension ( const GString& fext );
};

#endif
