/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_PROCESSLAUNCHER
#define __GLIB_PROCESSLAUNCHER

#include "glib/GThread.h"
#include "glib/GProgram.h"
#include "glib/util/GTime.h"
#include "glib/sys/GSystem.h"
#include "glib/io/GPipeInputStream.h"
#include "glib/io/GFileOutputStream.h" // TODO: Should be GPipeOutputStream

/**
 * This class can be used to launch a program or command as entered on
 * a command line.
 *
 * It is nice to have a separate thread doing this, because
 * then we can have the thread wait synchronously for the child process to
 * finish and the main thread can just test if the child process has
 * finished by testing the status of the secondary thread.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.07
 */
class GProcessLauncher : public GThread
{
   public:

      /**
       * The various error codes possibly returned by {@link #getErrorCode}.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.10.29
       */
      enum ErrorCode 
      {
         /** Success. */
         EC_NoError = 0,

         /** 
          * "Error activating directory: '%s'\n". 
          * Erronous directory is in {@link #workingDir}. 
          */
         EC_ErrActDirX, 

         /** 
          * "Not a program file: '%s'\n". 
          * Erronous program is in {@link #prgName}. 
          */
         EC_ErrNotAProgX, 

         /** 
          * "Program file not found: '%s'\n". 
          * Erronous program is in {@link #prgName}. 
          */
         EC_ErrPrgNotFoundX, 

         /** 
          * "Error loading module '%s' required by program '%s'.\n". 
          * Erronous module name is in {@link #nameOfMissingDll}. 
          * Erronous program is in {@link #prgName}. 
          */
         EC_ErrDllXReqByPrgY, 

         /** 
          * "Could not launch command '%s'. Error code: %d\n". 
          * Erronous command is in {@link #prgName}. 
          * Erronous system error code is in {@link #systemErrorCode}. 
          */
         EC_ErrLauchCmdXErrCodeY 
      };

   protected:

      GString prgName;
      GString paramStr;
      GString workingDir;
      bool forceNewSession;

      /** True if we shall automatically close Console session window when finished. */
      bool closeOnExit;

      /** True if command was run as a child process. */
      bool startedAsChild;

      /** True if Console program type is to be started in fullscreen mode. */
      bool startConFullScreen;

      /** True if Dos program type os to be started in fullscreen mode. */
      bool startDosFullScreen;

      /** 
       * Valid only while the child process is running. 
       * The PID of the running Child is in <i>childRes.codeTerminate</i>. 
       */
      mutable RESULTCODES childRes;

      /**
       * This is used by {@link #isChildWaitingForDataOnItsStdIn} 
       * to check the STDIN of our child processes to see if it waits for 
       * some input. It is valid (!= null) only while 
       * {@link #isRunningChild} returns true.
       */
      HEV pipeReaderMediator;

      /**
       * Counter for each named pipe created, so that we can create 
       * each pipe with unique names. Must be synchronized.
       */
      static GULong PipeCounter;

      /** Only valid after {@link #run} has returned. */
      ErrorCode errorCode;

      /** 
       * The exit code of the child process.
       * Only valid after {@link #waitForTheChildProcessToFinish} 
       * returns true. 
       */
      int exitCodeFromProcess;

      /** Name of missing DLL. Valid only when {@link #errorCode} equals {@link #EC_ErrDllXReqByPrgY}. */
      GString nameOfMissingDll;

      /** Error code from the system, but only when {@link #errorCode} != {@link #EC_NoError}. */
      APIRET systemErrorCode;

      /** Time when this launcher was instantiated. */
      GTime startTime;

      /** 
       * This output stream writes data to the write-end of the redirected 
       * STDIN pipe of the child process. We can write data to the child
       * via this stream. Used only if {@link #startedAsChild} is true and 
       * {@link #isRunningChild} returns true, else it is null. 
       */
      GFileOutputStream* childStdIn; // TODO: Should be GPipeOutputStream*
      
      /** 
       * This input stream reads data from the read-end of the redirected 
       * STDOUT pipe of the child process. We can read data from the child 
       * via this stream. Used only if {@link #startedAsChild} is true and 
       * {@link #isRunningChild} returns true, else it is null. 
       */
      GPipeInputStream* childStdOut;

      /** 
       * This input stream reads data from the read-end of the redirected 
       * STDERR pipe of the child process. We can read data from the child 
       * via this stream. Used only if {@link #startedAsChild} is true and 
       * {@link #isRunningChild} returns true, else it is null. 
       */
      GPipeInputStream* childStdErr;

      /** Data that is queued to the STDIN stream if the child process. */
      GArray<GString> stdInData;

      /** The PID of the child process that we will wait for in {@link #waitForTheChildProcessToFinish}. */
      int childPidToWaitFor;

   private:

      /**
       * Object used to synchronize execution of a child process so that 
       * only one thread can do that at a time. This is important in order 
       * to make sure that each child inherits the STDIN, STDOUT and STDERR
       * handles correctly.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.25
       */
      static GObject execSynch;

   public:

      /** Name of environment variable defining the SHELL. For OS/2, this is "COMSPEC". */
      static const GString COMSPEC;

      /** Default SHELL if COMSPEC environment variable is undefined. For OS/2, this is "CMD.EXE". */
      static const GString DEFCOMSPEC;

      /** Prefix arguments for system shell commands. */
      static const GString SysShellPrefixArgs;

   public:

      GProcessLauncher ( const GString& workingDir,
                         const GString& prgName,
                         const GString& paramStr = GString::Empty,
                         bool forceNewSession = false,
                         bool closeOnExit = true,
                         bool startConFullScreen = false,
                         bool startDosFullScreen = false );

      virtual ~GProcessLauncher ();

   private:

      /** Disable the copy constructor. */
      GProcessLauncher ( const GProcessLauncher& src ) {}

      /** Disable the assignment operator. */
      GProcessLauncher& operator= ( const GProcessLauncher& ) { return *this; }

   private:

      /**
       * Create a pipe for redirection of child process standard streams.
       * On OS/2 this will create a named pipe, not an anonymous pipe, 
       * because we must be able to use the OS/2 API DosPeekNPipe() and this 
       * API only works on named pipes. This API is useful for checking the 
       * number of bytes in the pipe that is available to read without 
       * blocking. There is no way to do this on anonymous pipes on OS/2.
       * Windows supports this also on anonymous pipes.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.05.11
       */
      void createPipe ( GSysFileHandle* hRead, GSysFileHandle* hWrite );

      GProcessLauncher::ErrorCode prepareTheWorkingDir ();

   protected:

      /**
       * This method is automatically called by {@link #run} in order to
       * let the subclass adjust the directory of which will be set as the
       * "current directory" for the program that is about to be launched.
       *
       * The default implementation will return a copy of the specified
       * default directory.
       */
      virtual GString getWorkingDir ( const GString& defaultDir ) const;

      /**
       * This method is automatically called by <i>run()</i> when it has
       * found the program to be an existing and runable program path on the
       * system.
       *
       * It is up to the sub-class to override this method and decide if
       * the program should be forced to run as a new session or not.
       * In order to force the program to run as a new session, the instance
       * variable <i>forceNewSession</i> must be set to true.
       *
       * @param  prgName   The program name of which has been validated.
       * @param  paramStr  The parameter string as is to be passed to the
       *                   program when it is executed.
       */
      virtual void programPathHasBeenValidated ( const GString& prgName, 
                                                 const GString& paramStr );

      /**
       * This method is automatically called when a new process has been
       * successfully launched.
       *
       * When this method is called the PID returned by 
       * {@link #getPidOfRunningChildProcess} is valid, but only 
       * if the launched process is a child process of course.
       *
       * @param asChild  True if and only if the new process was launched
       *                 as a child process.
       */
      virtual void processHasBeenLaunched ( bool asChild );

      /**
       * This method is automatically called when a process that was recently
       * started by {@link #processHasBeenLaunched} has finished.
       *
       * When this method is called the PID returned by 
       * {@link #getPidOfRunningChildProcess} is still valid, but only 
       * if the launched process is a child process of course.
       *
       * @param asChild    True if and only if the process was a child process.
       * @param result     The result code that was returned by the program.
       *                   This code is typically zero of the program exited
       *                   normally with no error. The result code is only valid
       *                   if <i>normalExit</i> is true, else the value of the
       *                   result code is undefined.
       * @param normalExit True if and only if the process did exit normally.
       *                   That is if it was not breaked or killed by any
       *                   means, but did indeed return normally from its
       *                   main function or called the exit function normally.
       */
      virtual void processHasFinished ( bool asChild, 
                                        int result, 
                                        bool normalExit );

      /**
       * Will actually start the process.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       * @see     #waitForTheChildProcessToFinish
       */
      virtual void run ();

      /**
       * Wait for some data to be available in the StdIn-queue of the
       * child process. Data is added to the StdIn-queue by some other 
       * thread by calling {@link #addTextToChildStdInQueue}.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.10.14
       */
      virtual void waitForSomeDataOnStdIn ();

      /**
       * This method is automatically called by {@link #run} if and when the 
       * process has been launched successfully and it was launched as a 
       * child process. 
       *
       * The point is to make it possible for the subclass to use the 
       * calling thread to perform some other tasks (e.g. reading the 
       * redirected stdout & stderr of the child process) while waiting for 
       * the child process to finish. This way we can prevent using a 
       * separate thread just for reading the redirected output.
       *
       * If the subclass overrides this method then it should return as 
       * soon as possible after the child process has finished. Use method 
       * {@link #isRunningChild} to test if this is the case, and 
       * call that method at least ten times per second.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      virtual void waitForTheChildProcessToFinish ();

   public:

      /**
       * Add the specified text to the queue of STDIN data to the 
       * currently running child process, in a thread safe manner.
       *
       * The specified text should typically always contain a trailing 
       * platform dependent linefeed sequence, because most console 
       * programs that takes input from its STDIN takes it from the 
       * console command line, which expects the user to press the 
       * [enter] key to commit each input.
       *
       * This method is thread safe and synchronized with 
       * {@link #getNextChildStdInText}. It is typically used by the
       * GUI-thread after the user has commited some text that is 
       * to be given to the child process. How this is done, however,
       * is completely up to the subclass.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.29
       * @see     #getNextChildStdInText
       */
      void addTextToChildStdInQueue ( const GString& txt );

      /**
       * Get a reference to the write side of the pipe to the STDIN stream
       * of the child process. This method should be used only when 
       * {@link isRunningChild} returns true, or else an 
       * {@link GIllegalStateException} will be thrown. 
       *
       * Also mark that the returned stream is valid to use only 
       * while the child process is still running. It is up to the caller
       * to make sure of this, either by modality or synchronization.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.24
       * @throws  GIllegalStateException if the child process is not running.
       */
      class GFileOutputStream& getChildStdIn ();

      /**
       * Get a reference to the read side of the pipe to the STDIN stream
       * of the child process. This method should be used only when 
       * {@link isRunningChild} returns true, or else an 
       * {@link GIllegalStateException} will be thrown. 
       *
       * Also mark that the returned stream is valid to use only 
       * while the child process is still running. It is up to the caller
       * to make sure of this, either by modality or synchronization.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @throws  GIllegalStateException if the child process is not running.
       */
      class GPipeInputStream& getChildStdOut ();

   protected:

      /**
       * Get the next data to be given to the STDIN stream of the 
       * child process, or an empty string if there are no such 
       * queued data at this time.
       *
       * This method is thread safe and synchronized with 
       * {@link #addTextToChildStdInQueue}. It is typically used by the
       * subclass implementation of {@link #waitForTheChildProcessToFinish}
       * in case it is STDIN-aware with respect to the child process.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.29
       * @see     #addTextToChildStdInQueue
       */
      GString getNextChildStdInText ();

   public:

      /**
       * Checks if the child process currently waits for some data to be 
       * available on its STDIN stream.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.24
       */
      bool isChildWaitingForDataOnItsStdIn () const;

      /**
       * Test if the child process is still running.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       * @see     #waitForTheChildProcessToFinish
       */
      bool isRunningChild () const;

      /**
       * Break/kill the child program that is managed by this process 
       * launcher. Return true if and only if the program was successfully 
       * breaked.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.26
       * @param   bt        How to break the process.
       * @param   breakTree True if we shall break all parts of the process 
       *                    tree, including its vhild process(es). If false 
       *                    is specified then we will break the process it
       *                    self only, and its child processes (if any) will
       *                    continue to run until they finish normally.
       */
      bool breakChildProg ( GSystem::BreakType bt, bool breakTree );

      /**
       * Get the error code after the command has been launched.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.29
       */
      GProcessLauncher::ErrorCode getErrorCode () const;

      /**
       * Get the exit code from the executed and finished child process.
       * This method should be used only after 
       * {@link #waitForTheChildProcessToFinish} returns true, else 
       * we will return -1.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      int getExitCodeFromprocess () const;

      /**
       * Get the system dependent PID of the child process that is currently 
       * running after being successfully launched by this process launcher.
       * If the launched process is not a child, or if it is not currently 
       * running, then we will return 0.
       *
       * This method should be used only in the period between when 
       * {@link #processHasBeenLaunched} and {@link #processHasFinished}
       * is called, else the returned PID will be zero.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.14
       */
      int getPidOfRunningChildProcess () const;

      const GString& getProgramName () const;

      const GString& getParamString () const;

      const GTime& getStartTime () const;
};

#endif
