/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/util/GParameterList.h"
#include "glib/primitives/GInteger.h"

GParameterList::GParameterList ( bool ignoreCase )
               :args(4, -3, ignoreCase)
{
}

GParameterList::GParameterList ( const GParameterList& src )
               :args(src.args.getCount(), 
                     -3, 
                     src.args.isIgnoreCase())
{
   copyFrom(src);
}

GParameterList::~GParameterList ()
{
}

GParameterList& GParameterList::operator= ( const GParameterList& src )
{
   return copyFrom(src);
}

GParameterList& GParameterList::copyFrom ( const GParameterList& src )
{
   removeAll();
   int num = src.args.getCount();
   for (int i=0; i<num; i++)
   {
      const GString& key = src.args.getKey(i);
      const GString& item = src.args.getIndexedItem(i);
      GString* newitem = new GString(item);
      args.put(key, newitem);
   }
   return *this;
}

GString GParameterList::toString () const
{
   return toParamListString("=", ", ", '"');
}

GString GParameterList::toParamListString ( const GString& equalStr, const GString& commaStr, char quote ) const
{
   GString str(512);
   int num = args.getCount();
   for (int i=0; i<num; i++)
   {
      const GString& key = args.getKey(i);
      const GString& item = args.getIndexedItem(i);
      if (i > 0)
         str += commaStr;
      str += key;
      str += equalStr;
      if (quote)
         str += quote;
      str += item;
      if (quote)
         str += quote;
   }
   return str;
}

void GParameterList::setParameter ( const GString& name, const GString& value )
{
   GString* valueToPut = new GString(value);
   if (!args.put(name, valueToPut))
      if (!args.update(name, valueToPut))
         delete valueToPut; // Shall never happen, but in case
}

GString GParameterList::getString ( const GString& name, const GString& defaultVal ) const
{
   const GString* val = args.get(name);
   if (val == null)
      return defaultVal;
   else
      return *val;
}

bool GParameterList::getBoolean ( const GString& name, bool defaultVal ) const
{
   const GString* val = args.get(name);
   if (val == null)
      return defaultVal;
   GString v = *val;
   v.toLowerCase();
   if (v == "0")
      return false;
   if (v == "false")
      return false;
   if (v == "no")
      return false;
   return true;
}

int GParameterList::getInteger ( const GString& name, int defaultVal ) const
{
   const GString* val = args.get(name);
   if (val == null)
      return defaultVal;
   try {
      return GInteger::ParseInt(*val);
   } catch (GNumberFormatException&) {
      return defaultVal;
   }
}

double GParameterList::getDouble ( const GString& name, double defaultVal ) const
{
   const GString* val = args.get(name);
   if (val == null)
      return defaultVal;
   else
      return atof(*val); // TODO: This is silly, no number format exception handling!
}

bool GParameterList::contains ( const GString& name ) const
{
   return args.containsKey(name);
}

bool GParameterList::remove ( const GString& name )
{
   return args.remove(name);
}

void GParameterList::removeAll ()
{
   return args.removeAll();
}

