/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_HASHTABLE
#define __GLIB_HASHTABLE

#include "glib/primitives/GString.h"
#include "glib/util/GEnumeration.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/exceptions/GNoSuchElementException.h"

/**
 * This hashtable maps {@link GObject} values of any type to a set of keys
 * which also can be {@link GObject}'s of any type. Both the values and their
 * associated keys must be non-null. The method {@link GObject#hashCode}
 * is used for hashing the keys, and method {@link GObject#equals} is used
 * to test for value equality.
 *
 * @author  Leif Erik Larsen
 * @since   2004.03.13
 * @param   TKey    The type of the objects that forms the keys.
 * @param   TValue  The type of the value objects to be contained in the table.
 */
template <class TKey, class TValue> class GHashtable : public GObject
{
#if __HAS_ANSI_COMPLIANT_INNER_CLASS_VISIBILITY__
   private:
#else
   public: 
#endif

      /** 
       * Used to wrap the actual data element and its "autoDelete" flag.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.11
       */
      class Entry
      {
         public:

            int hash;
            bool autoDeleteKey;
            bool autoDeleteValue;
            TKey* key;
            TValue* value;
            Entry* bucketNext;

         public:

            Entry ( int hash, TKey* key, TValue* value, bool autoDeleteKey, bool autoDeleteValue )
                : hash(hash),
                  key(key),
                  value(value),
                  autoDeleteKey(autoDeleteKey),
                  autoDeleteValue(autoDeleteValue),
                  bucketNext(null)
            {
            }

            virtual ~Entry ()
            {
               if (autoDeleteKey)
                  delete key;
               if (autoDeleteValue)
                  delete value;
            }
      };

      /** 
       * @author  Leif Erik Larsen
       * @since   2006.11.14
       */
      class Bucket
      {
         public:

            Entry* first;

            Bucket ()
               : first(null)
            {
            }

            virtual ~Bucket ()
            {
               Entry* next = first;
               while (next != null)
               {
                  Entry* e = next;
                  next = e->bucketNext;
                  delete e;
               }
            }
      };

      /** The table of buckets of hashed data. */
      Bucket** table;

      /** The total number of entries in the hash table. */
      int elementCount;

      /**
       * The table is rehashed when its size exceeds this threshold.  
       * The value of this field is (int)(capacity*loadFactor).
       */
      int threshold;

      /** The size of the bucket-table. */  
      int capacity;

      /** The load factor for the hashtable. */
      float loadFactor;

      friend class KeyEnumerator;
      friend class EntryEnumerator;

   public:

      /**
       * Class used to enumerate through all elements in a hash table,
       * by values.
       *
       * @author  Leif Erik Larsen
       * @since   2006.11.14
       * @see     #getKeyEnumerator
       * @see     #getValueEnumerator
       */
      class KeyEnumerator : public GEnumeration
      {
         private:

	         int index;
            Entry* entry;
            Entry* previousEntry;
            GHashtable& owner;

         public:
            
            KeyEnumerator ( GHashtable& owner ) 
               : index(owner.capacity),
                 entry(null),
                 previousEntry(null),
                 owner(owner)
            {
            }

            virtual ~KeyEnumerator () 
            {
            }
            
            virtual bool hasMoreElements () 
            { 
               if (entry != null)
                  return true;

               if (previousEntry != null)
               {
                  entry = previousEntry->bucketNext;
                  if (entry != null)
                  {
                     previousEntry = entry;
                     return true;
                  }
               }
               while (index > 0)
               {
                  Bucket* bucket = owner.table[--index];
                  if (bucket != null)
                  {
                     entry = bucket->first;
                     if (entry != null)
                     {
                        previousEntry = entry;
                        return true;
                     }
                  }
               }
               return false;
            }
            
            virtual TKey* nextElement () 
            { 
               if (entry == null && !hasMoreElements())
                  gthrow_(GNoSuchElementException(""));
               Entry& ret = *entry;
               entry = null; // Look for the next element upon next call!
               return ret.key; 
            }
      };

      /**
       * Class used to enumerate through all elements in a hash table,
       * by keys.
       *
       * @author  Leif Erik Larsen
       * @since   2006.11.14
       * @see     #getKeyEnumerator
       * @see     #getValueEnumerator
       */
      class ValueEnumerator : public GEnumeration
      {
         private:

	         int index;
            Entry* entry;
            Entry* previousEntry;
            GHashtable& owner;

         public:

            ValueEnumerator ( GHashtable& owner ) 
               : index(owner.capacity),
                 entry(null),
                 previousEntry(null),
                 owner(owner)
            {
            }
            
            virtual ~ValueEnumerator () 
            {
            }
            
            virtual bool hasMoreElements () 
            { 
               if (entry != null)
                  return true;

               if (previousEntry != null)
               {
                  entry = previousEntry->bucketNext;
                  if (entry != null)
                  {
                     previousEntry = entry;
                     return true;
                  }
               }
               while (index > 0)
               {
                  Bucket* bucket = owner.table[--index];
                  if (bucket != null)
                  {
                     entry = bucket->first;
                     if (entry != null)
                     {
                        previousEntry = entry;
                        return true;
                     }
                  }
               }
               return false;
            }
            
            virtual TValue* nextElement () 
            { 
               if (entry == null && !hasMoreElements())
                  gthrow_(GNoSuchElementException(""));
               Entry& ret = *entry;
               entry = null; // Look for the next element upon next call!
               return ret.value; 
            }
      };

   public:

      /**
       * Constructs a new, empty hashtable with the specified initial 
       * capacity and the specified load factor.
       *
       * @param  initialCapacity  The initial capacity of the hashtable.
       * @param  loadFactor       The load factor of the hashtable.
       * @throws GIllegalArgumentException If the initial capacity is less 
       *            than zero, or if the load factor is nonpositive.
       */
      explicit GHashtable ( int initialCapacity = 101, float loadFactor = 0.75F ) 
                     : table(null),
	                    elementCount(0),
                       threshold(0),
                       capacity(0),
                       loadFactor(loadFactor)
      {
         if (initialCapacity < 0)
            gthrow_(GIllegalArgumentException("initialCapacity < 0"));

         if (loadFactor <= 0)
            gthrow_(GIllegalArgumentException("loadFactor <= 0"));

         if (initialCapacity == 0)
            initialCapacity = 1;

         table = new Bucket*[initialCapacity];
         memset(table, 0, initialCapacity * sizeof(Bucket*));
         threshold = int(initialCapacity * loadFactor);
         capacity = initialCapacity;
      }

      virtual ~GHashtable () 
      { 
         clear();
         delete [] table;
      }

   private:

      /** Disable the copy constructor. */
      GHashtable ( const GHashtable& );

      /** Disable the assignment operator. */
      void operator= ( const GHashtable& );

   private:

      /** 
       * Get get the entry that is mapped to the specified key, 
       * or null if the key is unknown. 
       * 
       * @author  Leif Erik Larsen
       * @since   2005.01.07
       */
      Entry* getEntry ( const GObject& key )
      {
         int hash = key.hashCode();
         int index = (hash & 0x7FFFFFFF) % capacity;
         Bucket* b = table[index];
         if (b == null)
            return null;
         Entry* e = b->first;
         while (e != null) 
         {
            if ((e->hash == hash) && (e->key->equals(key))) 
               return e;
            e = e->bucketNext;
         }
         return null;
      }

   public:

      /**
       * Deletes all items from this hashtable.
       */
      void clear ()
      {
         for (int i=0; i<capacity; i++) 
         {
            delete table[i];
            table[i] = null;
         }
         elementCount = 0;
      }

      /**
       * Return true if the hashtable contains the specified key.
       * 
       * @author  Leif Erik Larsen
       * @since   2005.01.07
       */
      bool containsKey ( const TKey& key ) const
      {
         GHashtable* self = const_cast<GHashtable*>(this);
         Entry* e = self->getEntry(key);
         return e != null;
      }

      /**
       * Returns the value to which the specified key is mapped in 
       * this hashtable, or null if the key is not mapped to any 
       * value in this hashtable. Will also return null if the key
       * is associated with a null object.
       */
      TValue* get ( const TKey& key )
      {
         Entry* e = getEntry(key);
         if (e == null)
            return null;
         return e->value;
      }

      /**
       * @author  Leif Erik Larsen
       * @since   2006.11.14
       */
      KeyEnumerator getKeyEnumerator ()
      {
         return KeyEnumerator(*this);
      }

      /**
       * @author  Leif Erik Larsen
       * @since   2006.11.14
       */
      ValueEnumerator getValueEnumerator ()
      {
         return ValueEnumerator(*this);
      }

      /**
       * Return true if the hashtable contains no objects.
       *
       * @author  Leif Erik Larsen
       * @since   2006.11.13
       */
      bool isEmpty () const
      {
         return elementCount == 0;
      }

      /**
       * @author  Leif Erik Larsen
       * @since   2006.11.14
       */
      Bucket* putNewlyHashedEntry ( Entry* newEntry, Bucket* reuseBucketObj )
      {
         int hash = newEntry->hash;
         int index = (hash & 0x7FFFFFFF) % capacity;
         if (table[index] == null)
         {
            if (reuseBucketObj == null)
            {
               table[index] = new Bucket();
            }
            else
            {
               reuseBucketObj->first = null;
               table[index] = reuseBucketObj;
               reuseBucketObj = null;
            }
         }

         // Find the end of the bucket.
         Entry* first = table[index]->first;
         Entry* prev = null;
         Entry* e = first;
         while (e != null)
         {
            // Makes sure the hashtable contains only one of each key.
            if ((e->hash == hash) && (e->key->equals(*newEntry->key))) 
            {
               // Replace the old entry with the new entry.
               if (e == first)
               {
                  newEntry->bucketNext = first->bucketNext;                  
                  table[index]->first = newEntry;
               }
               else
               {
                  prev->bucketNext = newEntry;
                  newEntry->bucketNext = e->bucketNext;
               }
               delete e;
               return reuseBucketObj;
            }
            prev = e;
            e = e->bucketNext;
         }

         if (e == first)
            table[index]->first = newEntry;
         else
            prev->bucketNext = newEntry;
         elementCount++;
         return reuseBucketObj;
      }

      /**
       * Maps the specified key to the specified value in this hashtable. 
       * Neither the key nor the value can be null.
       *
       * The value can be retrieved by calling the {@link #get} method 
       * with a key that is equal to the original key.
       *
       * @throws GIllegalArgumentException If the key or value is null.
       */
      void put ( TKey* key, 
                 TValue* value,
                 bool autoDeleteKey,
                 bool autoDeletValue )
      {
         if (key == null)
            gthrow_(GIllegalArgumentException("key == null"));

         int hash = key->hashCode();
         Entry* newEntry = new Entry(hash, key, value, autoDeleteKey, autoDeletValue);
         putNewlyHashedEntry(newEntry, null);

         // Rehash the table if the threshold is exceeded.
         if (elementCount >= threshold - 1) 
            rehash();
      }

      /**
       * Maps the specified key to the specified value in this hashtable. 
       * Neither the key nor the value can be null.
       *
       * The value can be retrieved by calling the {@link #get} method 
       * with a key that is equal to the original key.
       *
       * @throws GIllegalArgumentException If the key or value is null.
       */
      void put ( const TKey& key, 
                 TValue* value,
                 bool autoDeletValue )
      {
         TKey* keyClone = new TKey(key);
         put(keyClone, value, true, autoDeletValue);
      }

      /**
       * Increases the capacity of and internally reorganizes this 
       * hashtable, in order to accommodate and access its entries more 
       * efficiently.
       *
       * This method is called automatically when the 
       * number of keys in the hashtable exceeds this hashtable's capacity 
       * and load factor.
       */
      void rehash ()
      {
         Bucket** oldTable = table;
         int oldCapacity = capacity;

         capacity += int(capacity * loadFactor) + 1;
         table = new Bucket*[capacity];
         memset(table, 0, capacity * sizeof(Bucket*));
         threshold = int(capacity * loadFactor);
         elementCount = 0;

         for (int i=0; i<oldCapacity; i++) 
         {
            Bucket* old = oldTable[i];
            if (old != null) 
            {
               oldTable[i] = null;
               Entry* e = old->first;
               old->first = null;
               while (e != null)
               {
                  Entry* next = e->bucketNext;
                  e->bucketNext = null;
                  old = putNewlyHashedEntry(e, old);
                  e = next;
               }
               if (old != null)
                  delete old;
            }
         }

         delete [] oldTable;
      }

      /**
       * Removes the first object which key hashcode equals the hashcode 
       * of the specified key and which equals() method also returns true. 
       * If there are more than one object with the specified key hashcode 
       * then we will remove only the first one.
       *
       * @author  Leif Erik Larsen
       * @since   2006.11.13
       * @return  True if we did indeed find and remove one object matching 
       *          the specified key, or else false if the are no such
       *          objects in the container.
       */
      bool remove ( const TKey& key )
      {
         int hash = key.hashCode();
         int index = (hash & 0x7FFFFFFF) % capacity;
         Bucket* bucket = table[index];
         if (bucket == null)
            return false;
         Entry* prev = null;
         Entry* e = bucket->first;
         while (e != null) 
         {
            if ((e->hash == hash) && (e->key->equals(key))) 
            {
               // We have found the item to be deleted:
               if (e == bucket->first)
                  bucket->first = e->bucketNext;
               else
                  prev->bucketNext = e->bucketNext;
               delete e;
               elementCount--;
               return true;
            }
            prev = e;
            e = e->bucketNext;
         }            

         return false;
      }

      /**
       * Returns the number of keys in this hashtable.
       */
      int size () const
      {
         return elementCount;
      }
};

#endif
