/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_EVENTSEMAPHORE
#define __GLIB_EVENTSEMAPHORE

#include "glib/primitives/GString.h"

/**
 * An Event Semaphore can be used to let one thread wait for
 * another thread to finish some task.
 *
 * For instance, this class is frequently used to send a message
 * synchronously from a background worker thread to the foreground GUI
 * thread. This is typically implemented by code that lets the background
 * thread post a message to the message queue of the foreground thread
 * and then enter the {@link #wait} method of an object of this class.
 * The background thread will then be in a wait state until some other
 * thread calls the {@link #notifyAll} method on the event semaphore
 * object. This is typically to be done by the foreground GUI thread
 * when it has finished getting and handling the posted message.
 *
 * @author  Leif Erik Larsen
 * @since   2000.09.26
 */
class GEventSemaphore
{
   public:

      typedef HEV HEvent;

   private:

      /** The system dependent semaphore handle. */
      HEvent hev;

   public:

      /**
       * Create a new unnamed event semaphore that is
       * private to the process.
       */
      GEventSemaphore ();

      /**
       */
      virtual ~GEventSemaphore ();

   public:

      /**
       * Get the number of times that the event semaphore has been
       * notified without any waiting thread.
       *
       * @author  Leif Erik Larsen
       * @since   2000.11.28
       */
      int getPostCount () const;

      /**
       */
      void notifyAll ();

      /**
       */
      void reset ();

      /**
       * Wait indefinitely until another thread calls {@link #notifyAll}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.26
       */
      void wait ();

      /**
       * Wait for a maximum of the specified number of milliseconds or
       * until another thread calls {@link #notifyAll}.
       *
       * If the timeout value is zero, or less than zero, then the calling
       * thread will return immediatly. Even if no other thread calls
       * {@link #notifyAll}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.23
       */
      void wait ( int timeoutMillis );
};

#endif

