/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/sys/GSystemInfo.h"
#include "glib/primitives/GInteger.h"
#include "glib/sys/gl16os2.h"

GTimeAmount::GTimeAmount () 
            :numDays(0),
             numHours(0),
             numMinutes(0),
             numSeconds(0),
             numMillis(0)
{ 
}

GTimeAmount::~GTimeAmount ()
{
}

GProcessInfo::GProcessInfo ( GSystem::ProcessID pid, 
                             GSystem::ProcessID ppid ) 
             :pid(pid), 
              ppid(ppid)
{
}

GProcessInfo::~GProcessInfo () 
{
}

bool GProcessInfo::isDescendant ( GSystem::ProcessID ancestorPid, 
                                  GHashtable<GInteger, GProcessInfo>& all )
{
   // In some rare cases it might happen that the list of all processes
   // contains circular pid/ppid references. Therefore we need to keep track
   // of which pids we have searched, in order to prevent endless loops.
   GVector<GSystem::ProcessID> visitedParents(16);

   // ---
   GProcessInfo* pi = this;
   for (;;)
   {
      if (pi->pid == ancestorPid)
         return true;

      if (pi->ppid == 0)
         return false;

      // Check if we have already visited the ppid (parent pid) 
      // and if so, return false.
      for (int i=visitedParents.getCount()-1; i>=0; i--)
         if (visitedParents[i] == pi->ppid)
            return false;

      // ---
      visitedParents.add(pi->ppid);
      pi = all.get(GInteger(pi->ppid));
      if (pi == null)
         return false;
   }
}

GSystemInfo::GSystemInfo ()
            :processes(256),
             numThreads(0),
             systemStateBufferSize(0),
             systemStateBuffer(null),
             physicalRAM(0),
             freePhysicalRAM(0)
{
   update();
}

GSystemInfo::~GSystemInfo ()
{
}

/**
 * Update all the information elements of the instance.
 */
void GSystemInfo::update ()
{
   physicalRAM = 0;
   freePhysicalRAM = 0;
   processes.clear();
   numThreads = 0;
   
   // Only one thread at a time here!
   GObject::Synchronizer synch(this);
   if (systemStateBufferSize <= 0) // If this is the first time here.
   {
      systemStateBufferSize = 4096;
      systemStateBuffer = new char[systemStateBufferSize];
   }

   // The method used here to obtain the system statistics requires 
   // a buffer which size is impossible to predict. Thus, loop to 
   // allocate a gradually larger buffer until OS/2 reports OK.
   // The first attempt is better being the same size as was 
   // reported OK by the recent time we was here.
   bool ok = true;
   for (;;)
   {
      PID pid = 0; // 0 means "all processes".
      memset(systemStateBuffer, 0, systemStateBufferSize);
      APIRET rc = ::DosQuerySysState(QS_PROCESS|QS_THREAD, 0, pid, 0, systemStateBuffer, systemStateBufferSize);
      if (rc == ERROR_BUFFER_OVERFLOW)
      {
         systemStateBufferSize += 4096;
         delete [] systemStateBuffer;
         systemStateBuffer = new char[systemStateBufferSize];
         continue;
      }
      ok = (rc == NO_ERROR);
      break;
   }

   // ---
   // Get the system statistics information.
   if (ok)
   {
      QTOPLEVEL* top = (QTOPLEVEL*) systemStateBuffer;
      QGLOBAL* g = (QGLOBAL*) top->gbldata;
      numThreads = g->threadcnt;

      // We must count the number of processes manually.
      PQPROCESS p = top->procdata;
      while (p && p->rectype == 1)
      {
         GInteger* key = new GInteger(p->pid);
         GProcessInfo *pi = new GProcessInfo(p->pid, p->ppid);
         processes.put(key, pi, true, true);
         PQTHREAD t = p->threads;
         for (int i=0; i<p->threadcnt; i++, t++)
            ; // pi->threads.add(new GThreadInfo(t->threadid));
         /*
         for (int i=0; i<p->sem16cnt; i++)
            pi->sem16s.add(new GInteger(p->sem16s[i]));
         for (int i=0; i<p->dllcnt; i++)
            pi->modules.add(new GInteger(p->dlls[i]));
         for (int i=0; i<p->shrmemcnt; i++)
            pi->sharedMems.add(new GInteger(p->shrmems[i]));
         */
         p = (PQPROCESS) t;
      }
   }

   // ---
   // Get the Physical RAM information.
   ULONG total = 0;
   ::DosQuerySysInfo(QSV_TOTPHYSMEM, QSV_TOTPHYSMEM, &total, sizeof(total));
   physicalRAM = total;
   ULONG freeRam = 0;
   ::GL16OS2_GetFreePhysicalRam(&freeRam);
   freePhysicalRAM = freeRam;

   // ---
   // Get the uptime of the system
   ulonglong ms = GSystem::CurrentTimeMillis();
   uptime.numDays = int(ms / 86400000);
   ms -= ulonglong(uptime.numDays) * ulonglong(86400000);
   uptime.numHours = int(ms / 3600000);
   ms -= uptime.numHours * 3600000;
   uptime.numMinutes = int(ms / 60000);
   ms -= uptime.numMinutes * 60000;
   uptime.numSeconds = int(ms / 1000);
   ms -= uptime.numSeconds * 1000;
   uptime.numMillis = int(ms);
}

const GTimeAmount& GSystemInfo::getUpTime ()
{
   return uptime;
}

/**
 * Get total number of physical RAM bytes that are installed on the system.
 */
ulonglong GSystemInfo::getPhysicalRAM ()
{
   return physicalRAM;
}

/**
 * Get number of physical free RAM bytes on the system.
 */
ulonglong GSystemInfo::getFreePhysicalRAM ()
{
   return freePhysicalRAM;
}

GHashtable<GInteger, GProcessInfo>& GSystemInfo::getProcessList ()
{
   return processes;
}

int GSystemInfo::getProcessCount ()
{
   return processes.size();
}

int GSystemInfo::getThreadCount ()
{
   return numThreads;
}
