/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_SYSTEM
#define __GLIB_SYSTEM

#include "glib/primitives/GString.h"
#include "glib/primitives/GDimension.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/gui/GColor.h"
#include "glib/GCmdLineParameters.h"

/**
 * This function must be defined by the program.
 * It will be called by the library initialization code in order for the
 * application program to start up.
 *
 * @author  Leif Erik Larsen
 * @since   2001.07.12
 * @param   params  The parsed version of the command line parameters that
 *                  was given to the program from the operating system.
 *                  This is the same reference as can also be accessed
 *                  by a call to {@link GProgram#getCmdLineParameters}.
 */
int gmain ( const GCmdLineParameters& params );

/**
 * This class implements various methods for acessing system dependent
 * properties in a platform independent way. It contains static methods
 * only, and cannot be instantiated.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.28
 */
class GSystem : public GObject
{
   public:

      /** 
       * A system dependent process identification (PID).
       * On Windows this is the Process-id, not the Process-handle.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.14
       */
      typedef PID ProcessID;

      /**
       * Identifiers for the various supported platforms.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.25
       * @see     #getPlatformID
       */
      enum PlatformID
      {
         Platform_OS2_2_0 = 1010, // Unsupported by GLib!
         Platform_OS2_2_1 = 1020, // Unsupported by GLib!
         Platform_OS2_2_11 = 1030, // Unsupported by GLib!
         Platform_OS2_3_0 = 1040,
         Platform_OS2_4_0 = 1050,
         Platform_OS2_4_5 = 1060
      };

      /**
       * The system metric IDs that can be used
       * with {@link #getSystemMetrics}.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.29
       */
      enum SystemMetricsID
      {
         SMID_CYTITLEBAR,  // Height of the title bar area of standard frame windows.
         SMID_CXDLGFRAME,  // Width of frame on standard dialog windows.
         SMID_CYDLGFRAME,  // Height of frame on standard dialog windows.
         SMID_INSERTMODE,  // If the insert mode is on or off.
         SMID_CXSCREEN,    // Width of the screen.
         SMID_CYSCREEN,    // Height of the screen.
         SMID_CXVSCROLL,   // Width of standard vertical scrollbars.
         SMID_CYHSCROLL,   // Height of standard horizontal scrollbars.
         SMID_CXICON,      // Width of standard icons.
         SMID_CXICONSMALL, // Width of small icons.
      };

      /**
       * The mouse cursor shape IDs that can be used
       * with {@link #setMouseCursorShape}.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.29
       */
      enum MouseCursorShapeID
      {
         MCSID_NORMAL,     // Normal cursor, usually an arror.
         MCSID_TEXT,       //
         MCSID_WAIT,       //
         MCSID_MOVE,       //
         MCSID_SIZENWSE,   //
         MCSID_SIZENESW,   //
         MCSID_SIZEWE,     // Size west/east arrow.
         MCSID_SIZENS      // Size north/south arrow.
      };

      /**
       * The system color IDs that can be used
       * with {@link #getSystemColor}.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.29
       */
      enum SystemColorID
      {
         SCID_DIALOGBCK,  // Background color of dialog boxes.
         SCID_DIALOGTXT,  // Text color of dialog boxes.
         SCID_BUTTONBCK,  // Background color of push buttons.
         SCID_BUTTONTXT,  // Text color of push buttons.
         SCID_BUTTONDBOX, // Color of the thin outer box on default or focused button.
         SCID_3DLIGHT,    // Light color for three-dimensional display elements (for edges facing the light source).
         SCID_3DSHADOW,   // Shadow color for three-dimensional display elements (for edges facing away from the light source).
         SCID_MENUBCK,    // Background color of menues.
         SCID_MENUTXT,
         SCID_MENUDISABLEDTXT,
         SCID_MENUHILITEBCK,
         SCID_MENUHILITETXT,
         SCID_SCROLLBARBCK,
         SCID_TEXTENTRYBCK,
         SCID_TEXTENTRYTXT

      };

      /**
       * Identification values to be used with {@link #IsShiftKeyDown}.
       */
      enum ShiftKeyId
      {
         SK_SHIFT,
         SK_ALT,
         SK_CONTROL,
         SK_CAPS_LOCK,
         SK_NUM_LOCK,
         SK_SCROLL_LOCK,

         /** Left mouse button. */
         SK_MOUSE_BUTTON1,

         /** Right mouse button. */
         SK_MOUSE_BUTTON2
      };

      /** 
       * Flags supported by {@link #BreakProcess}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.25
       */
      enum BreakType
      { 
         /** 
          * The softest way to break a child process. Typically the 
          * same as pressing Ctrl+C on the keyboard when some console 
          * window has input focus. Will not kill all processes. It 
          * depends on the Ctrl+C handler of the process.
          */
         BT_CtrlC, 

         /** 
          * Also a soft way to break a child process. Typically the 
          * same as pressing Ctrl+Break on the keyboard when some console 
          * window has input focus. Will not kill all processes. It 
          * depends on the Ctrl+Break handler of the process.
          */
         BT_CtrlBreak,

         /** 
          * A quite hard way of breaking a child process. Will invoke 
          * some kind of exit() function on the target process.
          * Will not kill all processes. It depends on the exit-hook
          * of the process.
          */
         BT_Exit,

         /** 
          * The most brutal way of breaking a child process. 
          * Should kill all processes. But process cleanup code 
          * might not be run, so this method might be somewhat unstable.
          */
         BT_Kill 
      };

   private:

      static GObject LockGetCpuUsageInPercent;

   private:

      /**
       * Prevent this class from being instantiated.
       */
      GSystem ();
      virtual ~GSystem ();

   private:

      static GVector<GSystem::ProcessID> GetProcessPids ( GSystem::ProcessID pid, bool inclTree );

   public:

      /**
       * This is a static method to perform a system beep with the specified
       * frequency and duration in milliseconds.
       */
      static void Beep ( int freq = 400, int ms = 50 );

      /**
       * Break or kill the specified process and optionally all its 
       * descendant child processes (if any).
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.14
       * @param   pid       The ID of which process to break.
       * @param   bt        How hard we should try to break it.
       * @param   breakTree True if we shall break all the descendant child 
       *                    processes of the specified process, including 
       *                    the specified process it self. If false is
       *                    specified then we will break the process it self
       *                    only, and its child processes (if any) will
       *                    continue to run until they finish normally.
       * @return  True on success, or else false.
       */
      static bool BreakProcess ( ProcessID pid, 
                                 BreakType bt,
                                 bool breakTree );

      /**
       * Put the specified text on the System Clipboard.
       *
       * We will allocate a mirror buffer of which will be given to the
       * system clipboard. Therefore, the calling function should manually
       * free the specifed text if it was allocated from the heap.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.13
       * @param   text  Text string of which to copy.
       * @return  True on success, or else false.
       * @see     #getClipboardText
       */
      static bool CopyTextToClipboard ( const GString& text );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.06
       * @param   dstPath Path to which file the shell object should point.
       * @param   workDir The working directory of the shell object, when 
       *                  it is launched by the user.
       * @param   name    The name of the new object, as it will occur 
       *                  on the system shell (e.g. Desktop on Windows).
       */
      static void CreateSystemShellObject ( const GString& dstPath,
                                            const GString& workDir,
                                            const GString& name );

      /**
       * Get the current system time counter in milliseconds.
       * This is the system uptime. It will support longer uptimes
       * than those ~49.7 days.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.28
       */
      static ulonglong CurrentTimeMillis ();

      /**
       * Get a copy of the error message string with respect to the
       * specified APIRET value. If rc is NO_ERROR then we will return
       * an empty string, else we will return the error text.
       */
      static GString GetApiRetString ( APIRET rc );
      static GString GetApiRetString ( class GError& err );

      /**
       * Get a copy of the text that is currently on the system clipboard.
       * If the system clipboard currently contains data that is not
       * text-compatible, or if the clipboard does not contain any data
       * at all, then return an empty string.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.13
       * @see     #copyTextToClipboard
       * @see     #isAnyTextOnClipboard
       */
      static GString GetClipboardText ();

      /**
       * Get the average CPU utilization value since this same method was
       * previously called. On NT the value is taken from the performance
       * counters and on 9x the value is taken from the registry.
       *
       * NOTE: PC system time is not very precise (~10ms resolution),
       * so use sufficiently long sampling intervals if you make use of
       * this method.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.25
       */
      static int GetCpuUsageInPercent ();

      /**
       * Get a string object containing the content of the LIBPATH
       * statement in the CONFIG.SYS in the root directory of the boot
       * drive of the current system.
       *
       * @return   The LIBPATH on success, or else an empty string.
       */
      static GString GetLibPath ();

      /**
       * Get an ID of which platform we are currently running.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.25
       */
      static PlatformID GetPlatformID ();

      /**
       * Get the name and version of the platform in which we are 
       * currently running.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.12
       */
      static GString GetPlatformName ();

      /**
       * Get the dimension of the current physical or virtual screen.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.13
       */
      static GDimension GetScreenSize ();

      /**
       * Get the system dependent color for the specified color ID.
       *
       * @author  Leif Erik Larsen
       * @since   2003.11.20
       * @throws  GIllegalArgumentException if the specified id is not
       *                                    part of the enumeration.
       */
      static GColor GetSystemColor ( SystemColorID id );

      /**
       * Get the system dependent value for the specified system
       * metric ID.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.28
       * @throws  GIllegalArgumentException if the specified id is not
       *                                    part of the enumeration.
       */
      static int GetSystemMetrics ( SystemMetricsID id );

      /**
       * Gut the specified string value from the system dependent
       * registry (profile management subsystem).
       * <p>
       * <b>Note</b> that the returned string can never be more than
       * 32000 characters long.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.30
       * @see     #setSystemRegistryString
       */
      static GString GetSystemRegistryString ( const GString& sectionName,
                                               const GString& entryName,
                                               const GString& defaultValue = GString::Empty );

      /**
       * Get the home directory of the current system defined user.
       * The returned directory will contain a trailing slash.
       *
       * <b>On OS/2 and Win95/98/ME:</b> This is the directory defined by 
       * the %HOME% environment variable. If this environment variable is 
       * undefined then we will return the directory of where the application
       * exie-file is contained.
       *
       * <b>On WinNT/2000/XP:</b> This is typically 
       * "C:\Documents and Settings\X\Application Data\MyApp\", 
       * where "X" is the name of the logged on user and 
       * "MyApp" is the specified <i>applName</i> (if not empty).
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.03
       * @param   applName  The name of the calling application.
       *                    Will be used only on some systems (e.g. Windows)
       *                    to form an application specific subdirectory 
       *                    under the system default application data 
       *                    directory.
       */
      static GString GetUserApplicationDataDirectory ( const GString& applName = GString::Empty );

      /**
       * Return true if and only if the system clipboard currently
       * contains some data that is text-compatible.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.13
       * @see     #getClipboardText
       */
      static bool IsAnyTextOnClipboard ();

      /**
       * Test if the specified keyboard shift-key is currently being
       * pressed down by the user.
       *
       * @author  Leif  Erik Larsen
       * @since   2004.03.19
       * @see     #IsShiftKeyToggledOn
       */
      static bool IsShiftKeyDown ( ShiftKeyId keyId );

      /**
       * Test if the specified keyboard shift-key is currently 
       * toggled on. This method is typically most usable with 
       * {@link #SK_CAPS_LOCK}, {@link #SK_NUM_LOCK} 
       * and {@link #SK_SCROLL_LOCK}.
       *
       * @author  Leif  Erik Larsen
       * @since   2005.03.04
       * @see     #IsShiftKeyDown
       */
      static bool IsShiftKeyToggledOn ( ShiftKeyId keyId );

      /**
       * Return true if and only if the current operating system is some 
       * version of OS/2 (not Windows, Linux, etc.).
       *
       * @author  Leif  Erik Larsen
       * @since   2005.07.22
       * @see     #IsPlatformWindows
       */
      static bool IsPlatformOS2 ();

      /**
       * Return true if and only if the current operating system is some 
       * version of Windows (not OS/2, Linux, etc.).
       *
       * @author  Leif  Erik Larsen
       * @since   2005.07.22
       * @see     #IsPlatformOS2
       */
      static bool IsPlatformWindows ();

      /**
       * Open the system dependent "Resycle Bin" (on Windows) or 
       * "Trashcan" (on OS/2) shell folder object.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.08
       * @return  True on success, or else false on any error.
       */
      static bool OpenRecycleBinObject ();

      /**
       * Execute the specified path as a system dependent shell object.
       * The path can be a directory, a program file or some data file. 
       * If it is a directory then we will open it as a system shell folder.
       * If it is a program file then the programn will be launched. 
       * If it is a data file then the associated application will be 
       * launched with the specified file as its input argument.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.09
       * @return  True on success, or else false on any error.
       */
      static bool OpenShellObject ( const GString& path );

      /**
       * Set the mouse cursor shape.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.29
       */
      static void SetMouseCursorShape ( MouseCursorShapeID id );

      /**
       * Put the specified string value into the system dependent
       * registry (profile management subsystem).
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.28
       * @see     #getSystemRegistryString
       */
      static APIRET SetSystemRegistryString ( const GString& sectionName,
                                              const GString& entryName,
                                              const GString& value );

      /**
       * @return    The string on success, or else an empty string.
       */
      static GString ScanSystemConfig ( const GString& keyWord );

      /**
       * Compare the two strings using the current code page and
       * country code. 
       * <p>
       * TODO: Doesn't respect code page and country code yet.
       *
       * @param    str1     The first of the two strings to compare.
       * @param    str2     The second of the two strings to compare.
       * @param    caseSen  True if the two strings should be compared as
       *                    case sensitive strings, or else false.
       * @return   0 if the two strings are equal, less than 0 if str1 is
       *           less than str2 or greater that 0 if str1 is greater
       *           than str2.
       */
      static int StrCompare ( const GString& str1,
                              const GString& str2,
                              bool caseSen );

      /**
       * Convert all lowercase the characters in the specified string
       * to uppercase, using the current code page and country code.
       *
       * @return   A reference to the same string as was specified
       *           in <i>str</i>.
       */
      static GString& StrUpper ( GString& str );
};

#endif

