/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_SYS_OS2
#define __GLIB_SYS_OS2

/**
 * This class contains powerful exception handlers, as well as some
 * easy-to-use macros for them.
 *
 * <B>Introduction</B>
 *
 * OS/2 exception handlers are a mess to program and, if installed wrongly,
 * almost impossible to debug. The problem is that for any program that
 * does a bit more than showing a message box, using exception handlers
 * is a must to avoid system hangs. This especially applies to multi-thread
 * programs using mutex semaphores (more on that below). The functions
 * and macros in here are designed to make that more simple.
 *
 * The GTRY/GCATCH-macros automatically insert code for properly registering
 * and deregistering the exception handlers. You should ALWAYS use these
 * macros instead  of directly registering the handlers to avoid
 * accidentally forgetting to deregister them. If you forget to deregister
 * an exception handler, this can lead to really strange errors (crashes,
 * hangs) which are nearly impossible to debug because the thread's
 * stack probably got completely messed up.
 *
 * The general idea of these macros is to define GTRY/GCATCH blocks similar
 * to C++. If an exception occurs in the GTRY block, execution is
 * transferred to the GCATCH block. (This works in both C and C++,
 * by the way.)
 *
 * The general usage is like this:
 *
 * <pre>
 *
 * int your_protected_func ( int ... )
 * {
 *    GTRY(e, true) { // true=with log, false=without log
 *       char *p = NULL;
 *       // ... the stuff in here is protected by the exception handler.
 *       *p = "A";
 *    } GCATCH(e) {
 *       // ... exception occured: react here.
 *    } GENDCATCH();
 * }
 *
 * </pre>
 *
 * The GCATCH block is _required_ even if you do nothing in there, because
 * the GCATCH() macro will deregister the handler.
 *
 * The "e" (exception-id) parameter to GTRY can be any C identifier which
 * is not used in your current variable scope, e.g. "excpt1". This is used
 * for creating an EXCEPTREG2 variable of that name on the stack. The "e"'s
 * in GTRY and GCATCH must match, since this is where the macros store
 * the exception handler data.
 *
 * These macros may be nested if you use different "e"'s for sub-macros.
 *
 * Inside the GTRY and GCATCH blocks, you must not use "goto" (to a location
 * outside the block) or "return", because this will not deregister
 * the handler.
 *
 * Keep in mind that all the code in the GTRY block is protected by the
 * handler, including all functions that get called. So if you enclose your
 * main() code in a GTRY block, your entire application is protected. If any
 * subfunction fails, execution is transferred to the closest GCATCH() that
 * was installed (as with C++ try and catch).
 *
 * <B>Asynchronous exceptions</B>
 *
 * The exception handlers in this file (which are installed with the
 * GTRY/GCATCH mechanism) only intercept synchronous exceptions, most
 * importantly, XCPT_ACCESS_VIOLATION. They do not protect your code
 * against asynchronous exceptions.
 *
 * OS/2 defines asynchronous exceptions to be those that
 * can be delayed. With OS/2, there are only three of these:
 *
 * -- XCPT_PROCESS_TERMINATE
 * -- XCPT_ASYNC_PROCESS_TERMINATE
 * -- XCPT_SIGNAL (thread 1 only)
 *
 * To protect yourself against these also, put the section
 * in question in a DosEnterMustComplete/DosExitMustComplete
 * block as well.
 *
 * <B>Mutex semaphores</B>
 *
 * The problem with OS/2 mutex semaphores is that they are sometimes not
 * automatically released when a thread terminates. If there are several
 * mutexes involved and they are released in improper order, you can get
 * zombie threads on exit. Even worse, if this happens to a PM thread,
 * this will hang the system.
 *
 * As a result, you should protect any section of code which requests a
 * semaphore with the exception handlers. To protect yourself against thread
 * termination, use must-complete sections as well (but be careful with
 * those if your code takes a long time to execute... but then you
 * shouldn't request a mutex in the first place).
 *
 * So _whenever_ you request a mutex semaphore, enclose the block with
 * GTRY/GCATCH in case the code crashes. Besides, enclose the GTRY/GCATCH
 * block in a must-complete section, like this:
 *
 * <pre>
 *
 * HMTX hmtx = ...
 *
 * int your_func ( int )
 * {
 *    BOOL  fSemOwned = FALSE;
 *    ULONG ulNesting = 0;
 *
 *    DosEnterMustComplete(&ulNesting);
 *    GTRY(excpt1, true) {
 *       fSemOwned = !WinRequestMutexSem(hmtx, ...);
 *       if (fSemOwned)
 *          ; // ... work on your protected data mutex gets released below
 *    } GCATCH(excpt1) {
 *    } GENDCATCH();
 *
 *    if (fSemOwned)
 *    {
 *       // this gets executed always, even if an exception occured
 *       DosReleaseMutexSem(hmtx);
 *       fSemOwned = FALSE;
 *    }
 *    DosExitMustComplete(&ulNesting);
 * }
 *
 * </pre>
 *
 * This way your mutex semaphore gets released in every
 * possible condition.
 */
#if 1
class GOS2Exception
{
   public:

      /**
       * Replacement class for OS/2's EXCEPTIONREGISTRATIONRECORD.
       */
      class Handler
      {
         private:

            volatile void* pNext; // As in OS/2's EXCEPTIONREGISTRATIONRECORD.
            volatile PFN pfnHandler; // Ditto.

         public:

            jmp_buf jmpThread; // Buffer for setjmp().
            BOOL log; // True if exception is to be logged.

         private:

            int handleIdx; // Index of this structure instance in handlers[].
            BOOL active;

            EXCEPTIONREPORTRECORD rep;
            CONTEXTRECORD ctx;

         public:

            Handler ( BOOL log );
            ~Handler ();

         public:

            GString getDescription ();
            void deactivate ();

         private:

            static GString describePage ( ULONG ulCheck );
            static ULONG _System exceptionHandler ( EXCEPTIONREPORTRECORD* prep, Handler* preg2, CONTEXTRECORD* pctx, void* pv );
      };

   private:

      GOS2Exception () {}
      ~GOS2Exception () {}

   public:

      #define GTRY(e, l)                                                \
              {                                                         \
                 GOS2Exception::Handler e(l);                           \
                 if (setjmp(e.jmpThread) == 0)                          \
                 {

                    // User defined and exception-protected code
                    // comes here...

      #define GCATCH(e)                                                 \
                 }                                                      \
                 else                                                   \
                 {                                                      \
                    e.deactivate();                                     \
                    if (e.log)                                          \
                    {                                                   \
                       GString msg = e.getDescription();                \
                       GLog::Log(null, msg);                            \
                    }

      #define GENDCATCH()                                               \
                 }                                                      \
              }
};
#endif

// #define OPEN_UNKNOWN      -1
#define OPEN_DEFAULT       0 // Used when calling WinOpenObject() or PFNWINOPENOBJECT()
// #define OPEN_CONTENTS      1
// #define OPEN_SETTINGS      2
// #define OPEN_HELP          3
// #define OPEN_RUNNING       4
// #define OPEN_PROMPTDLG     5

#ifndef DP_MINIICON // Undocumented (?) flag to WinDrawPointer().
#  define DP_MINIICON 0x0004
#endif

typedef HFILE GSysFileHandle;

#ifdef __cplusplus
extern "C" {
#endif

#if __IBMCPP4__ || __IBMCPP__
   // Some undocumented OS/2 API's that we need to use.
   #pragma import(DosQuerySysState, "", "DOSCALLS", 368)
   #pragma import(DosPerfSysCall, "", "DOSCALLS", 976)
   #pragma import(DosOpenL, "", "DOSCALLS", 981)
   #pragma import(DosSetFilePtrL, "", "DOSCALLS", 988)
   #pragma import(DosSetFileSizeL, "", "DOSCALLS", 989)

   // Wrappers for some 16-bit API's, since IBM VAC++ 4 cannot use them directly.
   #pragma import(GL16OS2_Test, "", "GL16OS2", 1)
   #pragma import(GL16OS2_GetFreePhysicalRam, "", "GL16OS2", 2)
#elif __EMX__
   // Compiling with EMX (GNU C++ on OS/2) should be ok.
#else
#error Unsupported compiler!
#endif

#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif

// Large File Support > 2GB.
APIRET APIENTRY DosOpenL ( PCSZ  pszFileName, PHFILE phf, PULONG pulAction, longlong cbFile, ULONG ulAttribute, ULONG fsOpenFlags, ULONG fsOpenMode, PEAOP2 peaop2 );
APIRET APIENTRY DosSetFilePtrL ( HFILE hFile, longlong ib, ULONG method, longlong* ibActual );
APIRET APIENTRY DosSetFileSizeL ( HFILE hFile, longlong cbSize );
typedef struct _FILESTATUS3L     /* fsts3L */
{
   FDATE    fdateCreation;
   FTIME    ftimeCreation;
   FDATE    fdateLastAccess;
   FTIME    ftimeLastAccess;
   FDATE    fdateLastWrite;
   FTIME    ftimeLastWrite;
   longlong cbFile;
   longlong cbFileAlloc;
   ULONG    attrFile;
} FILESTATUS3L;
typedef FILESTATUS3L *PFILESTATUS3L;
typedef struct
{
   ULONG    oNextEntryOffset;            /* new field */
   FDATE    fdateCreation;
   FTIME    ftimeCreation;
   FDATE    fdateLastAccess;
   FTIME    ftimeLastAccess;
   FDATE    fdateLastWrite;
   FTIME    ftimeLastWrite;
   longlong cbFile;
   longlong cbFileAlloc;
   ULONG    attrFile;                    /* widened field */
   UCHAR    cchName;
   CHAR     achName[CCHMAXPATHCOMP];
} FILEFINDBUF3L;
#define FIL_STANDARDL 11

/**
 * @param  command  ID of which system command to call.
 *                  E.g. {@link CMD_KI_RDCNT} to read the 
 *                  OS/2 perrformance counters.
 * @param  parm1    First parameter. Depends on the command.
 *                  For {@link CMD_KI_RDCNT} this parameter should 
 *                  be a pointer to a {@link CPUUTIL}.
 * @param  parm2    Second parameter. Depends on the command.
 *                  For {@link CMD_KI_RDCNT} this parameter should be null.
 * @param  parm3    Third parameter. Depends on the command.
 *                  For {@link CMD_KI_RDCNT} this parameter should be null.
 */
APIRET APIENTRY DosPerfSysCall ( ULONG command, 
                                 ULONG parm1,
                                 ULONG parm2, 
                                 ULONG parm3 );


/**
 * DosQuerySysState() returns information about various resources in use by
 * the system. The <i>entityList</i> paramter determines which information is
 * returned according to the bits sets in this parameter.
 *
 * The information returned begins with a pointer to the global record
 * structure, QGLOBAL. Following this will be a series of other records
 * which depend on which information was requested. Some of these subsequent
 * record structures contain an identifier as its first member, which enables
 * the returned information to be interpreted without any order being
 * imposed. Entities that may be requested are:
 *
 * Process information (QS_PROCESS, 0x0001)<br>
 * Semaphore informaion (QS_SEMAPHORE, 0x0002)<br>
 * Module information (QS_MTE, 0x0004)<br>
 * File system information (QS_FILESYS, 0x0008)<br>
 * Shared memory information (QS_SHMEMORY, 0x0010)<br>
 * Module Version information (QS_MODVER, 0x0200)<br>
 *
 * Not all entities have beed supported on earlier versions of OS2.
 *
 * @param  entityList  Determines what information is returned.
 *                     May be a combination of the various QS_* flags.
 * @param  entityLevel Determines the extent on information returned for a
 *                     given entity. Currently this only applies to QS_MTE
 *                     entities. If EntityLevel is also set to QS_MTE then
 *                     module object information is returned.
 * @param  pid         Restricts information to a particular process id.
 *                     If 0 is specified then entities for all processes
 *                     are returned.
 * @param  tid         Restricts information to a particular thread id.
 *                     Currently a value of zero only is supported,
 *                     requesting all threads of a process.
 * @param  dataBuff    Pointer to the buffer allocated by the user into
 *                     which entity structures are returned.
 *                     If the buffer is of insufficient size then a
 *                     ERROR_BUFFER_OVERFLOW is returned.
 * @param  cbBuf       Size of the buffer pointed to by pDataBuf, in bytes.
 */
APIRET APIENTRY DosQuerySysState ( ULONG entityList,
                                   ULONG entityLevel, 
                                   ULONG pid, 
                                   ULONG tid, 
                                   PVOID dataBuff, 
                                   ULONG cbBuf );
#ifdef __cplusplus
}
#endif

#define CMD_KI_RDCNT    0x63

typedef struct _CPUUTIL {
    ULONG ulTimeLow;     /* Low 32 bits of time stamp      */
    ULONG ulTimeHigh;    /* High 32 bits of time stamp     */
    ULONG ulIdleLow;     /* Low 32 bits of idle time       */
    ULONG ulIdleHigh;    /* High 32 bits of idle time      */
    ULONG ulBusyLow;     /* Low 32 bits of busy time       */
    ULONG ulBusyHigh;    /* High 32 bits of busy time      */
    ULONG ulIntrLow;     /* Low 32 bits of interrupt time  */
    ULONG ulIntrHigh;    /* High 32 bits of interrupt time */
} CPUUTIL, *PCPUUTIL;

#define QS_PROCESS      0x0001
#define QS_SEMAPHORE    0x0002
#define QS_MTE          0x0004
#define QS_FILESYS      0x0008
#define QS_SHMEMORY     0x0010
#define QS_DISK         0x0020
#define QS_HWCONFIG     0x0040
#define QS_NAMEDPIPE    0x0080
#define QS_THREAD       0x0100
#define QS_MODVER       0x0200

/** Valid <i>entityList</i> bit settings. */
#define QS_SUPPORTED (QS_PROCESS|QS_SEMAPHORE|QS_MTE|QS_FILESYS|QS_SHMEMORY|QS_MODVER)

typedef struct {
        ULONG   threadcnt;      /* Number of threads */
        ULONG   sem32cnt;       /* Number of 32 bit semaphores */
        ULONG   modulecnt;      /* Number of modules */
} QGLOBAL, *PQGLOBAL;

typedef struct {
        ULONG   rectype;        /* 256 for thread */
        USHORT  threadid;
        USHORT  slotid;
        ULONG   sleepid;
        ULONG   priority;
        ULONG   systime;
        ULONG   usertime;
        UCHAR   state;
        UCHAR   _reserved1_;    /* padding to ULONG */
        USHORT  _reserved2_;    /* padding to ULONG */
} QTHREAD, *PQTHREAD;

typedef struct {
        USHORT  sfn;
        USHORT  refcnt;
        ULONG   flags;
        ULONG   accmode;
        ULONG   filesize;
        USHORT  volhnd;
        USHORT  attrib;
        USHORT  _reserved_;
} QFDS, *PQFDS;

typedef struct qfile {
        ULONG           rectype;        /* 8 for file */
        struct qfile    *next;
        ULONG           opencnt;
        PQFDS           filedata;
        char            name[1];
} QFILE, *PQFILE;

typedef struct {
        ULONG   rectype;        /* 1 for process */
        PQTHREAD threads;
        USHORT  pid;
        USHORT  ppid;
        ULONG   type;
        ULONG   state;
        ULONG   sessid;
        USHORT  hndmod;
        USHORT  threadcnt;
        ULONG   privsem32cnt;
        ULONG   _reserved2_;
        USHORT  sem16cnt;
        USHORT  dllcnt;
        USHORT  shrmemcnt;
        USHORT  fdscnt;
        PUSHORT sem16s;
        PUSHORT dlls;
        PUSHORT shrmems;
        PUSHORT fds;
} QPROCESS, *PQPROCESS;

typedef struct sema {
        struct sema *next;
        USHORT  refcnt;
        UCHAR   sysflags;
        UCHAR   sysproccnt;
        ULONG   _reserved1_;
        USHORT  index;
        CHAR    name[1];
} QSEMA, *PQSEMA;

typedef struct {
        ULONG   rectype;
        ULONG   _reserved1_;
        USHORT  _reserved2_;
        USHORT  syssemidx;
        ULONG   index;
        QSEMA   sema;
} QSEMSTRUC, *PQSEMSTRUC;

typedef struct {
        USHORT  pid;
        USHORT  opencnt;
} QSEMOWNER32, *PQSEMOWNER32;

typedef struct {
        PQSEMOWNER32    own;
        PCHAR           name;
        PVOID           semrecs; /* array of associated sema's */
        USHORT          flags;
        USHORT          semreccnt;
        USHORT          waitcnt;
        USHORT          _reserved_; /* padding to ULONG */
} QSEMSMUX32, *PQSEMSMUX32;

typedef struct {
        PQSEMOWNER32    own;
        PCHAR           name;
        PQSEMSMUX32     mux;
        USHORT          flags;
        USHORT          postcnt;
} QSEMEV32, *PQSEMEV32;

typedef struct {
        PQSEMOWNER32    own;
        PCHAR           name;
        PQSEMSMUX32     mux;
        USHORT          flags;
        USHORT          refcnt;
        USHORT          thrdnum;
        USHORT          _reserved_;     /* padding to ULONG */
} QSEMMUX32, *PQSEMMUX32;

typedef struct semstr32 {
        struct semstr32 *next;
        QSEMEV32 evsem;
        QSEMMUX32  muxsem;
        QSEMSMUX32 smuxsem;
} QSEM32STRUC, *PQSEM32STRUC;

typedef struct shrmem {
        struct shrmem *next;
        USHORT  hndshr;
        USHORT  selshr;
        USHORT  refcnt;
        CHAR    name[1];
} QSHRMEM, *PQSHRMEM;

typedef struct module {
        struct module *next;
        USHORT  hndmod;
        USHORT  type;
        ULONG   refcnt;
        ULONG   segcnt;
        PVOID   _reserved_;
        PCHAR   name;
        USHORT  modref[1];
} QMODULE, *PQMODULE;

typedef struct {
        PQGLOBAL        gbldata;
        PQPROCESS       procdata;
        PQSEMSTRUC      semadata;
        PQSEM32STRUC    sem32data;      /* not always present */
        PQSHRMEM        shrmemdata;
        PQMODULE        moddata;
        PVOID           _reserved2_;
        PQFILE          filedata;       /* only present in FP19 or later or W4 */
} QTOPLEVEL, *PQTOPLEVEL;

#endif // #ifndef __GLIB_SYS_OS2
