/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <setjmp.h>
#include <assert.h>
#include "glib/sys/GGlue.h"
#include "glib/sys/GSystem.h"
#include "glib/sys/gl16os2.h"
#include "glib/util/GLog.h"
#include "glib/GProgram.h"
#include "glib/io/GPipeInputStream.h"
#include "glib/gc/defaultheapmanager.h"

// We need to have control of the static initialization sequence.
// Thus, all static variables are defined here, in correct sequence.
const GString GString::Blank = " ";
const GString GString::Empty = "";
const GString GString::WhiteChars = " \t\r\n";
const GString GString::EOL = "\r\n";
const GString GString::ThreeDots = "...";
GThread* GThread::MainThread = null;
GProgram* GProgram::Program = null;
GCmdLineParameters* GProgram::Params = null;
const ulonglong GProgram::TimeMillisEarlyStartup = GSystem::CurrentTimeMillis();
HAB GProgram::hAB = null;
HMQ GProgram::hMQ = null;
GString GLog::LogFilePath;
int GLog::Count = 0;
GLog::FilterId GLog::FilterLevel = GLog::PROD;
FILE* GLog::Logfile = null;
GObject GLog::Lock;
const GString GVfsLocal::SlashStr = "\\";
const GString GVfsArchiveFile::SlashStr = "/";
const char GFile::SlashChar = '\\';
const GString GFile::SlashStr(GFile::SlashChar);
const GString GFile::LegalChars("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789");
const GString GFile::LegalFNChars("abcdefguijklmnopqerstuvxyz1234567890_");
GFile::CurDirs GFile::CurDirForEachDrive;
GAtomicCounter<int> GVfsLocal::TempFileCounter = int(GSystem::CurrentTimeMillis());
GVfsLocal GPipeInputStream::LocalVfs;

/**
 * Main entry point when running on OS2.
 * Here we will do some library initializations before
 * calling the application defined mainStart() function.
 *
 * @author  Leif Erik Larsen
 * @since   2003.09.19
 */
int main ( int argc, const char *args[] )
{
   new GThread("MainThread", 0); // Will assign it self to {@link GThread::MainThread}.
   int ret = 0;
   GProgram::hAB = ::WinInitialize(0);
   GProgram::hMQ = ::WinCreateMsgQueue(GProgram::hAB, 64);
   ::DosSetMaxFH(25); // Maximum 25 concurrent open file handles (streams).
   ::DosError(FERR_DISABLEHARDERR); // Don't show hardware error messages.
   GSystem::PlatformID plid = GSystem::GetPlatformID();
   if (plid <= GSystem::Platform_OS2_2_11)
   {
      GString os2Name = GSystem::GetPlatformName();
      GString msg("Unsupported OS/2 platform: %s", GVArgs(os2Name));
      GLog::PrintLn(msg);
      ::WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, msg, "Error", 0, MB_OK | MB_ICONEXCLAMATION | MB_MOVEABLE);
      ret = -1;
   }
   else
   {
      try {
         bool ok = true;
         GTRY(e, true) {
            DefaultHeapManager heapMngr;
            GGlue glue(heapMngr);
            GCmdLineParameters params(argc, args);
            GProgram::SetCmdLineParameters(&params);
            ret = gmain(params);
         } GCATCH(e) {
            ok = false;
         } GENDCATCH();
         if (!ok)
            gthrow_(GException("System trap in main thread."));
      } catch (GException& e) {
         GString msg("Unhandled exception: %s", GVArgs(e.toString()));
         GString stackTrace = e.getStackTrace(msg);
         GLog::PrintLn(stackTrace);
         ::WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, stackTrace, "Error", 0, MB_OK | MB_ICONEXCLAMATION | MB_MOVEABLE);
         return -1;
      } catch (std::exception& e) {
         GString msg("Unhandled exception: %s", GVArgs(e.what()));
         GLog::PrintLn(msg);
         ::WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, msg, "Error", 0, MB_OK | MB_ICONEXCLAMATION | MB_MOVEABLE);
         ret = -1;
      }
   }
   ::WinDestroyMsgQueue(GProgram::hMQ);
   ::WinTerminate(GProgram::hAB);
   return ret;
}

static GVector<GOS2Exception::Handler*> myOS2ExcpHandlers;

GOS2Exception::Handler::Handler ( BOOL log )
                       :pNext(null),
                        pfnHandler((PFN) exceptionHandler),
                        log(log),
                        handleIdx(myOS2ExcpHandlers.getCount()),
                        active(false)
{
   memset(&jmpThread, 0, sizeof(jmpThread));
   memset(&rep, 0, sizeof(rep));
   memset(&ctx, 0, sizeof(ctx));
   myOS2ExcpHandlers.add(this);
   ::DosSetExceptionHandler((EXCEPTIONREGISTRATIONRECORD*) this);
   active = true;
}

GOS2Exception::Handler::~Handler ()
{
   deactivate();
   if (handleIdx >= 0 && handleIdx < myOS2ExcpHandlers.getCount())
      myOS2ExcpHandlers.remove(handleIdx);
}

void GOS2Exception::Handler::deactivate ()
{
   if (active)
   {
      active = false;
      ::DosUnsetExceptionHandler((EXCEPTIONREGISTRATIONRECORD*) this);
   }
}

GString GOS2Exception::Handler::describePage ( ULONG ulCheck )
{
   GString ret(64);

   ULONG ulCountPages = 1;
   ULONG ulFlagsPage = 0;
   APIRET arc = ::DosQueryMem((void*) ulCheck, &ulCountPages, &ulFlagsPage);

   if (arc == NO_ERROR)
   {
      ret += GString("valid, flags: ");
      if (ulFlagsPage & PAG_READ)
         ret += GString("read ");
      if (ulFlagsPage & PAG_WRITE)
         ret += GString("write ");
      if (ulFlagsPage & PAG_EXECUTE)
         ret += GString("execute ");
      if (ulFlagsPage & PAG_GUARD)
         ret += GString("guard ");
      if (ulFlagsPage & PAG_COMMIT)
         ret += GString("committed ");
      if (ulFlagsPage & PAG_SHARED)
         ret += GString("shared ");
      if (ulFlagsPage & PAG_FREE)
         ret += GString("free ");
      if (ulFlagsPage & PAG_BASE)
         ret += GString("base ");
   }
   else
   if (arc == ERROR_INVALID_ADDRESS)
      ret += GString("invalid");

   ret.trim();
   return ret;
}

GString GOS2Exception::Handler::getDescription ()
{
   ULONG aulBuf[3];
   GString ver = "Unknown";
   ::DosQuerySysInfo(QSV_VERSION_MAJOR, QSV_VERSION_MINOR, &aulBuf, sizeof(aulBuf));
   // Warp 3 is reported as 20.30
   // Warp 4 is reported as 20.40
   // Aurora is reported as 20.45
   if (aulBuf[0] == 20)
   {
      switch (aulBuf[1])
      {
         case 30: ver = "Warp 3"; break;
         case 40: ver = "Warp 4"; break;
         case 45: ver = "WSeB kernel"; break;
      }
   }

   GString typeStr;
   switch (rep.ExceptionNum)
   {
      case XCPT_ACCESS_VIOLATION:
      {
         typeStr = "XCPT_ACCESS_VIOLATION: ";
         if (rep.ExceptionInfo[0] & XCPT_READ_ACCESS)
            typeStr += GString("Invalid read access from 0x%04X:%08X", GVArgs(ctx.ctx_SegDs).add(rep.ExceptionInfo[1]));
         else
         if (rep.ExceptionInfo[0] & XCPT_WRITE_ACCESS)
            typeStr += GString("Invalid write access to 0x%04X:%08X", GVArgs(ctx.ctx_SegDs).add(rep.ExceptionInfo[1]));
         else
         if (rep.ExceptionInfo[0] & XCPT_SPACE_ACCESS)
            typeStr += GString("Invalid space access at 0x%04X", GVArgs(rep.ExceptionInfo[1]));
         else
         if (rep.ExceptionInfo[0] & XCPT_LIMIT_ACCESS)
            typeStr += GString("Invalid limit access occurred");
         else
         if (rep.ExceptionInfo[0] == XCPT_UNKNOWN_ACCESS)
            typeStr += GString("Unknown at 0x%04X:%08X", GVArgs(ctx.ctx_SegDs).add(rep.ExceptionInfo[1]));
         break;
      }

      case XCPT_INTEGER_DIVIDE_BY_ZERO: typeStr = "XCPT_INTEGER_DIVIDE_BY_ZERO"; break;
      case XCPT_ILLEGAL_INSTRUCTION: typeStr = "XCPT_ILLEGAL_INSTRUCTION"; break;
      case XCPT_PRIVILEGED_INSTRUCTION: typeStr = "XCPT_PRIVILEGED_INSTRUCTION"; break;
      case XCPT_INTEGER_OVERFLOW: typeStr = "XCPT_INTEGER_OVERFLOW"; break;
      case XCPT_FLOAT_DENORMAL_OPERAND: typeStr = "XCPT_FLOAT_DENORMAL_OPERAND"; break;
      case XCPT_FLOAT_DIVIDE_BY_ZERO: typeStr = "XCPT_FLOAT_DIVIDE_BY_ZERO"; break;
      case XCPT_FLOAT_INEXACT_RESULT: typeStr = "XCPT_FLOAT_INEXACT_RESULT"; break;
      case XCPT_FLOAT_INVALID_OPERATION: typeStr = "XCPT_FLOAT_INVALID_OPERATION"; break;
      case XCPT_FLOAT_OVERFLOW: typeStr = "XCPT_FLOAT_OVERFLOW"; break;
      case XCPT_FLOAT_STACK_CHECK: typeStr = "XCPT_FLOAT_STACK_CHECK"; break;
      case XCPT_FLOAT_UNDERFLOW: typeStr = "XCPT_FLOAT_UNDERFLOW"; break;
      default: typeStr = "Unknown exception number. Look this up in the OS/2 header files."; break;
   }

   GString ret(1024);

   ret += GString("Exception information as follows:\n");
   ret += GString("\n");
   ret += GString("General information:\n");
   ret += GString("   OS/2 version:   %s.%s (%s)\n", GVArgs(aulBuf[0]).add(aulBuf[1]).add(ver));
   ret += GString("   Exception type: %08X (%s)\n", GVArgs(rep.ExceptionNum).add(typeStr));
   ret += GString("   Address:        %08X\n", GVArgs(ULONG(rep.ExceptionAddress)));
   ret += GString("   Params:         ");
   if (rep.cParameters <= 0)
   {
      ret += GString("None\n");
   }
   else
   {
      const ULONG num = rep.cParameters;
      for (ULONG i=0; i<num; i++)
         ret += GString("%08X%s", GVArgs(rep.ExceptionInfo[i]).add((i==num-1) ? "\n" : ", "));
   }

   PTIB ptib = null;
   PPIB ppib = null;
   ::DosGetInfoBlocks(&ptib, &ppib);
   if (ptib != null && ppib != null)
   {
      HMODULE hMod1 = null;
      CHAR szMod1[CCHMAXPATH] = "unknown";

      if (ctx.ContextFlags & CONTEXT_CONTROL)
      {
         hMod1 = ppib->pib_hmte;
         ::DosQueryModuleName(hMod1, sizeof(szMod1), szMod1);
      }

      ret += GString("Process information:\n");
      ret += GString("   Process ID:     0x%X\n", GVArgs(ppib->pib_ulpid));
      ret += GString("   Process module: 0x%X (%s)\n", GVArgs(hMod1).add(szMod1));

      ret += GString("Trapping thread information:\n");
      if (ptib->tib_ptib2 != null)
      {
         ret += GString("   Thread ID:      0x%X\n", GVArgs(ptib->tib_ptib2->tib2_ultid));
         ret += GString("   Priority:       0x%X\n", GVArgs(ptib->tib_ptib2->tib2_ulpri));
      }
      else
         ret += GString("   Not available.\n");
   }

   ret += GString("Registers:\n");
   if (ctx.ContextFlags & CONTEXT_INTEGER)
   {
      ret += GString("   DS  = %08X (%s)\n", GVArgs(ctx.ctx_SegDs).add(describePage(ctx.ctx_SegDs)));
      ret += GString("   ES  = %08X (%s)\n", GVArgs(ctx.ctx_SegEs).add(describePage(ctx.ctx_SegEs)));
      ret += GString("   FS  = %08X (%s)\n", GVArgs(ctx.ctx_SegFs).add(describePage(ctx.ctx_SegFs)));
      ret += GString("   GS  = %08X (%s)\n", GVArgs(ctx.ctx_SegGs).add(describePage(ctx.ctx_SegGs)));

      ret += GString("   EAX = %08X (%s)\n", GVArgs(ctx.ctx_RegEax).add(describePage(ctx.ctx_RegEax)));
      ret += GString("   EBX = %08X (%s)\n", GVArgs(ctx.ctx_RegEbx).add(describePage(ctx.ctx_RegEbx)));
      ret += GString("   ECX = %08X (%s)\n", GVArgs(ctx.ctx_RegEcx).add(describePage(ctx.ctx_RegEcx)));
      ret += GString("   EDX = %08X (%s)\n", GVArgs(ctx.ctx_RegEdx).add(describePage(ctx.ctx_RegEdx)));
      ret += GString("   ESI = %08X (%s)\n", GVArgs(ctx.ctx_RegEsi).add(describePage(ctx.ctx_RegEsi)));
      ret += GString("   EDI = %08X (%s)\n", GVArgs(ctx.ctx_RegEdi).add(describePage(ctx.ctx_RegEdi)));
   }
   else
      ret += GString("   Not available.\n");

   if (ctx.ContextFlags & CONTEXT_CONTROL)
   {
      ret += GString("Instruction pointer:\n");
      ret += GString("   CS:EIP = %04X:%08X (%s)\n", GVArgs(ctx.ctx_SegCs).add(ctx.ctx_RegEip).add(describePage(ctx.ctx_RegEip)));
      ret += GString("   EFLAGS = %08X\n", GVArgs(ctx.ctx_EFlags));

      ret += GString("Stack:\n");
      ret += GString("   Base:         %08X\n", GVArgs(ULONG(ptib ? ptib->tib_pstack : 0)));
      ret += GString("   Limit:        %08X\n", GVArgs(ULONG(ptib ? ptib->tib_pstacklimit : 0)));
      ret += GString("   SS:ESP = %04X:%08X (%s)\n", GVArgs(ctx.ctx_SegSs).add(ctx.ctx_RegEsp).add(describePage(ctx.ctx_RegEsp)));
      ret += GString("   EBP    =      %08X (%s)\n", GVArgs(ctx.ctx_RegEbp).add(describePage(ctx.ctx_RegEbp)));
   }

   return ret;
}

ULONG _System GOS2Exception::Handler::exceptionHandler ( EXCEPTIONREPORTRECORD* prep,
                                                         Handler* preg2,
                                                         CONTEXTRECORD* pctx,
                                                         void* pv )
{
   if (prep->fHandlerFlags & EH_EXIT_UNWIND)
      return XCPT_CONTINUE_SEARCH;
   if (prep->fHandlerFlags & EH_UNWINDING)
      return XCPT_CONTINUE_SEARCH;
   if (prep->fHandlerFlags & EH_NESTED_CALL)
      return XCPT_CONTINUE_SEARCH;

   switch (prep->ExceptionNum)
   {
      case XCPT_ACCESS_VIOLATION:
      case XCPT_INTEGER_DIVIDE_BY_ZERO:
      case XCPT_ILLEGAL_INSTRUCTION:
      case XCPT_PRIVILEGED_INSTRUCTION:
      case XCPT_INVALID_LOCK_SEQUENCE:
      case XCPT_INTEGER_OVERFLOW:
      case XCPT_FLOAT_DENORMAL_OPERAND:
      case XCPT_FLOAT_DIVIDE_BY_ZERO:
      case XCPT_FLOAT_INEXACT_RESULT:
      case XCPT_FLOAT_INVALID_OPERATION:
      case XCPT_FLOAT_OVERFLOW:
      case XCPT_FLOAT_STACK_CHECK:
      case XCPT_FLOAT_UNDERFLOW:
      {
         // Make sure that preg2 is actually an instance of Handler
         int handleIdx = -1;
         for (int i=0, num=myOS2ExcpHandlers.getCount(); i<num && handleIdx<0; i++)
            if (preg2 == myOS2ExcpHandlers[i])
               handleIdx = i;

         // If the preg2 object is known to us then we can assume that we
         // are not called due to GTRY. Only in that case we can safely
         // attempt to handle the exception, because preg2->jmpThread is
         // set up with proper information.
         if (handleIdx >= 0)
         {
            preg2->rep = *prep;
            preg2->ctx = *pctx;

            // Jump back to failing code (that is; jump to GCATCH-block).
            longjmp(preg2->jmpThread, prep->ExceptionNum);
         }
         return XCPT_CONTINUE_SEARCH; // Will actually never be executed, due to above longjmp().
      }

      default:
         return XCPT_CONTINUE_SEARCH;
   }
}
