/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TEXTRESOURCE
#define __GLIB_TEXTRESOURCE

#include "glib/resource/GAbstractResource.h"

/**
 * Each TEXT resource is represented with an instance of this class.
 */
class GTextResource : public GAbstractResource
{
   public:

      /** Text ID, which is possibly a link to another TEXT. */
      GString textID;

      /** Hint ID, which is possibly a link to another TEXT. */
      GString hintID;

      /** Alternative text #1. Is possibly a link to another TEXT. */
      GString alt1;

      /** Alternative text #2. Is possibly a link to another TEXT. */
      GString alt2;

      /** Alternative text #3. Is possibly a link to another TEXT. */
      GString alt3;

      /** Alternative text #4. Is possibly a link to another TEXT. */
      GString alt4;

      /** Alternative text #5. Is possibly a link to another TEXT. */
      GString alt5;

      /** Alternative text #6. Is possibly a link to another TEXT. */
      GString alt6;

      /** Alternative text #7. Is possibly a link to another TEXT. */
      GString alt7;

      /** Alternative text #8. Is possibly a link to another TEXT. */
      GString alt8;

      /** Alternative text #9. Is possibly a link to another TEXT. */
      GString alt9;

   public:

      explicit GTextResource ( const GString& id );

   public:

      /**
       * Convert all occurences of non-text ascii characters in the specified
       * string to the text resource compatible escape format characters as 
       * in hard coded C++/Java/Resource text strings.
       *
       * Also convert all occurences of double quotes to the text
       * resource compatible format "\"" as in hard coded text strings.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @param   str   The string of binary characters to translate.
       * @param   quote The quote character to use to enclose on the 
       *                returned string, or a null-character if we should 
       *                not quote the returned string at all.
       * @return  The string object after it has been converted.
       * @see     #ParseTextLiteral
       */
      static GString MakeTextLiteral ( const GString& str, char quote = '\0' );

      /**
       * Load and compile the following resource definition from the
       * specified tokenizer for the calling resource compiler.
       *
       * Since we assume that the ID-part of the resource has already been
       * parsed the next token (which is the first, from our point of view)
       * returned from the tokenizer must be "(".
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.08
       * @param   tokenizer   The tokenizer of where to get tokens to parse.
       * @throws  GRcException in case of any error.
       */
      virtual void loadFromScript ( class GRcTokenizer& tokenizer, class GResourceTable* table );

      /**
       * Parse and convert the specified quoted text literal to a binary 
       * format text string. The specified string must be of a format that 
       * is compatible with standard Java/C++/Text Resource text literals,
       * and it must be enclosed in a pair of either single or double 
       * quote characters.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @param   str       The string of which to parse and translate.
       * @param   isQuoted  True if the specified string is already quoted,
       *                    or else false. We will achieve somewhat better 
       *                    performance if the string is already quoted
       *                    and this parameter is true. If this parameter 
       *                    is true and the string is actually not quoted
       *                    (or vice verca) is an error that result in 
       *                    undefined behaviour.
       * @see     #MakeTextLiteral
       */
      static GString ParseTextLiteral ( const GString& str, bool isQuoted );
};

#endif
