/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GRcTokenizer.h"
#include "glib/resource/GRcException.h"
#include "glib/exceptions/GSyntaxErrorException.h"
#include "glib/vfs/GFile.h"
#include "glib/io/GStringStream.h"
#include "glib/io/GOpenFileException.h"
#include "glib/GProgram.h"

const GString GRcTokenizer::Token_comma(",");
const GString GRcTokenizer::Token_dir("#");
const GString GRcTokenizer::Token_eolcom("//");
const GString GRcTokenizer::Token_eq("=");
const GString GRcTokenizer::Token_lcom("/*");
const GString GRcTokenizer::Token_lpar("(");
const GString GRcTokenizer::Token_rcom("*/");
const GString GRcTokenizer::Token_rpar(")");
const GString GRcTokenizer::Token_lbracket("{");
const GString GRcTokenizer::Token_rbracket("}");

const GString GRcTokenizer::Token_debug("debug");
const GString GRcTokenizer::Token_define("define");
const GString GRcTokenizer::Token_endif("endif");
const GString GRcTokenizer::Token_else("else");
const GString GRcTokenizer::Token_error("error");
const GString GRcTokenizer::Token_if("if");
const GString GRcTokenizer::Token_ifdef("ifdef");
const GString GRcTokenizer::Token_ifndef("ifndef");
const GString GRcTokenizer::Token_include("include");
const GString GRcTokenizer::Token_log("log");
const GString GRcTokenizer::Token_undef("undef");
const GString GRcTokenizer::Token_warning("warning");

const GString GRcTokenizer::Token_alt("alt");
const GString GRcTokenizer::Token_control("control");
const GString GRcTokenizer::Token_num("num");
const GString GRcTokenizer::Token_up("up");
const GString GRcTokenizer::Token_text("text");
const GString GRcTokenizer::Token_hint("hint");
const GString GRcTokenizer::Token_id("id");
const GString GRcTokenizer::Token_key("key");
const GString GRcTokenizer::Token_shift("shift");
const GString GRcTokenizer::Token_userdata("userdata");
const GString GRcTokenizer::Token_xpos("xpos");
const GString GRcTokenizer::Token_ypos("ypos");
const GString GRcTokenizer::Token_width("width");
const GString GRcTokenizer::Token_height("height");
const GString GRcTokenizer::Token_constraints("constraints");
const GString GRcTokenizer::Token_oily("oily");
const GString GRcTokenizer::Token_disabled("disabled");
const GString GRcTokenizer::Token_toggleon("toggleon");
const GString GRcTokenizer::Token_hidden("hidden");
const GString GRcTokenizer::Token_acceltable("acceltable");
const GString GRcTokenizer::Token_menu("menu");
const GString GRcTokenizer::Token_toolbar("toolbar");
const GString GRcTokenizer::Token_dialog("dialog");
const GString GRcTokenizer::Token_icon("icon");
const GString GRcTokenizer::Token_layout("layout");
const GString GRcTokenizer::Token_closebutton("closebutton");
const GString GRcTokenizer::Token_resizable("resizable");
const GString GRcTokenizer::Token_colorbits("colorbits");
const GString GRcTokenizer::Token_palette("palette");
const GString GRcTokenizer::Token_pixels("pixels");
const GString GRcTokenizer::Token_userdata1("userdata1");
const GString GRcTokenizer::Token_invisible("invisible");
const GString GRcTokenizer::Token_item("item");
const GString GRcTokenizer::Token_popup("popup");
const GString GRcTokenizer::Token_separator("separator");
const GString GRcTokenizer::Token_alt1("alt1");
const GString GRcTokenizer::Token_alt2("alt2");
const GString GRcTokenizer::Token_alt3("alt3");
const GString GRcTokenizer::Token_alt4("alt4");
const GString GRcTokenizer::Token_alt5("alt5");
const GString GRcTokenizer::Token_alt6("alt6");
const GString GRcTokenizer::Token_alt7("alt7");
const GString GRcTokenizer::Token_alt8("alt8");
const GString GRcTokenizer::Token_alt9("alt9");

GRcTokenizer::SingleIf::SingleIf ( bool isTrue ) 
                       :isTrue(isTrue), 
                        hasReachedElse(false) 
{
}

GRcTokenizer::SingleIf::~SingleIf () 
{
}

GRcTokenizer::GRcTokenizer ( GInputStream& stream )
             :expParser(null),
              incl(4),
              curIncl(null),
              ifCB(8),
              defs(16),
              curDefPos(0)
{
   curIncl = new GRcIncludeModule(&stream, -1, false);
   srcPath = curIncl->path;
   incl.push(curIncl);
}

GRcTokenizer::GRcTokenizer ( const GString& str, 
                             bool isFileName, 
                             const GSearchPath& searchPath, 
                             const GKeyBag<GString>* ppdefines )
             :expParser(null),
              searchPath(searchPath),
              incl(4),
              curIncl(null),
              ifCB(8),
              defs(16),
              curDefPos(0)
{
   // Define the predefined macros to the precompiler.
   // These are the macros that can be used by the resource script to support 
   // some level of selective resource tables that are automatically adjusted 
   // with respect to the environment in which it is parsed.
   defineMacro("_MODULENAME", "\"" + srcPath + "\"");
   defineMacro("__OS2__", "1");

   // Add all the application specified preprocessor macros.
   if (ppdefines != null)
   {
      for (int i=0, num=ppdefines->getCount(); i<num; i++)
      {
         const GString& name = ppdefines->getKey(i);
         const GString& value = ppdefines->getIndexedItem(i);
         defineMacro(name, value);
      }
   }

   // TODO: Should add much more predefined macros.
   // E.g. OS-identification, version numbers of both the OS,
   // the program and the resource parser. Name and path of the 
   // executable file. Home directory of the user. Etc.
   /*
   const GString& ver = ownerProg.getVersionString();
   defineMacro("__PRG_VERSTR__", "\"" + ver + "\"");
   defineMacro("__VERSTR__", "\"1.0\"");
   */

   // ---
   if (isFileName)
   {
      try {
         curIncl = new GRcIncludeModule(vfs, str, -1);
         srcPath = curIncl->path;
         incl.push(curIncl);
      } catch (GOpenFileException& e) {
         // Can't open or find resource script file: \"%s\"\nMessage from the system: %s
         GString emsg = e.toString();
         gthrow_(GRcException(*this, GRcException::ERRSRCFNFOUND, GVArgs(str).add(emsg)));
      }
   }
   else
   {
      GStringStream* stream = new GStringStream(str);
      curIncl = new GRcIncludeModule(stream, -1, true);
      srcPath = curIncl->path;
      incl.push(curIncl);
   }
}

GRcTokenizer::~GRcTokenizer ()
{
   delete expParser;
}

void GRcTokenizer::defineMacro ( const char* symName, const GString& defValue )
{
   if (defs.containsKey(symName))
      gthrow_(GRcException(*this, GRcException::ERRSYMALREADYDEFINED, GVArgs(symName))); // Symbol already defined: %s
   GString* val = new GString(defValue);
   defs.put(symName, val);
}

bool GRcTokenizer::readNextLine ()
{
   GRcIncludeModule& inc = *curIncl;

   inc.srcLinePos = 0; // Reset current position to start.

   for (;;)
   {
      inc.lineNr++;
      if (inc.stream->readString(inc.srcLine) == "")
         return false;
      if (inc.srcLine != "")
         break;
   }

   return true;
}

bool GRcTokenizer::parseIf ()
{
   aptr<GExpressionParser::RValue> val = parseExpression();
   return val->isTrue();
}

int GRcTokenizer::parseInteger ()
{
   aptr<GExpressionParser::RValue> val = parseExpression();
   try {
      return val->getInt();
   } catch (GSyntaxErrorException& e) {
      gthrow_(GRcException(*this, e.getMessage(), GRcException::ERRSYNTAXERROR)); // Syntax error!
   }
}

double GRcTokenizer::parseFloat ()
{
   aptr<GExpressionParser::RValue> val = parseExpression();
   try {
      return val->getDouble();
   } catch (GSyntaxErrorException& e) {
      gthrow_(GRcException(*this, e.getMessage(), GRcException::ERRSYNTAXERROR)); // Syntax error!
   }
}

char GRcTokenizer::getNextChar ()
{
   if (curDefPos >= 0)
   {
      char next = curDef[curDefPos++];
      if (next == '\0')
         curDefPos = -1;
      else
         return next;
   }

   // Don't increment srcLinePos if we have reached the end-of-line!
   GRcIncludeModule& im = *curIncl;
   if (im.srcLinePos < im.srcLine.length())
      return im.srcLine[im.srcLinePos++];
   return '\0';
}

void GRcTokenizer::undoLastCharacter ()
{
   if (curDefPos > 0)
      curDefPos--;
   else
      curIncl->srcLinePos--;
}

void GRcTokenizer::readNextToken_ ()
{
   currentToken.clear();

   // Loop through all white characters (tabs, spaces) in start of line.
   char nextChar;
   while (isspace(nextChar = getNextChar()) || nextChar == 0x1A);
   if (nextChar == '\0')
      return;

   if (nextChar == '\"' || nextChar == '\'')
   {
      char cBracket = nextChar;

      currentToken.quoted = true;
      currentToken.quoteChar = cBracket;

      // We are parsing text within brackets, so look for the terminating
      // bracket character (to stop copying text as a word).

      while ((nextChar = getNextChar()) != '\0')
      {
         if (nextChar == '\\')        /* If forcing character */
         {
            nextChar = getNextChar();
            switch (nextChar)
            {
               case 't':
                    nextChar = '\t';
                    break;

               case 'n':
                    nextChar = '\n';
                    break;

               case 'r':
                    nextChar = '\r';
                    break;

               case '\n':
               case '\0':
                    gthrow_(GRcException(*this, GRcException::ERRUNTERMSTRING)); // Unterminated string!

               default:
                    break;
            }
         }
         else
         {
            if (nextChar == cBracket)
               break;
         }

         if (nextChar == '\n')
         {
            const int len = currentToken.getLength();
            if ((len == 0 || currentToken.getCharacter(len-1) != '\r'))
               currentToken += '\r'; // Linefeeds are to be prefixed with a CR.
         }

         currentToken += nextChar; // Copy next character.
      }
   }

   else for (;;) // Next word is not within quotes.
   {
      if (nextChar == '/')
      {
         // Possibly the start of another comment, but it can also be
         // a division character, which is to be treated as a special
         // single character token.
         if (currentToken.getLength() > 0)
            undoLastCharacter();
         else
         {
            currentToken += nextChar;
            nextChar = getNextChar();
            if ((nextChar == '/') || (nextChar == '*'))
               currentToken += nextChar;
            else
            if (nextChar != '\0')
               undoLastCharacter();
         }
         break;
      }

      else
      if (nextChar == '*')
      {
         // Possibly the end of a (current) comment, but it can also
         // be a multiplication character, which is to be treated as a
         // special single character token.
         if (currentToken.getLength() > 0)
            undoLastCharacter();
         else
         {
            currentToken += nextChar;
            nextChar = getNextChar();
            if (nextChar == '/')
               currentToken += nextChar;
            else
            if (nextChar != '\0')
               undoLastCharacter();
         }
         break;
      }

      else
      if (nextChar == ',' ||          /* , */
          nextChar == '(' ||          /* ( */
          nextChar == ')' ||          /* ) */
          nextChar == '+' ||          /* + */
          nextChar == '-' ||          /* - */
          nextChar == '#')            /* # */
      {
         if (currentToken.getLength() > 0)
            undoLastCharacter();
         else
            currentToken += nextChar;
         break;
      }

      else
      if (nextChar == '=' ||          /* = or == */
          nextChar == '!' ||          /* ! or != */
          nextChar == '<' ||          /* < or <= */
          nextChar == '>')            /* > or >= */
      {
         if (currentToken.getLength() > 0)
            undoLastCharacter();
         else
         {
            currentToken += nextChar;
            nextChar = getNextChar();
            if (nextChar == '=')
               currentToken += nextChar;
            else
            if (nextChar != '\0')
               undoLastCharacter();
         }
         break;
      }

      else
      if (nextChar == '&')            /* & or && */
      {
         if (currentToken.getLength() > 0)
            undoLastCharacter();
         else
         {
            currentToken += nextChar;
            nextChar = getNextChar();
            if (nextChar == '&')
               currentToken += nextChar;
            else
            if (nextChar != '\0')
               undoLastCharacter();
         }
         break;
      }

      else
      if (nextChar == '|')            /* | or || */
      {
         if (currentToken.getLength() > 0)
            undoLastCharacter();
         else
         {
            currentToken += nextChar;
            nextChar = getNextChar();
            if (nextChar == '|')
               currentToken += nextChar;
            else
            if (nextChar != '\0')
               undoLastCharacter();
         }
         break;
      }

      currentToken += nextChar;

      nextChar = getNextChar();
      if (nextChar == '\0')
         break;

      if (isspace(nextChar) || nextChar == 0x1A)
         break; // Spaces separates each unquoted token.
   }
}

void GRcTokenizer::readNextToken ( bool doPreCompile, bool acceptEOS )
{
   int iCommentLevel = 0; /* Current level of nested comment */
   GRcIncludeModule* pIncl = curIncl;

   currentToken.clear();

   for (;;)
   {
      // Read next line if we have reached the end of current line.
      if (pIncl->srcLinePos >= pIncl->srcLine.length())
      {
         if (!readNextLine())
         {
            // We have reached EOF.
            if (iCommentLevel)
               gthrow_(GRcException(*this, GRcException::ERRUNCLOSEDCOMT)); // Unclosed comment at EOF!
            if (ifCB.getCount()-1 > incl.top()->startIfLevel)
               gthrow_(GRcException(*this, GRcException::ERRUNCLOSEDIFELSE)); // Unclosed #IF or #ELSE at EOF!
            incl.pop(true);
            curIncl = incl.top();
            pIncl = curIncl;
            if (pIncl == null)
            {
               if (acceptEOS)
                  return;
               gthrow_(GRcException(*this, GRcException::ERRUNEXPECTEDEOS)); // Unexpected End-Of-Stream!
            }
         }
      }

      readNextToken_();

      if (iCommentLevel)
      {
         // We are currently within a comment block, so ignore everything
         // until we find the end of the comment.

         if (currentToken == Token_rcom) // "*/"
            iCommentLevel--;
         else
         if (currentToken == Token_lcom) // "/*"
            iCommentLevel++; // Assume that nested comments are alloweed.
      }
      else
      if (currentToken == Token_eolcom) // "//"
      {
         // Force next line to be read simply by skipping rest of current line.
         currentToken.clear();
         pIncl->srcLine = "";
         pIncl->srcLinePos = 0;
      }
      else
      if (currentToken == Token_lcom) // "/*"
         iCommentLevel++;
      else
      if (!currentToken.isEmpty())
      {
         if (doPreCompile && !currentToken.isQuoted())
         {
            const GString* strVal = defs.get(currentToken.tokenStr);
            if (strVal != null)
            {
               if (curDefPos >= 0)
                  ; // TODO: What about nested macros?
               else
                  curDef = *strVal;
               curDefPos = 0;
               continue;
            }
         }

         break; // We have fetched out another word to parse.
      }
   }
}

const GRcToken* GRcTokenizer::nextPreCompiledToken ( bool acceptEOS )
{
   bool bNextIsDirective = false;
   for (;;)
   {
      readNextToken(true, acceptEOS);
      if (currentToken.isEmpty())
         break;

      GRcTokenizer::SingleIf* ifl = ifCB.top();
      if (ifl != null && !ifl->isTrue)
      {
         // We are currently inside a block of code that shouldn't be parsed
         // because of an recent #if-test that didn't match. So skip all
         // tokens except a few ones that might be of interest in order
         // to terminate the dirty #if-block.

         if (currentToken.isQuoted())
            continue;
         else
         if (currentToken == Token_dir) // "#"
            bNextIsDirective = true;
         else
         if (!bNextIsDirective)
            continue;
         else
         {
            bNextIsDirective = false;

            if (currentToken == Token_ifdef ||
                currentToken == Token_ifndef ||
                currentToken == Token_if)
            {
               GRcTokenizer::SingleIf* ifl = new GRcTokenizer::SingleIf(false);
               ifCB.push(ifl);
               continue;
            }
            else
            if (currentToken == Token_else)
            {
               bool isTrue;
               GRcTokenizer::SingleIf* ifl = ifCB.top();
               ifl->hasReachedElse = true;
               if (ifCB.getCount() > 1 && !ifCB[ifCB.getCount()-2].isTrue)
                  isTrue = false;
               else
                  isTrue = (bool) !ifl->isTrue;
               ifl->isTrue = isTrue;
            }
            else
            if (currentToken == Token_endif)
            {
               GRcTokenizer::SingleIf* ifl = ifCB.top();
               if (ifl == null)
                  gthrow_(GRcException(*this, GRcException::ERRENDIFWITHOUTIF)); // #ENDIF without #IF
               ifCB.pop(true);
               continue;
            }
            else
               continue;
         }
      }

      else
      {
         // We are currently inside a block of code that should be parsed
         // normally. That is, the code is not within a comment or
         // dirty #if-test. So just go ahead!

         if (currentToken == Token_dir && // "#"
             !currentToken.isQuoted())
         {
            bNextIsDirective = true;
            continue;
         }

         if (bNextIsDirective)
         {
            bNextIsDirective = false;

            if (currentToken.isQuoted())
               gthrow_(GRcException(*this, GRcException::ERRSYNTAXERROR)); // Syntax error!

            if (currentToken == Token_include)
            {
               readNextToken(true, false);
               if (!currentToken.isQuoted())
                  gthrow_(GRcException(*this, GRcException::ERRINCMUSTBEQUOTED)); // #INCLUDE... module must be inside a pair of quotes!

               const int iflevel = ifCB.getCount() - 1;
               const GString inclModuleNameAsGiven = currentToken.toString();

               // Always look first in the same directory as of the parent 
               // module, by default.
               bool found = false;
               GString path = srcPath;
               GFile::CutFileName(path);
               path += inclModuleNameAsGiven;
               if (vfs.existFile(path)) 
               {
                  try {
                     curIncl = new GRcIncludeModule(vfs, path, iflevel);
                     incl.push(curIncl);
                     found = true;
                  } catch (GOpenFileException& /*e*/) {
                     // Normal behaviour if the include module was no found.
                     // Continue looking for the module in the search-path.
                  }
               }

               // If not found in the default location, look for the module 
               // in the search-path.
               for (int i=0, count=searchPath.getCount(); !found && i<count; i++)
               {
                  path = searchPath.getIndexedDir(i);
                  vfs.slash(path);
                  path += inclModuleNameAsGiven;
                  if (vfs.existFile(path)) 
                  {
                     try {
                        curIncl = new GRcIncludeModule(vfs, path, iflevel);
                        incl.push(curIncl);
                        found = true;
                     } catch (GOpenFileException& /*e*/) {
                        // Normal behaviour if the include module was no found.
                        // Continue looking for the module in the search-path.
                     }
                  }
               }

               // Throw an exception if we could not find the include-module.
               if (!found)
                  gthrow_(GRcException(*this, GRcException::ERRINCFNFOUND, GVArgs(inclModuleNameAsGiven))); // Can't find #include module: %s
            }
            else
            if (currentToken == Token_define)
            {
               readNextToken(false, false);
               GString symbolName = currentToken.toString();

               readNextToken(true, true);
               if (currentToken.isQuoted())
               {
                  GString val(currentToken.getLength() + 2);
                  val += '"';
                  val += currentToken.tokenStr;
                  val += '"';
                  defineMacro(symbolName, val);
               }
               else
               {
                  if (currentToken == Token_lpar) // "("
                  {
                     // Transform e.g. "#define X (2*2+1)" to "#define X 5".
                     aptr<GExpressionParser::RValue> val = parseExpression();
                     GString str = val->getString();
                     defineMacro(symbolName, str);
                  }
                  else
                  {
                     defineMacro(symbolName, currentToken.tokenStr);
                  }
               }
            }
            else
            if (currentToken == Token_ifdef)
            {
               readNextToken(false, false);
               bool isTrue = defs.containsKey(currentToken.tokenStr);
               GRcTokenizer::SingleIf* ifl = new GRcTokenizer::SingleIf(isTrue);
               ifCB.push(ifl);
            }
            else
            if (currentToken == Token_ifndef)
            {
               readNextToken(false, false);
               bool isTrue = !defs.containsKey(currentToken.tokenStr);
               GRcTokenizer::SingleIf* ifl = new GRcTokenizer::SingleIf(isTrue);
               ifCB.push(ifl);
            }
            else
            if (currentToken == Token_if)
            {
               readNextToken(true, false);
               if (currentToken != Token_lpar)
                  gthrow_(GRcException(*this, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(currentToken.tokenStr))); // Expected '(' but found '%2' in statement!

               GRcTokenizer::SingleIf* ifl = new GRcTokenizer::SingleIf(true);
               ifCB.push(ifl);

               // Parse the expression.
               ifl->isTrue = parseIf();
            }
            else
            if (currentToken == Token_else)
            {
               GRcTokenizer::SingleIf* ifl = ifCB.top();
               if (ifl == null)
                  gthrow_(GRcException(*this, GRcException::ERRELSEWITHOUTIF)); // #ELSE without #IF

               ifl->hasReachedElse = true;
               ifl->isTrue = (bool) !ifl->isTrue;
            }
            else
            if (currentToken == Token_endif)
            {
               if (ifCB.isEmpty())
                  gthrow_(GRcException(*this, GRcException::ERRENDIFWITHOUTIF)); // #ENDIF without #IF

               ifCB.pop(true);
            }
            else
            if (currentToken == Token_undef)
            {
               readNextToken(false, false);
               defs.remove(currentToken.tokenStr);
            }
            else
            if (currentToken == Token_error)
            {
               readNextToken(true, false);
               gthrow_(GRcException(*this, GRcException::ERRERRORDIRECTIVE, GVArgs(currentToken.tokenStr))); // %s
            }
            else
            if (currentToken == Token_warning)
            {
               readNextToken(true, false);
               GLog::Log(this, "##warning: %s\n", GVArgs(currentToken.tokenStr));
            }
            else
            if (currentToken == Token_log)
            {
               readNextToken(true, false);
               GLog::Log(this, GString("##log: \"%s\"", GVArgs(currentToken.tokenStr)));
            }
            else
            if (currentToken == Token_debug)
            {
               // This directive might be useful when debugging this resource
               // compiler. Just put a #DEBUG statement somewhere in the resource
               // script and put a breakpoint at the below C-statement.
               // Then you are on the road to debug the next resource script
               // statements...
               GInteger::ToString(0); // A dummy statement, just to have a place to set the breakpoint
            }
            else
               gthrow_(GRcException(*this, GRcException::ERRUNKNOWDIREC, GVArgs(currentToken.tokenStr))); // Unknown preprocessor directive: %s
         }
         else
         if (!currentToken.isEmpty())
         {
            break; // We have fetched out another word to parse.
         }
      }
   }

   return &currentToken;
}

const GRcToken* GRcTokenizer::queryArgValue ()
{
   nextPreCompiledToken(false);
   if (currentToken != Token_eq)
      gthrow_(GRcException(*this, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("=").add(currentToken.tokenStr))); // Expected '=' but found '%2' in statement!
   return nextPreCompiledToken(false);
}

int GRcTokenizer::queryArgValueInt ()
{
   queryArgValue();
   if (currentToken == Token_lpar) // "("
   {
      return parseInteger();
   }
   else
   {
      try {
         return GInteger::ParseInt(currentToken.tokenStr);
      } catch (GNumberFormatException& /*e*/) {
         gthrow_(GRcException(*this, GRcException::ERREXPECTEDINTEGER, GVArgs(currentToken.tokenStr))); // Expected an INTEGER, but found %s.
      }
   }
}

double GRcTokenizer::queryArgValueFloat ()
{
   queryArgValue();
   if (currentToken == Token_lpar) // "("
   {
      return parseFloat();
   }
   else
   {
      try {
         return GDouble::ParseDouble(currentToken.tokenStr);
      } catch (GNumberFormatException& /*e*/) {
         gthrow_(GRcException(*this, GRcException::ERREXPECTEDFLOAT, GVArgs(currentToken.tokenStr))); // Expected a FLOAT, but found %s.
      }
   }
}

aptr<GExpressionParser::RValue> GRcTokenizer::parseExpression ()
{
   if (expParser == null)
      expParser = new GExpressionParser(*this);

   try {
      return expParser->getParsedExpression();
   } catch (GSyntaxErrorException& e) {
      gthrow_(GRcException(*this, e.getMessage(), GRcException::ERRSYNTAXERROR)); // Syntax error!
   }
}

const GAbstractToken* GRcTokenizer::getNextAbstractToken ()
{
   try {
      return nextPreCompiledToken(false);
   } catch (GRcException& e) {
      // There was some error reading the tokenizer input stream.
      // The best way to report this to the caller from here is to let the
      // method <i>isEmpty()</i> of the returned token return true.
      GLog::Log(this, e.getStackTrace(e.toString()));
      currentToken.clear();
      return &currentToken;
   }
}
