/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GIconResource.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"
#include "glib/gui/GBitmap.h"
#include "glib/primitives/GRectangle.h"

GIconResource::GIconResource ()
              :GAbstractResource(),
               width(0),
               height(0),
               colorBits(0),
               palette(0),
               pixels(0)
{
}

GIconResource::GIconResource ( const GIconResource& src )
              :GAbstractResource(src),
               width(src.width),
               height(src.height),
               colorBits(src.colorBits),
               palette(src.palette),
               pixels(src.pixels)
{
}

GIconResource::GIconResource ( const GString& id )
              :GAbstractResource(id),
               width(0),
               height(0),
               colorBits(0),
               palette(16),
               pixels(16*16)
{
}

GIconResource::~GIconResource ()
{
}

GIconResource& GIconResource::operator= ( const GIconResource& src )
{
   if (this == &src)
      return *this;
   GAbstractResource::operator=(src);
   this->width = src.width;
   this->height = src.height;
   this->colorBits = src.colorBits;
   this->palette = src.palette;
   this->pixels = src.pixels;
   return *this;
}

bool GIconResource::operator== ( const GIconResource& src ) const
{
   if (this == &src)
      return true;
   else
      return GAbstractResource::operator==(src) &&
             width == src.width &&
             height == src.height &&
             colorBits == src.colorBits &&
             palette == src.palette &&
             pixels == src.pixels;
}

bool GIconResource::operator!= ( const GIconResource& src ) const
{
   return !operator==(src);
}

int GIconResource::MakeColorFromString ( const GString& str )
{
   if (str.length() != 6)
      return -1;

   try {
      int r = GInteger::ParseInt(str.substring(0, 2), 16);
      int g = GInteger::ParseInt(str.substring(2, 4), 16);
      int b = GInteger::ParseInt(str.substring(4, 6), 16);
      return (r << 16) | (g << 8) | b;
   } catch (GNumberFormatException& /*e*/) {
      return -1;
   }
}

void GIconResource::loadFromScript ( GRcTokenizer& tokenizer, GResourceTable* /*table*/ )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lbracket)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("{").add(token->getString()))); // Expected '(' but found '%2' in statement!

   width = 0;
   height = 0;
   colorBits = 0;
   palette.removeAll();
   pixels.removeAll();
   bool hasParsedColorBits = false;

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);
      if (*token == GRcTokenizer::Token_comma)
         continue;
      else
      if (*token == GRcTokenizer::Token_rbracket)
      {
         int pixcount = pixels.getCount();
         if (pixcount <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRNOPIXELS, GVArgs(getIDString()))); // Icon '%s' contains no PIXELS!
         if (pixcount != width * height)
            gthrow_(GRcException(tokenizer, GRcException::ERRNROFPIXELS, GVArgs(getIDString()))); // Illegal number of PIXELS defined in icon '%s'!
         break;
      }
      else
      if (*token == GRcTokenizer::Token_width)
      {
         if (width != 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRWIDTHARDEF)); // WIDTH already defined!
         width = tokenizer.queryArgValueInt();
         if (width <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRWIDTHLTHAN0)); // WIDTH must be larger than zero!
      }
      else
      if (*token == GRcTokenizer::Token_height)
      {
         if (height)
            gthrow_(GRcException(tokenizer, GRcException::ERRHEIGHTARDEF)); // HEIGHT already defined!
         height = tokenizer.queryArgValueInt();
         if (height <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRHEIGHTLTHAN0)); // HEIGHT must be larger than zero!
      }
      else
      if (*token == GRcTokenizer::Token_colorbits)
      {
         if (hasParsedColorBits)
            gthrow_(GRcException(tokenizer, GRcException::ERRCOLBITSARDEF)); // COLORBITS already defined!
         colorBits = tokenizer.queryArgValueInt();
         hasParsedColorBits = true;
         if (colorBits != 0 && colorBits != 4 && colorBits != 8)
            gthrow_(GRcException(tokenizer, GRcException::ERRILLEGCOLBITS)); // Illegal number of COLORBITS!
      }
      else
      if (*token == GRcTokenizer::Token_palette)
      {
         if (!hasParsedColorBits)
            gthrow_(GRcException(tokenizer, GRcException::ERRPALBEFORECOLBITS)); // Can't define PALETTE before COLORBITS.
         if (colorBits == 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRPALNOTNEEDEDRGB)); // PALETTE not needed for RGB color mode icons.
         if (pixels.getCount() != 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRPALBEFOREPIXELS)); // PALETTE must come before PIXELS.
         if (palette.getCount() != 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRPALETTEARDEF)); // PALETTE already defined.

         token = tokenizer.nextPreCompiledToken(false);
         if (*token != GRcTokenizer::Token_lpar)
            gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

         int numColors = (colorBits == 4) ? 16 : 256;
         palette.ensureCapacity(numColors);
         int pos = 0;
         for (;;)
         {
            token = tokenizer.nextPreCompiledToken(false);
            if (*token == GRcTokenizer::Token_rpar)
               break;
            if (pos >= numColors)
               gthrow_(GRcException(tokenizer, GRcException::ERRINVPALETTESIZE)); // Invalid PALETTE size.
            try {
               const GString& str = token->getString();
               int rgb = GInteger::ParseInt(str, 16);
               palette.add(rgb);
               pos++;
            } catch (GNumberFormatException& /*e*/) {
               gthrow_(GRcException(tokenizer, GRcException::ERRNOTHEXINT, GVArgs(token->getString()))); // Not a hexadecimal integer: "%s".
            }
         }
         if (pos < numColors)
               gthrow_(GRcException(tokenizer, GRcException::ERRINVPALETTESIZE)); // Invalid PALETTE size.
      }
      else
      if (*token == GRcTokenizer::Token_pixels)
      {
         if (height <= 0 || width <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRPIXELSWOWH)); // Can't define PIXELS before WIDTH and HEIGHT.

         if (pixels.getCount() > 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRPIXELSARDEF)); // PIXELS already defined!

         token = tokenizer.nextPreCompiledToken(false);
         if (*token != GRcTokenizer::Token_lpar) // Start-of-pixels-block.
            gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

         // Add one to iPixelsSize so that there will be room for one extra
         // pixel in case the PIXELS definition is to long. In that case we
         // will return an error message.

         pixels.ensureCapacity(width * height);

         if (palette.getCount() == 0)
         {
            if (colorBits == 0)
               ; // Need no palette for "true" RGB color mode.
            else
            if (colorBits == 4)
               palette = GColor::GetStd16ColorPalette();
            else
            if (colorBits == 8)
               palette = GColor::GetStd256ColorPalette();
         }

         for (;;)
         {
            token = tokenizer.nextPreCompiledToken(false);
            if (*token == GRcTokenizer::Token_rpar)
               break;

            if (colorBits == 0) // Each pixel is given in hex RGB.
            {
               const GString& str = token->getString();
               int pix = GIconResource::MakeColorFromString(str);
               pixels.add(pix); // If pix == -1 then it is transparent.
            }
            else
            if (colorBits == 4) // Pixels are given as string of hex characters.
            {
               const int num = token->getLength();
               for (int i=0; i<num; i++)
               {
                  int pix = -1; // Transparent until opposite proven.
                  char chr = token->getCharacter(i);
                  if (chr >= '0' && chr <= '9')
                  {
                     pix = chr - '0';
                  }
                  else
                  {
                     chr = GCharacter::ToUpperCase(chr);
                     if (chr >= 'A' && chr <= 'F')
                        pix = 10 + (chr - 'A');
                  }
                  if (pix < 0 || pix > palette.getCount())
                     pix = -1;
                  pixels.add(pix);
               }
            }
            else
            if (colorBits == 8) // Each pixel is given as hex index.
            {
               const GString& str = token->getString();
               int pix = -1; // Transparent until opposite proven.
               if (str != "XX") try {
                  pix = GInteger::ParseInt(str, 16);
                  if (pix < 0 || pix > palette.getCount())
                     pix = -1;
               } catch (GNumberFormatException& /*e*/) {
               }
               pixels.add(pix);
            }
         }
      }
      else
         gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(token->getString()))); // Unknown token: %s
   }
}

int GIconResource::getColorBits () const 
{ 
   return colorBits; 
}

int GIconResource::getHeight () const 
{ 
   return height; 
}

int GIconResource::getWidth () const 
{ 
   return width; 
}

const GVector<int>& GIconResource::getPalette () const 
{ 
   return palette; 
}

const GVector<int>& GIconResource::getPixels () const 
{ 
   return pixels; 
}

int GIconResource::getPixelAt ( int x, int y ) const
{
   int idx = (y * width) + x;
   return pixels[idx];
}

void GIconResource::setPixelAt ( int x, int y, int pixelValue )
{
   int idx = (y * width) + x;
   pixels[idx] = pixelValue;
}
