/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ATOMICCOUNTER
#define __GLIB_ATOMICCOUNTER

#include "glib/primitives/GObject.h"

/**
 * Integral value that can be incremented and decremented. 
 * In a multi-threaded environment, access to the counter is synchronized 
 * to prevent lost updates.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.15
 */
template <class T> class GAtomicCounter : public GObject
{
private:
   
   T volatile counterValue;

public:

   GAtomicCounter ( T initialValue = 0 )
      :counterValue(initialValue)
   {
   }

   virtual ~GAtomicCounter ()
   {
   }

   /** Conversion operator. */
   operator T () const
   { 
      GObject::Synchronizer(this);
      return counterValue; 
   }

   /** Assignment operator. */
   T operator= ( T i )
   {
      GObject::Synchronizer(this);
      return counterValue = i;
   }

   /** Prefix increment operator. */
   T operator++ ()
   {
      GObject::Synchronizer(this);
      return ++counterValue;
   }

   /** Postfix increment operator. */
   T operator++ ( int )
   {
      GObject::Synchronizer(this);
      return counterValue++;
   }

   /** Prefix decrement operator. */
   T operator-- ()
   {
      GObject::Synchronizer(this);
      return --counterValue;
   }

   /** Postfix decrement operator. */
   T operator-- ( int )
   {
      GObject::Synchronizer(this);
      return counterValue--;
   }

   /** Addition operator. */
   T operator+= ( T i )
   {
      GObject::Synchronizer(this);
      return counterValue += i;
   }

   /** Subtraction operator. */
   T operator-= ( T i )
   {
      GObject::Synchronizer(this);
      return counterValue -= i;
   }
};

#endif // #ifndef __GLIB_ATOMICCOUNTER
