/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_RANDOMACCESSFILE
#define __GLIB_RANDOMACCESSFILE

#include "glib/vfs/GVfs.h"
#include "glib/vfs/GFile.h"
#include "glib/io/GInputStream.h"
#include "glib/io/GOutputStream.h"
#include "glib/io/GFileNotFoundException.h"
#include "glib/exceptions/GIllegalArgumentException.h"

/**
 * Instances of this class support both reading and writing to a
 * random access file.
 *
 * A random access file behaves like a large array of bytes stored in
 * the file system. There is a kind of cursor, or index into the implied
 * array, called the <i>file pointer</i>; input operations read bytes
 * starting at the file pointer and advance the file pointer past the
 * bytes read. If the random access file is created in read/write mode,
 * then output operations are also available; output operations write
 * bytes starting at the file pointer and advance the file pointer past
 * the bytes written. Output operations that write past the current end
 * of the implied array cause the array to be extended. The file pointer
 * can be read by the {@link #getFilePointer} method and set by the
 * {@link #seek} method.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.29
 */
class GRandomAccessFile : public GInputStream, 
                          public GOutputStream
{
   private:

      class GVfs& vfs;
      bool textMode;
      bool readable;
      bool writable;
      GFile fileDescr;
      GVfs::FileHandle hfile;

      /* Used only if {@link #textMode} is true. */
      mutable int nextByte;

   public:

      /**
       * Creates a random access file stream to read from, and optionally
       * to write to-, a file with the specified name.
       *
       * The mode argument must either be equal to <code>"r"</code>,
       * <code>"w"</code>, <code>"rw"</code> or <code>"a"</code>,
       * indicating that the file is to be opened for input (read) only,
       * for output (write) only, for both input and output (read and write)
       * or for appending output (write append) only, respectively.
       *
       * The write methods on this object will always throw an
       * {@link GIOException} if the file is opened with a mode of
       * <code>"r"</code>.
       *
       * The read methods on this object will always throw an
       * {@link GIOException} if the file is opened with a mode of
       * <code>"w"</code> or <code>"a"</code>.
       *
       * If the mode is <code>"w"</code>, <code>"rw"</code> or
       * <code>"a"</code> and the file does not exist, then an attempt is
       * made to create it.
       *
       * @param   vfs    The Virtual File System of where to open the file.
       * @param   name   The vfs-dependent filename of which to open.
       * @param   mode   The access mode.
       * @param   text   False if file is to be opened in binary mode, or
       *                 else true if it is to be opened in text mode.
       * @throws  GIllegalArgumentException If the mode argument is
       *                                    not valid.
       * @throws  GFileNotFoundException    If the file exists but is a
       *                                    directory rather than a regular
       *                                    file, or if file does not
       *                                    exist at all. <b>Note</b> that
       *                                    this class is a subclass of
       *                                    {@link GOpenFileException}.
       * @throws  GOpenFileException        If the file exist but cannot be
       *                                    opened or created for some
       *                                    other reason.
       */
      GRandomAccessFile ( class GVfs& vfs, 
                          const GString& name, 
                          const GString& mode, 
                          bool text );

      /**
       * Same as {@link GRandomAccessFile(const GString&, const GString&)},
       * but giving a file object instead of just the file name.
       *
       * @throws  The same as {@link GRandomAccessFile(const GString&, const GString&)}.
       */
      GRandomAccessFile ( class GVfs& vfs, 
                          const GFile& file, 
                          const GString& mode, 
                          bool text );

      /**
       * Close the file.
       */
      virtual ~GRandomAccessFile ();

   private:

      /**
       * Common construction code.
       *
       * @throws  The same as {@link GRandomAccessFile(const GString&, const GString&)}.
       */
      void init ( const GString& mode );

      /**
       * Prevent usage of the copy constructor.
       */
      GRandomAccessFile ( const GRandomAccessFile& src ) : vfs(src.vfs) {}

      /**
       * Prevent usage of the assignment operator.
       */
      const GRandomAccessFile& operator= ( const GRandomAccessFile& src ) const { return *this; }

      /**
       * Check if the file object is readable.
       *
       * If it is then we will do nothing but return normally.
       *
       * @throws  GIOException if the file object is not readable.
       */
      void checkReadable () const;

      /**
       * Check if the file object is writable.
       *
       * If it is then we will do nothing but return normally.
       *
       * @throws  GIOException if the file object is not writable.
       */
      void checkWritable () const;

   public:

      /**
       * Flush the buffered output stream bytes, if any.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.10
       * @throws  GIOException In case of any error flushing the buffer.
       */
      virtual void flush () const;

      /**
       * Use this method to get access to the details (such as the path-) 
       * of the file being accessed by this object.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.30
       */
      const class GFile& getFileDescription () const;

      /**
       * Get a reference to the Virtual File System containing the 
       * file being opened by this object.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.30
       */
      class GVfs& getVFS ();

      /**
       * Implements {@link GInputStream#readByte}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual int readByte () const;

      /**
       * Implements {@link GInputStream#readString}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual GString& readString ( GString& str ) const;

      /**
       * Implements {@link GInputStream#read}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual int read ( void* buff, int count ) const;

      /**
       * Implements {@link GInputStream#readExact}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       */
      virtual void readExact ( void* buff, int count ) const;

      /**
       * Returns the current offset in this file.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The offset from the beginning of the file, in bytes,
       *          at which the next read or write occurs.
       * @throws  GIOException  If an I/O error occurs.
       */
      longlong getFilePointer ();

      /**
       * Sets the file-pointer offset, measured from the current file
       * pointer position of this file, at which the next read or
       * write occurs.
       *
       * The offset may be set beyond the end of the file if file is
       * opened in write mode.
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.28
       * @param   pos   The offset position, measured in bytes from the
       *                current file pointer position of the file, at
       *                which to set the file pointer.
       * @throws  GIOException  In case of any error, such as an
       *                        invalid seek position specified.
       */
      void seekFromCur ( longlong pos );

      /**
       * Sets the file-pointer offset, measured from the end of the file.
       *
       * The offset may be set beyond the end of the file if file is
       * opened in write mode. Else the offset will usually be either
       * zero or a negative value.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.13
       * @param   pos   The offset position, measured in bytes from the
       *                end of the file.
       * @throws  GIOException  In case of any error, such as an
       *                        invalid seek position specified.
       */
      void seekFromEnd ( longlong pos );

      /**
       * Sets the file-pointer offset, measured from the beginning of this
       * file, at which the next read or write occurs.
       *
       * The offset may be set beyond the end of the file if file is
       * opened in write mode.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   pos   The offset position, measured in bytes from the
       *                beginning of the file, at which to set the file
       *                pointer.
       * @throws  GIOException  In case of any error, such as an
       *                        invalid seek position specified.
       */
      void seekFromStart ( longlong pos );

      /**
       * Returns the length of this file, in bytes.
       *
       * This method is support on all file modes. That is, you can use
       * this method for files in read mode as well as for files in
       * write mode.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The length of this file, measured in bytes.
       * @throws  GIOException  If an I/O error occurs. If this happens
       *                        then the caller must be aware that the
       *                        file pointer may have been changed by
       *                        this method.
       */
      longlong length ();

      /**
       * Attempts to skip over <code>n</code> bytes of input
       * discarding the skipped bytes.
       *
       * This method may skip over some smaller number of bytes,
       * possibly zero. This may result from any of a number of
       * conditions; reaching end of file before <code>n</code> bytes
       * have been skipped is only one possibility. This method never
       * throws an <code>GEOFException</code>.
       *
       * The actual number of bytes skipped is returned.
       * If <code>n</code> is negative, no bytes are skipped.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   n   The number of bytes to be skipped.
       * @return  The actual number of bytes skipped.
       * @throws  GIOException  If an I/O error occurs.
       */
      longlong skipBytes ( longlong n );

      /**
       * Implements {@link GOutputStream#writeByte}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GIOException  in case of any error writing the byte.
       */
      virtual void writeByte ( int b ) const;

      /**
       * Implements {@link GOutputStream#write}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GIOException  in case of any error writing the buffer.
       */
      virtual void write ( const void* buff, int size, int count ) const;

      /**
       * Implements {@link GOutputStream#print}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GIOException  in case of any error writing the string.
       */
      virtual int print ( const GString& str ) const;

      /**
       * Implements {@link GOutputStream#printf}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GIOException  in case of any error writing the
       *                        formatted string.
       */
      virtual int printf ( const char* str, const GVArgs& args = GVArgs() ) const;
};

#endif
