/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/io/GRandomAccessFile.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GLog.h"

GRandomAccessFile::GRandomAccessFile ( GVfs& vfs, 
                                       const GString& name, 
                                       const GString& mode, 
                                       bool text )
                  :vfs(vfs),
                   textMode(text),
                   readable(false),
                   writable(false),
                   fileDescr(name),
                   hfile(null),
                   nextByte(-1)
{
   init(mode);
}

GRandomAccessFile::GRandomAccessFile ( GVfs& vfs, 
                                       const GFile& file, 
                                       const GString& mode, 
                                       bool text )
                  :vfs(vfs),
                   textMode(text),
                   readable(false),
                   writable(false),
                   fileDescr(file),
                   hfile(null),
                   nextByte(-1)
{
   init(mode);
}

GRandomAccessFile::~GRandomAccessFile ()
{
   if (hfile != null)
      vfs.closeFile(hfile);
}

void GRandomAccessFile::init ( const GString& mode )
{
   if (mode != "r" && mode != "w" && mode != "rw" && mode != "a")
      gthrow_(GIllegalArgumentException("Mode must be r, w, rw or a."));
   readable = (mode == "r" || mode == "rw");
   writable = (mode == "w" || mode == "rw" || mode == "a");
   GString path = fileDescr.getFullPath();
   if (!writable)
      if (!vfs.existFile(path))
         gthrow_(GFileNotFoundException("Filename: " + path));
   GError err;
   bool readonly = readable && !writable;
   hfile = vfs.openFile(path, &err, 
                        readonly ? GVfs::Mode_ReadOnly : GVfs::Mode_ReadWrite, 
                        readonly ? GVfs::Create_Never : GVfs::Create_IfNew, 
                        GVfs::Share_DenyNone,
                        GVfs::OF_FLAG_RANDOM_ACCESS);
   if (hfile == null)
   {
      GString sysMsg = err.getErrorMessage();
      GString msg("%s\n\nFilename:\n%s", GVArgs(sysMsg).add(path));
      gthrow_(GOpenFileException(msg));
   }
}

const GFile& GRandomAccessFile::getFileDescription () const
{
   return fileDescr;
}

GVfs& GRandomAccessFile::getVFS ()
{
   return vfs;
}

void GRandomAccessFile::checkReadable () const
{
   if (!readable)
      gthrow_(GIOException("File not readable."));
}

void GRandomAccessFile::checkWritable () const
{
   if (!writable)
      gthrow_(GIOException("File not writable."));
}

void GRandomAccessFile::flush () const
{
   // Nothing to flush, since we use no buffering!
}

int GRandomAccessFile::readByte () const
{
   checkReadable();

   BYTE byte_;

   if (nextByte != -1)
   {
      byte_ = BYTE(nextByte);
      nextByte = -1;
   }
   else
   {
      int numRead = 0;
      GError rc = vfs.readFromFile(hfile, &byte_, 1, &numRead);
      if (rc != GError::Ok)
         gthrow_(GIOException("Filename: " + fileDescr.getFullPath()));
      if (numRead != 1)
         return -1; // End of file stream.
   }

   if (textMode)
   {
      if (byte_ == '\r')
      {
         nextByte = readByte();
         if (nextByte == '\n')
         {
            nextByte = -1;
            byte_ = '\n';
         }
      }
   }

   if (textMode && byte_ == 0x1A) // If end-of-stream (ctrl+Z).
      return -1;
   else
      return byte_;
}

GString& GRandomAccessFile::readString ( GString& str ) const
{
   str.clear();
   checkReadable();
   return GInputStream::readString(str);
}

int GRandomAccessFile::read ( void* buff, int count ) const
{
   checkReadable();

   if (count < 0)
      return 0;

   // For text-files: Read and convert byte-by-byte, via default 
   // super-implementation.
   if (textMode || nextByte != -1) 
      return GInputStream::read(buff, count);

   // For binary files: Read block directly, for speed optimizing reasons.
   int numRead = 0;
   GError rc = vfs.readFromFile(hfile, buff, count, &numRead);
   if (rc != GError::Ok)
      gthrow_(GIOException("Filename: " + fileDescr.getFullPath()));
   if (numRead == 0)
      return -1; // End-of-file.
   return numRead;
}

void GRandomAccessFile::readExact ( void* buff, int count ) const
{
   checkReadable();
   GInputStream::readExact(buff, count);
}

longlong GRandomAccessFile::getFilePointer ()
{
   GError rc;
   longlong ret = vfs.getFileSeekPos(hfile, &rc);
   if (rc != GError::Ok)
      gthrow_(GIOException(rc.getErrorMessage(), rc));
   return ret;
}

void GRandomAccessFile::seekFromStart ( longlong pos )
{
   checkReadable();
   if (pos < 0)
      gthrow_(GIOException(GString("Invalid offset: ") + pos));
   GError err = vfs.setFileSeekPosFromStart(hfile, pos);
   if (err != GError::Ok)
      gthrow_(GIOException(err.getErrorMessage(), err));
}

void GRandomAccessFile::seekFromCur ( longlong pos )
{
   if (pos == 0)
      return;
   checkReadable();
   GError err = vfs.setFileSeekPosFromCurrent(hfile, pos);
   if (err != GError::Ok)
      gthrow_(GIOException(err.getErrorMessage(), err));
}

void GRandomAccessFile::seekFromEnd ( longlong pos )
{
   checkReadable();
   GError err = vfs.setFileSeekPosFromEnd(hfile, pos);
   if (err != GError::Ok)
      gthrow_(GIOException(err.getErrorMessage(), err));
}

longlong GRandomAccessFile::length ()
{
   GError rc;
   longlong fsize = vfs.getFileSize(hfile, &rc);
   if (rc != GError::Ok)
      gthrow_(GIOException(rc.getErrorMessage(), rc));
   return fsize;
}

longlong GRandomAccessFile::skipBytes ( longlong n )
{
   if (n <= 0)
      return 0;

   ulonglong pos = getFilePointer();
   ulonglong len = length();
   ulonglong newpos = pos + n;
   if (newpos > len)
      n = len - pos;

   seekFromCur(n);
   return n;
}

void GRandomAccessFile::writeByte ( int b ) const
{
   checkWritable();
   BYTE byte_ = BYTE(b);
   GError err = vfs.writeToFile(hfile, &byte_, 1);
   if (err != GError::Ok)
      gthrow_(GIOException(err.getErrorMessage(), err));
}

void GRandomAccessFile::write ( const void* buff, int size, int count ) const
{
   checkWritable();

   if (textMode)
   {
      BYTE* byteBuff = (BYTE*) buff;
      for (int i=0; i<count; i++)
      {
         for (int i2=0; i2<size; i2++)
         {
            BYTE next = *byteBuff++;
            if (next == '\n')
            {
               writeByte('\r');
               writeByte('\n');
            }
            else
            {
               writeByte(next);
            }
         }
      }
   }
   else
   {
      int written = 0;
      GError err = vfs.writeToFile(hfile, buff, size*count, &written);
      if (written != size*count)
         gthrow_(GIOException(err.getErrorMessage(), err));
   }
}

int GRandomAccessFile::print ( const GString& str ) const
{
   checkWritable();
   int len = str.length();
   write(str.cstring(), len, 1);
   return len;
}

int GRandomAccessFile::printf ( const char* str, const GVArgs& args ) const
{
   checkWritable();
   GString data(str, args);
   int len = data.length();
   const char* buff = data.cstring();
   write(buff, len, 1);
   return len;
}
