/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TREENODE
#define __GLIB_TREENODE

#include "glib/primitives/GObject.h"

/**
 * A class representing a single node in a {@link GTreeModel}.
 *
 * This is the public interface of the tree nodes. Internally, each node
 * is actually an instance of {@link GMutableTreeNode}. However, in order 
 * to help the user code to always manipulate the content of a tree 
 * via methods in {@link GTreeModel} rather than by directly on the 
 * GTreeNode it self, the public interface of {@link GTreeModel} and 
 * {@link GTreeView} only expose this node interface.
 * 
 * @author  Leif Erik Larsen
 * @since   2006.01.04
 */
class GTreeNode : public GObject
{
   friend class GTreeView;

private:

   typedef struct OS2NodeRecord* SystemNodeHandle;

   /** 
    * The system dependent handle of this node, when contained in some 
    * kind of system dependent tree view control. This member variable is
    * maintained by various methods of {@link GTreeView} only.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.14
    * @see     #getSysHandle
    */
   SystemNodeHandle hSysHandle; // On Windows, this is a HTREEITEM. On OS/2, this is a OS2NodeRecord*.

protected:

   /** Optional user object. */
   GObject* userObject;

   /** True if we shall destroy the user object upon desctruction. */
   bool autoDeleteUserObject;

protected:

   /**
    * @author  Leif Erik Larsen
    * @since   2006.01.14
    * @param   userObject  The optional user object to be associated with 
    *                      this node.
    * @param   autoDeleteUserObject True if we shall automatically destroy
    *                      the specified user object when the tree node it
    *                      self is destructed.
    */
   GTreeNode ( GObject* userObject, bool autoDeleteUserObject );

public:

   virtual ~GTreeNode ();

public:

   /** Returns the child node at the specified index. */
   virtual GTreeNode& getChildAt ( int index ) const = 0;

   /** Returns the number of children nodes contained in this node. */
   virtual int getChildCount () const = 0;

   /** Returns the parent node of this node, or null if this is the root node. */
   virtual GTreeNode* getParent () = 0;

   /** Returns the index of the specified node in this node, or -1 if this node does not contain the specified node. */
   virtual int getIndex ( const GTreeNode& node ) const = 0;

   /** Returns true if this node allows children. */
   virtual bool getAllowsChildren () const = 0;

   /**
    * This method retrieves the next sibling of the tree node, or null
    * if the tree node has no more siblings.
    *
    * @author  Leif Erik Larsen
    * @since   2006.02.21
    */
   virtual GTreeNode* getNextSibling () = 0;

   /**
    * Returns the path from the root, to get to this node. The last
    * element in the path is this node.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @return  A copy of the full path to this node.
    */
   class GTreePath getPath () const;

   /**
    * Get the system specific handle used to represent this node.
    * On Windows, this is a HTREEITEM. On OS/2, this is a OS2NodeRecord*.
    *
    * @author  Leif Erik Larsen
    * @since   2006.02.23
    */
   GTreeNode::SystemNodeHandle getSysHandle () const;

   /**
    * Returns this node's user object.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.14
    * @return	The object stored at this node by the user.
    * @see	   GMutableTreeNode#setUserObject
    * @see	   #toString
    */
   class GObject* getUserObject ();

   /**
    * Returns true if this node has no children. To distinguish between
    * nodes that have no children and nodes that <i>cannot</i> have
    * children (e.g. to distinguish files from empty directories), use this
    * method in conjunction with {@link #getAllowsChildren}.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @return	True if this node has no children.
    */
   bool isLeaf () const;

   /**
    * Returns true if <code>aNode</code> is a direct child of this node.  
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   bool isNodeChild ( const GTreeNode& aNode ) const;
};

#endif
