/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TREEMODELEVENT
#define __GLIB_TREEMODELEVENT

#include "glib/gui/tree/GTreeNode.h"
#include "glib/util/GVector.h"
#include "glib/util/GArray.h"

/**
 * @author  Leif Erik Larsen
 * @since   2006.01.04
 */
class GTreeModelEvent
{

private:

   /** Path to the parent of the nodes that have changed, or null if it was the root node that was changed. */
   const class GTreePath* path;

   /** Indexes identifying the position of where the children were. */
   GVector<int> childIndexes;

   /** Children that have been removed. */
   GArray<GTreeNode> children;

public:

   /**
    * Used to create an event when nodes have been changed, inserted, or
    * removed, identifying the path to the parent of the modified items. 
    * All of the modified objects are siblings which are
    * direct descendents (not grandchildren) of the specified parent.
    * The positions at which the inserts, deletes, or changes occurred are
    * specified by an array of <code>int</code>. The indexes in that array
    * must be in order, from lowest to highest. Use {@link #addChildNode}
    * once for each affected child.
    * 
    * For changes, the indexes in the model correspond exactly to the indexes
    * of items currently displayed in the UI. As a result, it is not really
    * critical if the indexes are not in their exact order. But after multiple
    * inserts or deletes, the items currently in the UI no longer correspond
    * to the items in the model. It is therefore critical to specify the
    * indexes properly for inserts and deletes.
    * 
    * For inserts, the indexes represent the <i>final</i> state of the tree,
    * after the inserts have occurred. Since the indexes must be specified in
    * order, the most natural processing methodology is to do the inserts
    * starting at the lowest index and working towards the highest. Accumulate
    * a vector of <code>int</code> that specify the insert-locations as you go, 
    * then call this method once for each element as part of creating and 
    * initializing the event. When the postition-index equals zero, the node 
    * is inserted at the beginning of the list. When the position index equals 
    * the size of the list, the node is "inserted" at (appended to) the end of 
    * the list.
    * 
    * For deletes, the indexes represent the <i>initial</i> state of the tree,
    * before the deletes have occurred. Since the indexes must be specified in
    * order, the most natural processing methodology is to use a delete-counter.
    * Start by initializing the counter to zero and start work through the
    * list from lowest to higest. Every time you do a delete, add the current
    * value of the delete-counter to the index-position where the delete occurred,
    * and append the result to a vector of delete-locations. Then increment the 
    * delete-counter. The index positions stored in the vector therefore reflect 
    * the effects of all previous deletes, so they represent each object's 
    * position in the initial tree. (You could also start at the highest index 
    * and working back towards the lowest, accumulating a vector of 
    * delete-locations as you go using the {@link GArray#insert(int, 0)}.)
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   path    Identifying the path to the parent of the modified 
    *                  item(s), or null if the event is about the root node.
    *                  If null is specified then both <i>childIndexes</i>
    *                  and <i>children</i> must be empty.
    * @param   childIndexes Indexes of affected child nodes. See comments
    *                  above for important details.
    *          children The affected child nodes.
    * @see     #addChildNode
    */
   GTreeModelEvent ( const GTreePath* path, 
                     const GVector<int>& childIndexes,  
                     const GArray<GTreeNode>& children );

   /**
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   virtual ~GTreeModelEvent ();

private:

   /** Disable the copy constructor. */
   GTreeModelEvent ( const GTreeModelEvent& src ) : path(null) {}

   /** Disable the assignment operator. */
   GTreeModelEvent& operator= ( const GTreeModelEvent& src ) { return* this; }

public:

   /**
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  The path to the node containing the changed nodes, or null 
    *          if the root node was changed.
    * @see     GTreePath#getLastPathComponent
    */
   const class GTreePath* getTreePath () const;

   /**
    * Returns references to the child nodes of the node identified by
    * {@link #getTreePath} at the locations specified by 
    * {@link #getChildIndexes}. If this is a removal event the
    * returned nodes are no longer children of the parent node.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  References to the children specified by the event.
    */
   const GArray<GTreeNode>& getChildren () const;

   /**
    * Returns the index of the child nodes. If this is a removal event
    * the indexes point to locations in the initial list where items
    * were removed. If it is an insert, the indexes point to locations
    * in the final list where the items were added. For node changes,
    * the indexes point to the locations of the modified nodes.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  The index locations for the children specified by the event.
    */
   const GVector<int>& getChildIndexes () const;
};

#endif
