/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TREEMODEL
#define __GLIB_TREEMODEL

/**
 * @author  Leif Erik Larsen
 * @since   2006.01.04
 */
class GTreeModel
{

public:

   /**
    * Returns the root of the tree. Returns <code>null</code>
    * only if the tree has no nodes.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  The root of the tree.
    */
   virtual class GTreeNode* getRoot () = 0;

   /**
    * Returns the child of <code>parent</code> at index <code>index</code>
    * in the parent's child array. The specified parent must be a node 
    * previously obtained from this data source.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   parent  A node in the tree, obtained from this data source.
    * @param   index   Index of which child to return.
    * @return  The child of <code>parent</code> at the specified index.
    * @throws  GArrayIndexOutOfBoundsException if the specified index is 
    *                                          outside range.
    */
   virtual class GTreeNode& getChild ( class GTreeNode& parent, int index ) = 0;

   /**
    * Returns the number of children of the specified parent.
    * Returns 0 if the node is a leaf or if it has no children.  
    * The specified parent must be a node previously obtained 
    * from this data source.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   parent  A node in the tree, obtained from this data source.
    * @return  The number of children of the specified parent node.
    */
   virtual int getChildCount ( const class GTreeNode& parent ) const = 0;

   /**
    * Returns true if the specified node is a leaf.
    * It is possible for this method to return false even if the node has 
    * no children. A directory in a filesystem, for example, may contain 
    * no files; the node representing the directory is not a leaf, but it 
    * also has no children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   node  A node in the tree, obtained from this data source.
    * @return  true if the specified node is a leaf.
    */
   virtual bool isLeaf ( const class GTreeNode& node ) const = 0;

   /**
    * Returns the index of child in parent.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   parent A node in the tree, obtained from this data source.
    * @param   child  The node we are interested in.
    * @return  The index of the child in the parent, or -1 if the 
    *          child is not contained in the specified parent at all or 
    *          if the specified parent is not part of this tree model.
    */
   virtual int getIndexOfChild ( const class GTreeNode& parent, const class GTreeNode& child ) const = 0;

   /**
    * Adds a listener for the {@link GTreeModelEvent} posted after 
    * the tree changes.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   l    The listener of which to add.
    * @see     #removeTreeModelListener
    */
   virtual void addTreeModelListener ( class GTreeModelListener* l ) = 0;

   /**
    * Removes a listener previously added with {@link #addTreeModelListener}.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   l    The listener of which to remove.
    * @see     #addTreeModelListener
    */  
   virtual void removeTreeModelListener ( class GTreeModelListener* l ) = 0;
};

#endif
