/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_GRIDLAYOUT
#define __GLIB_GRIDLAYOUT

#include "glib/gui/layout/GLayoutManager.h"

/**
 * The <i>GridLayout</i> class is a layout manager that
 * lays out a container's components in a rectangular grid.
 *
 * The container is divided into equal-sized rectangles,
 * and one component is placed in each rectangle.
 *
 * When both the number of rows and the number of columns have
 * been set to non-zero values, either by a constructor or
 * by the <tt>setRows</tt> and <tt>setColumns</tt> methods, the number of
 * columns specified is ignored.  Instead, the number of
 * columns is determined from the specified number or rows
 * and the total number of components in the layout. So, for
 * example, if three rows and two columns have been specified
 * and nine components are added to the layout, then they will
 * be displayed as three rows of three columns.  Specifying
 * the number of columns affects the layout only when the
 * number of rows is set to zero.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.23
 * @see     GWindow#setLayoutManager
 */
class GGridLayout : public GLayoutManager
{
   private:

      /**
       * This is the number of rows specified for the grid. The number
       * of rows can be changed at any time.
       * This should be a non negative integer, where '0' means
       * 'any number' meaning that the number of Rows in that
       * dimension depends on the other dimension.
       */
      int rows;

       /**
        * This is the number of columns specified for the grid. The number
        * of columns can be changed at any time.
        * This should be a non negative integer, where '0' means
        * 'any number' meaning that the number of Columns in that
        * dimension depends on the other dimension.
        */
      int cols;

      /**
       * This is the horizontal gap (in pixels) which specifies the space
       * between columns. They can be changed at any time.
       * This should be a non negative integer.
       */
      int hgap;

      /**
       * This is the vertical gap (in pixels) which specifies the space
       * between rows. They can be changed at any time.
       * This should be a non negative integer.
       */
      int vgap;

   public:

      /**
       * Creates a grid layout with the specified number of rows and
       * columns. All components in the layout are given equal size.
       *
       * In addition, the horizontal and vertical gaps are set to the
       * specified values. Horizontal gaps are placed at the left and
       * right edges, and between each of the columns. Vertical gaps are
       * placed at the top and bottom edges, and between each of the rows.
       *
       * One, but not both, of <code>rows</code> and <code>cols</code> can
       * be zero, which means that any number of objects can be placed in a
       * row or in a column.
       *
       * @param   rows   the rows, with the value zero meaning
       *                 any number of rows.
       * @param   cols   the columns, with the value zero meaning
       *                 any number of columns.
       * @param   hgap   the horizontal gap.
       * @param   vgap   the vertical gap.
       */
      GGridLayout ( int rows, int cols, int hgap = 0, int vgap = 0 );
      virtual ~GGridLayout ();

   public:

      /**
       * @see GLayoutManager#preferredLayoutSize
       */
      GDimension preferredLayoutSize ( GWindow& parent );

      /**
       * @see GLayoutManager#preferredLayoutWidth
       */
      int preferredLayoutWidth ( GWindow& parent );

      /**
       * @see GLayoutManager#preferredLayoutHeight
       */
      int preferredLayoutHeight ( GWindow& parent );

      /**
       * @see GLayoutManager#layoutWindow
       */
      void layoutWindow ( GWindow& parent );
};

#endif

