/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_WINDOWMESSAGE
#define __GLIB_WINDOWMESSAGE

#include "glib/primitives/GString.h"

/**
 * The used to contain a single "low level" message event that is delivered 
 * to the {@link GWindow#handleWindowMessage} method of a specific window.
 *
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
class GWindowMessage : public GObject
{
   public:

      typedef ULONG IDType;
      typedef MPARAM Param1;
      typedef MPARAM Param2;
      typedef MRESULT Answer;

   public:

      /**
       * Class used to contain mouse and keyboard status flags that are
       * delivered with mouse and keyboard input message events, such as
       * {@link GWindow#onButton1Down} and {@link GWindow#onMouseMove}.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.29
       */
      class InputFlags : public GObject
      {
         private:

            int flags;
            bool mouseButton1Down;
            bool mouseButton2Down;

         public:

            InputFlags ( int flags );
            virtual ~InputFlags ();

         public:

            int getFlags () const;
            bool isAltKeyDown () const;
            bool isControlKeyDown () const;
            bool isShiftKeyDown () const;
            bool isMouseButton1Pressed () const;
            bool isMouseButton2Pressed () const;
            bool isAnyMouseButtonPressed () const;
      };

   private:

      int id;
      Param1 mp1;
      Param2 mp2;
      Answer answer;

   public:

      GWindowMessage ( int id, Param1 mp1, Param2 mp2 );

      explicit GWindowMessage ( const QMSG& msg );

      virtual ~GWindowMessage ();

   public:

      int getID () const;
      Answer getAnswer () const;

      /**
       * Get the first message parameter.
       */
      Param1 getParam1 () const;

      /**
       * Get the first message parameter as an integer.
       */
      int getParam1Int () const;

      /**
       * Get the first message parameter as a boolean.
       */
      bool getParam1Bool () const;

      /**
       * Get the low order short of the first message parameter.
       */
      int getParam1LoShort () const;

      /**
       * Get the high order short of the first message parameter.
       */
      int getParam1HiShort () const;

      /**
       * Get the low order unsigned short of the first message parameter.
       */
      int getParam1LoUShort () const;

      /**
       * Get the high order unsigned short of the first message parameter.
       */
      int getParam1HiUShort () const;

      /**
       * Get the second message parameter.
       */
      Param2 getParam2 () const;

      /**
       * Get the second message parameter as an integer.
       */
      int getParam2Int () const;

      /**
       * Get the secind message parameter as a boolean.
       */
      bool getParam2Bool () const;

      /**
       * Get the low order short of the second message parameter.
       */
      int getParam2LoShort () const;

      /**
       * Get the high order short of the second message parameter.
       */
      int getParam2HiShort () const;

      /**
       * Get the low order unsigned short of the second message parameter.
       */
      int getParam2LoUShort () const;

      /**
       * Get the high order unsigned short of the second message parameter.
       */
      int getParam2HiUShort () const;

      // Setter methods.
      void setID ( int id );
      void setAnswer ( Answer answer );
      void setParam1 ( Param1 mp1 );
      void setParam2 ( Param2 mp2 );

      /**
       * Get a string representation of the message object.
       * The returned string should be used for debugging purposes only.
       * Its format is likely to change in the future.
       */
      GString toString () const;
};

#endif
