/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_BORDER
#define __GLIB_BORDER

#include "glib/gui/GColor.h"
#include "glib/primitives/GRectangle.h"

/**
 * This is the abstract base class of all border classes.
 * It has one pure virtual method that must be implemented by the subclass,
 * and that is {@link #paintTheBorder}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.17
 */
class GBorder : public GObject
{
   public:

      /** Only valid just after {@link #onPaint} has been called. */
      mutable GRectangle leftBackgroundArea;
      /** Only valid just after {@link #onPaint} has been called. */
      mutable GRectangle rightBackgroundArea;
      /** Only valid just after {@link #onPaint} has been called. */
      mutable GRectangle topBackgroundArea;
      /** Only valid just after {@link #onPaint} has been called. */
      mutable GRectangle bottomBackgroundArea;

   public:

      GBorder ();
      virtual ~GBorder ();

   public:

      /**
       * Paint all parts of the border, including its background area.
       * Will first calculate the border background area's, 
       * and then call the overridable {@link #paintTheBorder}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       * @param   g    The graphics object of where to paint the border. 
       * @param   rect The outer bounds rectangle border area.
       * @param   ins  The dimensions of the border, inside rectangle.
       */
      void onPaint ( class GGraphics& g,
                     const class GRectangle& rect, 
                     const class GInsets& ins ) const;

   protected:

      /**
       * Can be called by the subclass overridden {@link #paintTheBorder} 
       * to paint the background area of the border. The background area 
       * is defined by the background area rectangles that is calculated 
       * by {@link #onPaint}, e.g. {@link #leftBackgroundArea}, 
       * {@link #bottomBackgroundArea}, etc.
       *
       * The default implementation of this method justs clears the 
       * background area rectangles using the current color of the 
       * specified graphics object.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       * @param   g    The graphics object of where to paint the background.
       */
      virtual void paintBackGround ( class GGraphics& g ) const;

      /**
       * This method is automatically called by {@link #onPaint} in order 
       * to let the subclass paint all parts of the border, including its 
       * background area.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       * @param   g    The graphics object of where to paint the border. 
       * @param   rect The outer bounds rectangle border area.
       * @param   ins  The dimensions of the border, inside rectangle.
       */
      virtual void paintTheBorder ( class GGraphics& g,
                                    const class GRectangle& rect, 
                                    const class GInsets& ins ) const = 0;
};

#endif
