/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TREEVIEW
#define __GLIB_TREEVIEW

#include "glib/gui/GWindow.h"
#include "glib/gui/tree/GTreeModelListener.h"
#include "glib/gui/tree/GTreeViewListener.h"

/**
 * @author  Leif Erik Larsen
 * @since   2006.01.04
 */
class GTreeView : public GWindow, public GTreeModelListener
{

   private:

      /**
       * This class is used to wrap the system dependent tree window,
       * which is contained as a child window of GTreeView in order to hide
       * it from the user code.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.04
       */
      class Peer : public GWindow
      {
         friend class GTreeView;

         private:

            GTreeView& tree;

         public:

            /** The owner push button. */
            explicit Peer ( GTreeView& tree );
            virtual ~Peer ();

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) : tree(src.tree) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }

            /**
             * Get the node at the specified mouse pixel position.
             * If there is no node at that point, return null.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.23
             */
            class GMutableTreeNode* getNodeAtPoint ( int xpos, int ypos );

            /**
             * Get the rectangle of the node +/- button of the specified node.
             * If the node is outside visible area of the tree view then
             * we will return a zero rectangle (zero width & zero height).
             * If an unknown node (nor part of this tree) is specified then 
             * the return value is unspecified.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.23
             * @see     #getNodeRectangle
             * @see     #getNodeIconAndTextRectangle
             */
            GRectangle getNodeButtonRectangle ( const class GMutableTreeNode& node ) const;

            /**
             * Get the rectangle of where the icon and the text of the 
             * node is painted. If the node is outside visible area of the 
             * tree view then we will return a zero rectangle (zero 
             * width & zero height). If an unknown node (nor part of this 
             * tree) is specified then the return value is unspecified.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.28
             * @see     #getNodeButtonRectangle
             * @see     #getNodeRectangle
             */
            GRectangle getNodeIconAndTextRectangle ( const class GMutableTreeNode& node ) const;

            /**
             * Get the painting rectangle area of the specified node.
             * If the node is outside visible area of the tree view then
             * we will return a zero rectangle (zero width & zero height).
             * If an unknown node (nor part of this tree) is specified then 
             * the return value is unspecified.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.22
             * @see     #getNodeButtonRectangle
             * @see     #getNodeIconAndTextRectangle
             */
            GRectangle getNodeRectangle ( const class GMutableTreeNode& node ) const;

            /**
             * Draws the open/close button.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.21
             */
            void paintButton ( GGraphics& g, const GRectangle& r, bool expanded );

            /**
             * Paints the lines of an item.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.21
             */
            void paintItemLines ( class GMutableTreeNode& node, 
                                  class GMutableTreeNode* parent, 
                                  class GGraphics& g, 
                                  const class GRectangle& r );

            /**
             * This method recursively paints the parent item lines 
             * (vertical). The lines are drawn with the current color 
             * and pen type of the specified grapgics object.
             *
             * @author  Leif Erik Larsen
             * @since   2006.02.21
             */
            void paintParentLine ( class GMutableTreeNode& parent, 
                                   class GGraphics& g, 
                                   class GRectangle& rect );

         protected:

            virtual bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
            virtual bool onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
            virtual bool onKeyDown ( const GKeyMessage& key );
      };

      friend class Peer;

      bool autoDeleteTreeModel;
      class GTreeModel* treeModel;

      /** Listeners. */
      GVector<GTreeViewListener*> listenerList;

      /**
       * For some reason, using CM_QUERYRECORDEMPHASIS to query
       * current selection doesn't work. It seems to always return null. 
       * I was therefore forced to find a hack for an alternative way to 
       * keep track of the current selection. Using an instance variable 
       * that is updated whenever the current selection change seems to 
       * be a fine and stable solution.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.13
       */
      class GTreeNode* currentSelectedNode;

      bool hasLines;
      bool linesAtRoot;
      bool hasButtons;
      bool showRoot;
      bool autoGrabFocus;
      int indentWidth;
      class GIcon* iconButtExpanded;
      class GIcon* iconButtCollapsed;

      mutable Peer peer;

   public:

      GTreeView ( const GString& name,
                  const GString& constraints,
                  GWindow& parentWin,
                  bool border = true,
                  bool hasLines = true,
                  bool linesAtRoot = false,
                  bool hasButtons = true,
                  bool showRoot = false,
                  bool autoGrabFocus = true,
                  long winStyle = WS_VISIBLE,
                  long winStyle2 = 0 );

      virtual ~GTreeView ();

   private:

      /** Disable the copy constructor. */
      GTreeView ( const GTreeView& );

      /** Disable the assignment operator. */
      void operator= ( const GTreeView& );

   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2006.03.02
       */
      void drawNode ( class GGraphics& g, 
                      class GMutableTreeNode& node,
                      const class GRectangle& rect_,
                      bool nodeIsExpanded,
                      bool nodeIsSelected,
                      bool nodeHasCaretFocus );

      void fireTreeNodeIsCollapsing ( class GTreeNode& node );
      void fireTreeNodeIsExpanding ( class GTreeNode& node );
      void fireTreeNodeSelectionHasChanged ();

      /** Part of {@link GTreeModelListener}. */
      virtual void treeNodesChanged ( const class GTreeModelEvent& e );

      /** Part of {@link GTreeModelListener}. */
      virtual void treeNodesInserted ( const class GTreeModelEvent& e );

      /** Part of {@link GTreeModelListener}. */
      virtual void treeNodesRemoved ( const class GTreeModelEvent& e );

   protected:

      virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );
      virtual bool onBackgroundColorChanged ( const GColor& color );
      virtual bool onForegroundColorChanged ( const GColor& color );
      virtual bool onFontNameSizeChanged ( const GString& fontNameSize );

   public:

      /**
       * Adds a listener for events on this Tree View.
       *
       * @see     #removeTreeViewListener
       * @param   l   The listener to add.
       */
      virtual void addTreeViewListener ( class GTreeViewListener* l );

      /**
       * Collapse the specified node only.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.23
       * @see     #expandNode
       */
      void collapseNode ( class GTreeNode& node );

      /**
       * Ensures that the specified node is visible, expanding the parent 
       * item and/or scrolling the tree-view, if necessary.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.03
       */
      void ensureNodeIsVisible ( const class GTreeNode& node );

      /**
       * Expand the specified node only.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.03
       * @see     #collapseNode
       */
      void expandNode ( class GTreeNode& node );

      /**
       * Expand all nodes of the specified path.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.03
       */
      void expandPath ( const class GTreePath& path );

      /**
       * Get the currently selected node of this tree view, 
       * or null if the tree view has no current selected node.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.03
       * @see     #setSelectedNode
       */
      class GMutableTreeNode* getSelectedNode ();

      /**
       * Check if the specified node is currently expanded or not.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.23
       * @return  True if the node is expanded, or false if it is collapsed.
       * @see     #expandNode
       */
      bool isNodeExpanded ( const class GTreeNode& node ) const;

      /**
       * Return true if and only if the tree view is set to show the 
       * root node. This is a property that is set only upon 
       * object constructioon.
       *
       * @author  Leif Erik Larsen
       * @since   2006.09.08
       */
      bool isShowRoot () const;

      /**
       * Returns the string value of the current selected node,
       * or an empty string if the tree has no current selected node.
       *
       * @author  Leif Erik Larsen
       * @since   2006.03.03
       * @see     #getSelectedNode
       */
      virtual GString queryValue () const;

      /**
       * Removes a listener previously added with {@link #addTreeViewListener}.
       *
       * @see     #addTreeViewListener
       * @param   l   The listener to remove.
       */  
      virtual void removeTreeViewListener ( class GTreeViewListener* l );

      /**
       * Set the specified node to be the currently selected node 
       * of this tree view.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.28
       * @see     #getSelectedNode
       * @see     #unselectSelectedNode
       */  
      void setSelectedNode ( const class GTreeNode& node );

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.14
       */
      void setTreeModel ( class GTreeModel* newModel, bool autoDelete );

      /**
       * Expand the specified node if it is currently collapsed,
       * or collapse it if it is currently expanded.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.23
       * @return  The new expanded state of the node. True if we did expand  
       *          it, or else false if we did collapse it.
       * @see     #expandNode
       */
      bool toggleNodeExpanded ( class GTreeNode& node );

      /**
       * Unselect the currently selected node of the tree, making the 
       * tree having no currently selected node.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.28
       * @see     #setSelectedNode
       */  
      void unselectSelectedNode ();
};

#endif
