/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GTreeView.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GIcon.h"
#include "glib/gui/border/GEtchedBorder.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/gui/tree/GTreePath.h"
#include "glib/gui/tree/GTreeModelEvent.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/gui/tree/GDefaultTreeModel.h"
#include "glib/gui/tree/GMutableTreeNode.h"
#include "glib/gui/event/GKeyMessage.h"

/**
 * @author  Leif Erik Larsen
 * @since   2006.03.03
 */
struct OS2NodeRecord : public RECORDCORE
{
   /** A pointer to the additional data of ours. */
   GTreeNode* node;
};

GTreeView::Peer::Peer ( GTreeView& tree )
            :GWindow(&tree,
                     "Peer",
                     GBorderLayout::CENTER,                     
                     WS2_DEFAULTPAINT | WS2_NOT_STATIC | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT,
                     GGraphics::DUMMY_COLOR, // defBckColor
                     GGraphics::DUMMY_COLOR, // defFrgColor
                     GString::Empty,         // defFont
                     null,                   // ownerWin
                     WS_VISIBLE | /* TODO: CCS_AUTOPOSITION | */ CCS_READONLY | CCS_SINGLESEL,
                     GWindowClass::TREE),
             tree(tree)
{
   setFocusable(true);
}

GTreeView::Peer::~Peer ()
{
}

bool GTreeView::Peer::onKeyDown ( const GKeyMessage& key )
{
   GKey::KeyCode code = key.getCode();
   switch (code)
   {
      case GKey::KEY_RIGHT:
      case GKey::KEY_LEFT:
      case GKey::KEY_UP:
      case GKey::KEY_DOWN:
         if (dynamic_cast<GDialogFrame*>(&getTopLevelWindow()) == null)
            break;
         // Give this keyboard event directly to the system so the 
         // Tree View Control can handle these keys instead of 
         // the dialog box grabbing them.
         return callDefaultKeyHandler(key, true, true);

      default:
         // The other listbox navigation keys (e.g. HOME and PAGEUP)
         // can be ignored here because they won't be eaten up by the 
         // dialog keyboard handler and will therefore go to the 
         // system dependent keyboard handler so that the user can 
         // use the system default listbox keyboard navigation keys.
         break;
   }
   return GWindow::onKeyDown(key);
}

GTreeView::GTreeView ( const GString& name,
                       const GString& constraints,
                       GWindow& parentWin,
                       bool border,
                       bool hasLines,
                       bool linesAtRoot,
                       bool hasButtons,
                       bool showRoot,
                       bool autoGrabFocus,
                       long winStyle,
                       long winStyle2 )
          :GWindow(&parentWin,
                   name,
                   constraints,
                   winStyle2 | WS2_OS2Y | WS2_NOT_STATIC | WS2_AUTO_PAINT_BACKGROUND,
                   GGraphics::DUMMY_COLOR,
                   GGraphics::DUMMY_COLOR,
                   GString::Empty,
                   null,
                   winStyle),
           autoDeleteTreeModel(false),
           treeModel(null),
           listenerList(4),
           currentSelectedNode(null),
           hasLines(hasLines),
           linesAtRoot(linesAtRoot),
           hasButtons(hasButtons),
           showRoot(showRoot),
           autoGrabFocus(autoGrabFocus),
           indentWidth(0),
           iconButtExpanded(null), // TODO: Should be configurable!
           iconButtCollapsed(null), // TODO: Should be configurable!
           peer(*this)
{
   // ---
   if (border)
   {
      setInsets(new GInsets(3, 3, 3, 3), true);
      setBorder(new GEtchedBorder(), true);
   }

   // ---
   indentWidth = 18;
   // Setup the tree control to our needs.
   CNRINFO ci = { 0 };
   ci.flWindowAttr = CV_TREE | CV_ICON | CA_OWNERDRAW;
   ci.cxTreeIndent = indentWidth;
   ci.cyLineSpacing = 2;
   ci.slBitmapOrIcon.cx = 9;
   ci.slBitmapOrIcon.cy = 9;
   ci.slTreeBitmapOrIcon.cx = 20;
   ci.slTreeBitmapOrIcon.cy = 20;
   UINT attr = CMA_FLWINDOWATTR | CMA_LINESPACING | CMA_CXTREEINDENT | CMA_SLBITMAPORICON | CMA_SLTREEBITMAPORICON;
   peer.sendMessage(CM_SETCNRINFO, GWindowMessage::Param1(&ci), GWindowMessage::Param2(attr));

   // ---
   setBackgroundColor(GColor::WHITE); // GSystem::GetSystemColor(GSystem::SCID_BUTTONBCK));
   setForegroundColor(GColor::BLACK); // GSystem::GetSystemColor(GSystem::SCID_BUTTONTXT));
   setLayoutManager(new GBorderLayout(), true);
}

GTreeView::~GTreeView ()
{
   if (autoDeleteTreeModel)
      delete treeModel;
}

bool GTreeView::onBackgroundColorChanged ( const GColor& color )
{
   // Peer should have the same background color as requested by user code.
   peer.setBackgroundColor(color);
   GWindow::onBackgroundColorChanged(color);
   return true;
}

bool GTreeView::onForegroundColorChanged ( const GColor& color )
{
   // Peer should have the same foreground color as requested by user code.
   peer.setForegroundColor(color);
   GWindow::onForegroundColorChanged(color);
   return true;
}

bool GTreeView::onFontNameSizeChanged ( const GString& fontNameSize )
{
   // Peer should have the same font as requested by user code.
   peer.setFontNameSize(fontNameSize);
   GWindow::onFontNameSizeChanged(fontNameSize);
   return true;
}

void GTreeView::addTreeViewListener ( GTreeViewListener* l )
{
   if (l == null)
      gthrow_(GIllegalArgumentException("l == null"));
   if (!listenerList.contains(l)) // Don't add if it is already there!
      listenerList.add(l);
}

void GTreeView::removeTreeViewListener ( GTreeViewListener* l )
{
   int idx = listenerList.indexOf(l);
   if (idx >= 0)
      listenerList.remove(idx);
}

void GTreeView::fireTreeNodeSelectionHasChanged ()
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = listenerList.getCount();
   GVector<GTreeViewListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(listenerList[i]);

   // Invoke the listeners.
   for (int i=0; i<count; i++) 
   {
      GTreeViewListener* l = llist[i];
      l->treeNodeSelectionHasChanged();
   }

   // Also fire the value-change listeners (if any).
   fireValueChangeListeners();
}

void GTreeView::fireTreeNodeIsCollapsing ( class GTreeNode& node )
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   int count = listenerList.getCount();
   GVector<GTreeViewListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(listenerList[i]);

   // Invoke the listeners.
   for (int i=0; i<count; i++) 
   {
      GTreeViewListener* l = llist[i];
      l->treeNodeIsCollapsing(node);
   }
}

void GTreeView::fireTreeNodeIsExpanding ( class GTreeNode& node )
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   int count = listenerList.getCount();
   GVector<GTreeViewListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(listenerList[i]);

   // Invoke the listeners.
   for (int i=0; i<count; i++) 
   {
      GTreeViewListener* l = llist[i];
      l->treeNodeIsExpanding(node);
   }
}

void GTreeView::Peer::paintParentLine ( GMutableTreeNode& parent, 
                                        GGraphics& g, 
                                        GRectangle& r )
{ 
   // Check if the parent has a parent itself and process it.
   GTreeNode* grand = parent.getParent();
   if (dynamic_cast<GMutableTreeNode*>(grand) != null)
   {
      GMutableTreeNode& mgrand = dynamic_cast<GMutableTreeNode&>(*grand);
      paintParentLine(mgrand, g, r);
   }
   
   // Check if the parent has a sibling. If so, draw the vertical line.
   if (tree.hasLines && parent.getNextSibling() != null)
   { 
      bool isVisibleRoot = false;
      if (tree.showRoot)
         isVisibleRoot = (&parent == tree.treeModel->getRoot());
      else
         isVisibleRoot = (grand == tree.treeModel->getRoot());

      // Now, check if this is a root item.
      int xpos = r.x + tree.indentWidth/2;
      if (!isVisibleRoot || tree.linesAtRoot)
         g.drawLine(xpos, r.y, xpos, r.y + r.height);
   }

   // Advance the drawing position.
   if (tree.showRoot || &parent != tree.treeModel->getRoot())
      r.x += tree.indentWidth;
}

void GTreeView::Peer::paintItemLines ( GMutableTreeNode& node, 
                                       GMutableTreeNode* parent, 
                                       GGraphics& g, 
                                       const GRectangle& r )
{
   int x = r.x + tree.indentWidth / 2;
   int y = r.y;
   int xm = tree.indentWidth / 2;
   int ym = r.height / 2;

   if (!tree.showRoot && parent == tree.treeModel->getRoot())
      parent = null;

   if (parent != null || node.getPreviousSibling() != null)
   {
      if (parent != null || tree.linesAtRoot)
         g.drawLine(x, y, x, y + ym); // Connect to prev / parent.
   }
	
   y += ym;
   g.drawLine(x, y, x + xm + 1, y); // Connect to text / icon.
	
   if (node.getNextSibling() != null)
   {
      if (parent != null || tree.linesAtRoot)
         g.drawLine(x, y, x, y + ym); // Connect to next
   }
}

void GTreeView::Peer::paintButton ( GGraphics& g, const GRectangle& r, bool expanded )
{
   if (tree.iconButtCollapsed != null && tree.iconButtExpanded != null)
   {
      int ypos = r.y;
      int xpos = r.x + (tree.indentWidth - 16)/2;
      GIcon* icon = (expanded ? tree.iconButtExpanded : tree.iconButtCollapsed);
      g.drawIcon(xpos, ypos, *icon);
   }
   else
   {
      int h = r.height;
      int x = r.x + (tree.indentWidth - 9) / 2;
      int y = r.y + (h - 9) / 2 + 1;

		// Draw the box.
      g.setColor(GColor::DGRAY);
      g.setLineType(GGraphics::LTSolid);
      g.drawRectangle(x, y, 9, 9);
      g.drawFilledRectangle(x+1, y+1, 7, 7, GColor::WHITE);

      // Draw the - or + sign.
      g.setColor(GColor::BLACK);
      g.drawLine(x + 2, y + 4, x + 7, y + 4); // '-'
      if (!expanded)
         g.drawLine(x + 4, y + 2, x + 4, y + 7); // '+'
   }
}

GMutableTreeNode* GTreeView::Peer::getNodeAtPoint ( int xpos, int ypos )
{
   int os2y = sysY(ypos);
   QUERYRECFROMRECT qr = { 0 };
   qr.cb = sizeof(qr);
   qr.rect.xLeft = xpos;
   qr.rect.xRight = xpos + 1;
   qr.rect.yBottom = os2y;
   qr.rect.yTop = os2y + 1;
   qr.fsSearch = CMA_PARTIAL;
   LONG res = LONG(sendMessage(CM_QUERYRECORDFROMRECT, MPARAM(CMA_FIRST), MPARAM(&qr)));
   if (res == NULL || res == -1)
      return null;
   OS2NodeRecord* item = (OS2NodeRecord*) res;
   GMutableTreeNode* node = dynamic_cast<GMutableTreeNode*>(item->node);
   return node;
}

bool GTreeView::Peer::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (tree.autoGrabFocus)
      grabFocus();

   GMutableTreeNode* node = getNodeAtPoint(xpos, ypos);
   if (node == null)
   {
      tree.unselectSelectedNode();
      return true;
   }

   if (tree.hasButtons)
   {
      GRectangle buttonRect = getNodeButtonRectangle(*node);
      if (buttonRect.isPointInRect(xpos, ypos))
      {
         tree.toggleNodeExpanded(*node);
         return true;
      }
   }

   GRectangle rect = getNodeIconAndTextRectangle(*node);
   if (rect.isPointInRect(xpos, ypos))
   {
      tree.setSelectedNode(*node);
      return true;
   }

   tree.unselectSelectedNode();
   return true;
}

bool GTreeView::Peer::onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   GMutableTreeNode* node = getNodeAtPoint(xpos, ypos);
   if (node == null)
      return true;

   GRectangle buttonRect = getNodeButtonRectangle(*node);
   if (buttonRect.isPointInRect(xpos, ypos))
      tree.toggleNodeExpanded(*node);

   return true;
}

GRectangle GTreeView::Peer::getNodeButtonRectangle ( const class GMutableTreeNode& node ) const
{
   if (!tree.hasButtons || !node.getAllowsChildren())
      return GRectangle();
   GRectangle rect = getNodeRectangle(node);
   if (rect.width > 0 && rect.height > 0)
   {
      int level = node.getLevel();
      if (!tree.showRoot)
         level -= 1;
      rect.x = (tree.indentWidth * level) + 5;
      rect.y = rect.y + (rect.height - 9) / 2 + 1;
      rect.width = 9;
      rect.height = 9;
   }
   return rect;
}

GRectangle GTreeView::Peer::getNodeIconAndTextRectangle ( const class GMutableTreeNode& node ) const
{
   GRectangle rect = getNodeRectangle(node);
   const GIcon* icon = node.getIconClosed();
   GString text = node.toString();
   int level = node.getLevel();
   if (!tree.showRoot)
      level -= 1;
   rect.x = (tree.indentWidth * level) + 19;
   rect.width = getWidthOfString(text) + 4;
   if (icon != null)
      rect.width += icon->getWidth() + 4;
   return rect;
}

GRectangle GTreeView::Peer::getNodeRectangle ( const class GMutableTreeNode& node ) const
{
   GTreeView::Peer* self = const_cast<GTreeView::Peer*>(this); // Throw away const.
   RECTL sysRect = { 0 };
   QUERYRECORDRECT qr = { 0 };
   qr.cb = sizeof(QUERYRECORDRECT);
   qr.pRecord = node.getSysHandle();
   qr.fsExtent = CMA_TREEICON | CMA_ICON | CMA_TEXT;
   self->sendMessage(CM_QUERYRECORDRECT, MPARAM(&sysRect), MPARAM(&qr));
   sysRect.xLeft = 0; // We want the rect of the whole item line.
   sysRect.xRight = getWindowFaceRect().width - 1; // TODO: getNodeWidth()
   GRectangle rect = GWindow::MakeLibRect(sysRect, *this);
   int itemHeight = 20 + 2; // TODO: ???
   if (rect.height < itemHeight)
   {
      int inflate = (itemHeight - rect.height + 1) / 2;
      rect.inflateRect(0, inflate);
   }
   return rect;
}

GMutableTreeNode* GTreeView::getSelectedNode ()
{
   return dynamic_cast<GMutableTreeNode*>(currentSelectedNode);
}

void GTreeView::setSelectedNode ( const GTreeNode& node )
{
   ensureNodeIsVisible(node);
   OS2NodeRecord* item = node.getSysHandle();
   USHORT emphasis = TRUE;
   USHORT attr = CRA_SELECTED;
   sendMessage(CM_SETRECORDEMPHASIS, MPARAM(item), MPFROM2SHORT(emphasis, attr));
}

void GTreeView::unselectSelectedNode ()
{
   RECORDCORE* item = (RECORDCORE*) sendMessage(CM_QUERYRECORDEMPHASIS, MPARAM(CMA_FIRST), MPARAM(CRA_SELECTED));
   if (item != NULL && LONG(item) != -1)
   {
      USHORT emphasis = FALSE;
      USHORT attr = CRA_SELECTED;
      sendMessage(CM_SETRECORDEMPHASIS, MPARAM(item), MPFROM2SHORT(emphasis, attr));
   }
}

void GTreeView::drawNode ( GGraphics& g, 
                           GMutableTreeNode& node, 
                           const GRectangle& rect_,
                           bool nodeIsExpanded,
                           bool nodeIsSelected,
                           bool nodeHasCaretFocus )
{
   // Get a working copy of the rectangle area of the node.
   GRectangle rect = rect_;

   // Draw the parent lines, if any.
   g.setColor(GColor::DGRAY);
   g.setLineType(GGraphics::LTSolid);
   GMutableTreeNode* parent = dynamic_cast<GMutableTreeNode*>(node.getParent());
   if (parent != null)
      peer.paintParentLine(*parent, g, rect);

   // Draw the lines connecting to the previous and next items, if any.
   if (hasLines)
      peer.paintItemLines(node, parent, g, rect);

   // Paint the +/- button, if any.
   if (hasButtons)
   {
      if (nodeIsExpanded)
         peer.paintButton(g, rect, true);
      else
      if (node.getAllowsChildren() || node.getChildCount() > 0)
         peer.paintButton(g, rect, false);
      else
         ; // No, don't paint the +/- button at all.
   }

   // If we have buttons or line, we must make room for them.
   if (hasButtons || hasLines)
      rect.x += indentWidth;

   // Draw the icon, if any.
   const GIcon* iconOpened = node.getIconOpened();
   const GIcon* iconClosed = node.getIconClosed();
   const GIcon* icon = null;
   if (nodeIsExpanded)
      icon = iconOpened != null ? iconOpened : iconClosed;
   else
      icon = iconClosed != null ? iconClosed : iconOpened;
   if (icon != null)
   {
      int xpos = rect.x + 1;
      int ypos = rect.y + rect.height/2 - icon->getHeight()/2;
      g.drawIcon(xpos, ypos, *icon);
      rect.x += icon->getWidth() + 1;
   }

   // Calculate the text drawing rectangle.
   GRectangle textRect = rect;
   GString text = node.toString();
   textRect.width = g.getWidthOfString(text) + 4;
   textRect.x += 2;

   // Clear the background of the node.
   GRectangle textBox = textRect;
   GColor bckColor = nodeIsSelected ? GColor::DBLUE : getBackgroundColor();
   GColor frgColor = nodeIsSelected ? GColor::WHITE : getForegroundColor();
   g.drawFilledRectangle(textBox, bckColor);

   // Draw the text.
   textRect.x += 2;
   textRect.width -= 2;
   g.drawText(text, textRect, frgColor, bckColor);

   // Draw the focus rectangle.
   if (nodeHasCaretFocus)
   {
      // TODO: ???
   }
}

GWindowMessage::Answer GTreeView::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_DRAWITEM:
      {
         if (msg.getParam1LoUShort() != peer.getWindowID())
            return GWindow::handleWindowMessage(msg);
         OWNERITEM* oi = (OWNERITEM*) msg.getParam2();
         if (oi == null) // Should never happen, but in case.
            return MRESULT(FALSE);
         if ((oi->idItem & CMA_TEXT) == 0)
            return MRESULT(TRUE);
         CNRDRAWITEMINFO* iinf = (CNRDRAWITEMINFO*) oi->hItem;
         if (iinf == null) // Should never happen, but in case.
            return MRESULT(FALSE);
         OS2NodeRecord* item = (OS2NodeRecord*) iinf->pRecord;
         if (item == null) // Should never happen, but in case.
            return MRESULT(FALSE);
         GMutableTreeNode* node = dynamic_cast<GMutableTreeNode*>(item->node);
         if (node == null) // Should never happen, but in case.
            return MRESULT(FALSE);
         // Draw the node.
         GGraphics g(oi->hps, peer, false, false);
         GRectangle rect = peer.getNodeRectangle(*node);
         bool nodeIsExpanded = (item->flRecordAttr & CRA_EXPANDED);
         bool nodeIsSelected = (item->flRecordAttr & CRA_SELECTED);
         bool nodeHasCaretFocus = (nodeIsSelected && peer.hasFocus());
         drawNode(g, *node, rect, nodeIsExpanded, nodeIsSelected, nodeHasCaretFocus);
         // Clear, so PM won't highlight after we have returned.
         oi->fsState = oi->fsStateOld = false;
         return MRESULT(TRUE);
      }

      case WM_CONTROL:
      {
         int ctrlID = msg.getParam1LoUShort();
         if (ctrlID != peer.getWindowID())
            return GWindow::handleWindowMessage(msg);
         int notifyID = msg.getParam1HiUShort();
         int data = msg.getParam2Int();
         switch (notifyID)
         {
            case CN_EMPHASIS:
            {
               // Node is either selected or unselected.
               // Notify selection change only when node is selected.
               // This is to prevent the notification from occuring 
               // twice each time a new node is selected.
               NOTIFYRECORDEMPHASIS* nri = (NOTIFYRECORDEMPHASIS*) data;
               OS2NodeRecord* os2Node = (OS2NodeRecord*) nri->pRecord;
               GTreeNode* node = os2Node->node;
               if ((nri->fEmphasisMask & CRA_SELECTED))
               {
                  // Node is selected.
                  if (node != currentSelectedNode)
                  {
                     currentSelectedNode = node;
                     fireTreeNodeSelectionHasChanged();
                  }
               }
               else
               {
                  // Node is unselected.
                  if (node == currentSelectedNode)
                     currentSelectedNode = null;
               }
               return 0;
            }

            /* TODO: We rely on manually expanding nodes, and calling listeners when doing that.
            case CN_EXPANDTREE:
            {
               OS2NodeRecord* item = (OS2NodeRecord*) data;
               if (item != null)
               {
                  GTreeNode* node = item->node;
                  fireTreeNodeIsExpanding(node);
               }
               return 0;
            }

            case CN_COLLAPSETREE:
            {
               OS2NodeRecord* item = (OS2NodeRecord*) data;
               if (item != null)
               {
                  GTreeNode* node = item->node;
                  fireTreeNodeIsCollapsing(node);
               }
               return 0;
            }
            */
         }
         return GWindow::handleWindowMessage(msg);
      }

      default:
         return GWindow::handleWindowMessage(msg);
   }
}

GString GTreeView::queryValue () const
{
   GTreeView* self = const_cast<GTreeView*>(this); // Throw away const.
   GMutableTreeNode* node = self->getSelectedNode();
   if (node == null)
      return GString::Empty;
   GObject* userObj = node->getUserObject();
   if (userObj == null)
      return GString::Empty;
   return userObj->toString();
}

void GTreeView::ensureNodeIsVisible ( const GTreeNode& node )
{
   // TODO: What about fireTreeNodeIsExpanding(node)?
   // Make sure the path to the node is expanded.
   GTreePath path(node.getPath());
   expandPath(path);
   // Scroll the node into view.
   OS2NodeRecord* recc = node.hSysHandle;
   peer.sendMessage(CM_INVALIDATERECORD, MPFROMP(&recc), MPFROM2SHORT(1, CMA_ERASE | CMA_REPOSITION | CMA_TEXTCHANGED));
}

bool GTreeView::toggleNodeExpanded ( GTreeNode& node )
{
   if (isNodeExpanded(node))
   {
      collapseNode(node);
      return false;
   }
   else
   {
      expandNode(node);
      return true;
   }
}

bool GTreeView::isShowRoot () const
{
   return showRoot;
}

bool GTreeView::isNodeExpanded ( const GTreeNode& node ) const
{
   OS2NodeRecord* recc = node.hSysHandle;
   if (recc->flRecordAttr & CRA_EXPANDED)
      return true;
   else
      return false;
}

void GTreeView::collapseNode ( GTreeNode& node )
{
   if (!isNodeExpanded(node))
      return;
   fireTreeNodeIsCollapsing(node);
   OS2NodeRecord* recc = node.hSysHandle;
   peer.sendMessage(CM_COLLAPSETREE, MPARAM(recc), MPARAM(0));
}

void GTreeView::expandNode ( GTreeNode& node )
{
   if (isNodeExpanded(node))
      return;
   GMutableTreeNode* mnode = dynamic_cast<GMutableTreeNode*>(&node);
   if (mnode == null) // Should never happen, but in case.
      return;
   bool allowsChildren = mnode->getAllowsChildren();
   if (!allowsChildren)
      return;
   fireTreeNodeIsExpanding(node);
   bool newAllowsChildren = mnode->getAllowsChildren();
   if (!newAllowsChildren)
   {
      // User code has removed the + button.
      // Reflect this by repainting the node.
      GRectangle rect = peer.getNodeRectangle(*mnode);
      peer.invalidateRect(rect);
      return;
   }
   OS2NodeRecord* recc = node.hSysHandle;
   peer.sendMessage(CM_EXPANDTREE, MPARAM(recc), MPARAM(0));
}

void GTreeView::expandPath ( const GTreePath& path )
{
   int pathCount = path.getPathCount();
   for (int i=showRoot?0:1; i<pathCount; i++)
   {
      GTreePath& pathSelf = const_cast<GTreePath&>(path);
      GTreeNode& node = pathSelf.getPathComponent(i);
      expandNode(node);
   }
}

void GTreeView::setTreeModel ( GTreeModel* newModel, bool autoDelete )
{
   // Make sure the root node of the tree model is != null.
   GTreeNode* root = newModel->getRoot();
   if (root == null)
      gthrow_(GIllegalArgumentException("Root node cannot be null!"));

   // ---
   if (autoDeleteTreeModel)
      delete treeModel;
   treeModel = null;
   autoDeleteTreeModel = false;
   if (newModel != null)
   {
      autoDeleteTreeModel = autoDelete;
      treeModel = newModel;
      treeModel->addTreeModelListener(this);

      // TODO: Remove all nodes from the Tree Control.

      if (showRoot)
      {
         // Create the root node in the Tree Control.
         int extraBytes = sizeof(OS2NodeRecord) - sizeof(RECORDCORE);
         OS2NodeRecord* recc = (OS2NodeRecord*) peer.sendMessage(CM_ALLOCRECORD, MPARAM(extraBytes), MPARAM(1));
         recc->node = root;
         RECORDINSERT ri = { 0 };
         ri.cb = sizeof(ri);
         ri.pRecordOrder = (RECORDCORE*) CMA_FIRST;
         ri.pRecordParent = (RECORDCORE*) NULL;
         ri.zOrder = (ULONG) CMA_TOP;
         ri.cRecordsInsert = 1;
         peer.sendMessage(CM_INSERTRECORD, GWindowMessage::Param1(recc), GWindowMessage::Param2(&ri));
         root->hSysHandle = recc;
      }
   }
}

// Called when one or more new nodes has been added to the tree model.
void GTreeView::treeNodesInserted ( const GTreeModelEvent& e )
{
   const GVector<int>& indexes = e.getChildIndexes();
   const GArray<GTreeNode>& children = e.getChildren();
   for (int i=0, count=children.getCount(); i<count; i++)
   {
      int index = indexes[i];
      GTreeNode& node = children[i];
      GTreeNode& parent = *node.getParent();
      bool isRoot = (&parent == treeModel->getRoot());

      // Create the node in the Tree Control.
      int extraBytes = sizeof(OS2NodeRecord) - sizeof(RECORDCORE);
      OS2NodeRecord* recc = (OS2NodeRecord*) peer.sendMessage(CM_ALLOCRECORD, MPARAM(extraBytes), MPARAM(1));
      recc->node = &node;
      recc->flRecordAttr = CRA_COLLAPSED;
      RECORDINSERT ri = { 0 };
      ri.cb = sizeof(ri);
      ri.pRecordParent = (isRoot && !showRoot ? NULL : parent.hSysHandle);
      ri.pRecordOrder = (index <= 0 ? ((RECORDCORE*) CMA_FIRST) : parent.getChildAt(index-1).hSysHandle);
      ri.zOrder = ULONG(CMA_TOP);
      ri.cRecordsInsert = 1;
      peer.sendMessage(CM_INSERTRECORD, MPARAM(recc), MPARAM(&ri));
      node.hSysHandle = recc;
   }
}

void GTreeView::treeNodesRemoved ( const GTreeModelEvent& e )
{
   const GArray<GTreeNode>& children = e.getChildren();
   for (int i=0, count=children.getCount(); i<count; i++)
   {
      GTreeNode& node = children[i];

      // Remove the node from the Tree Control.
      OS2NodeRecord* recc = node.hSysHandle;
      USHORT flags = CMA_FREE | CMA_INVALIDATE;
      peer.sendMessage(CM_REMOVERECORD, MPARAM(&recc), MPFROM2SHORT(1, flags));
      node.hSysHandle = 0;
   }
}

void GTreeView::treeNodesChanged ( const GTreeModelEvent& e )
{
}
