/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TOOLTIP
#define __GLIB_TOOLTIP

#include "glib/primitives/GString.h"
#include "glib/gui/GColor.h"

/**
 * This is a singleton class used to display a small sticky window 
 * with a brief help about some item on screen, typically something 
 * like a clickable toolbar button.
 * 
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
class GTooltip : private GObject
{
   friend class TooltipWindow;

   public:

      /**
       * Used to define the position of the tooltip box, with respect 
       * to its owner window.
       *
       * @author  Leif Erik Larsen
       * @since   2005.12.21
       * @see     GWindow#setTooltipPosition
       */
      enum Position 
      {
         PosInheritFromParent = 0,
         PosBelow,
         PosAbove,
         PosOverlap
      };

   private:

      class GWindow* currentOwner;

      /**  We will create the window the first time it is to be shown. */
      class TooltipWindow* win;

      /** The default font for tooltip windows. */
      static const GString DefaultFont;

      /** The singleton instance of this class. */
      static GTooltip TheInstance;

      /** The application defined default value for "Auto Hide Milliseconds". */
      static int DefAutoHideMillis;

      /** The application defined default value for "Delay Show Milliseconds". */
      static int DefDelayShowMillis;

   public:

      static const GColor DefBckColor;
      static const GColor DefFrgColor;

   private:

      /**
       * Keep this constructor private in order to make sure only 
       * one single instance of this class is created.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.22
       */
      GTooltip ();

      virtual ~GTooltip ();

      void createTheWindow ( class GWindow& ownerWin );

   public:

      /**
       * Get a pointer to the window for which the tooltip window 
       * is currently captured, or else null if the tooltip window
       * is not currently captured to any window.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.13
       */
      static class GWindow* GetCurrentOwner ();

      /**
       * Show tooltip for the specified window.
       * The tooltip will actually be set visible only if the owner
       * window of the specified window is configured to "show tooltip".
       *
       * @author  Leif Erik Larsen
       * @since   2005.12.13
       * @param   forWin  The window for which to display the tooltip.
       * @param   text    The text of which to show in the tooltip box.
       * @param   delayShowMillis The number of milliseconds to wait until 
       *                  actually displaying the tooltip box. The tooltip
       *                  box will display immediately if zero is given.
       *                  A value of less than zero means that we should use 
       *                  the default value provided by a recent call to 
       *                  {@link #SetDefaultDelayShowMillis}.
       * @param   autoHideMillis The number olf milliseconds until the 
       *                  tooltip box gets automatically hidden, after it
       *                  has actuelly been visible. Or zero in order to 
       *                  never automatically hide it.
       *                  A value of less than zero means that we should use 
       *                  the default value provided by a recent call to 
       *                  {@link #SetDefaultAutoHideMillis}.
       */
      static void Show ( class GWindow& forWin, 
                         const GString& text, 
                         int delayShowMillis = -1,
                         int autoHideMillis = -1 );

      /**
       * If the tooltip window is current visible then close 
       * it and redisplay it.
       */
      static void ReDisplay ();

      /**
       * Hide the tooltip window if it is currently visible.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.22
       */
      static void Hide ();

      /** 
       * Set the value to use for the "autoHideMillis" parameter of
       * {@link #Show} in case a negative value is specified for that 
       * parameter.
       *
       * @author  Leif Erik Larsen
       * @since   2006.11.08
       */
      static void SetDefaultAutoHideMillis ( int value );

      /** 
       * Set the value to use for the "delayShowMillis" parameter of
       * {@link #Show} in case a negative value is specified for that 
       * parameter.
       *
       * @author  Leif Erik Larsen
       * @since   2006.11.08
       */
      static void SetDefaultDelayShowMillis ( int value );
};

#endif
